package de.fzi.sensidl.language.generator.factory.java

import de.fzi.sensidl.design.sensidl.Endianness
import de.fzi.sensidl.design.sensidl.dataRepresentation.Data
import de.fzi.sensidl.design.sensidl.dataRepresentation.DataConversion
import de.fzi.sensidl.design.sensidl.dataRepresentation.DataRange
import de.fzi.sensidl.design.sensidl.dataRepresentation.DataSet
import de.fzi.sensidl.design.sensidl.dataRepresentation.DataType
import de.fzi.sensidl.design.sensidl.dataRepresentation.LinearDataConversion
import de.fzi.sensidl.design.sensidl.dataRepresentation.LinearDataConversionWithInterval
import de.fzi.sensidl.design.sensidl.dataRepresentation.MeasurementData
import de.fzi.sensidl.design.sensidl.dataRepresentation.NonMeasurementData
import de.fzi.sensidl.language.generator.GenerationUtil
import de.fzi.sensidl.language.generator.SensIDLConstants
import de.fzi.sensidl.language.generator.SensIDLOutputConfigurationProvider
import de.fzi.sensidl.language.generator.factory.IDTOGenerator
import java.util.ArrayList
import java.util.HashMap
import java.util.List
import org.apache.log4j.Logger

/**
 * Java code generator for the SensIDL Model. 
 * Code will be generated by running the {@code generate()}-Method
 * 
 * @author Sven Eckhardt
 * @author Pawel Bielski 
 * @author Max Peters
 * @author Fabian Scheytt 
 */
 
class JavaDTOGenerator implements IDTOGenerator {
	private static Logger logger = Logger.getLogger(JavaDTOGenerator)
	
	private boolean createEmptyConstructor = true

	private List<DataSet> dataSet
	
	private boolean createProject = false
	
	private boolean bigEndian
	
	/**
	 * The constructor calls the constructor of the superclass to set a
	 * list of DataSet-elements.
	 * @param newDataSet - represents the list of DataSet-elements.
	 */
	new(List<DataSet> newDataSet) {
		this.dataSet = newDataSet
	}
	
	/**
	 * The constructor calls the constructor of the superclass to set a
	 * list of DataSet-elements and a member-variable.
	 * @param newDataSet - represents the list of DataSet-elements.
	 * @param createProject - indicates if a project should be created.
	 */
	new(List<DataSet> newDataSet,boolean createProject) {
		this.dataSet = newDataSet
		this.createProject = createProject
	}

	/**
	 * Generates the .java file for each data transfer object.
	 * @see IDTOGenerator#generate()
	 */
	override generate() {
		logger.info("Start with code-generation of a java data transfer object.")
		val filesToGenerate = new HashMap		
		
		if (GenerationUtil.getSensorInterface(this.dataSet.get(0).eContents.filter(Data).get(0).eContainer).encodingSettings.endianness == Endianness.BIG_ENDIAN){
			bigEndian = true;
		} else {
			bigEndian = false;
		}
		
		if (createProject) {
			for (d : this.dataSet) {
				filesToGenerate.put("src/de/fzi/sensidl/" + GenerationUtil.getSensorInterfaceName(this.dataSet.get(0).eContents.filter(Data).get(0).eContainer) +"/" + addFileExtensionTo(GenerationUtil.toNameUpper(d)), generateClassBody(GenerationUtil.toNameUpper(d), d))
				logger.info("File: " + addFileExtensionTo(GenerationUtil.toNameUpper(d)) + " was generated in " + SensIDLOutputConfigurationProvider.SENSIDL_GEN)
			}
		
		} else{
			for (d : this.dataSet) {
				filesToGenerate.put(addFileExtensionTo(GenerationUtil.toNameUpper(d)), generateClassBody(GenerationUtil.toNameUpper(d), d))
				logger.info("File: " + addFileExtensionTo(GenerationUtil.toNameUpper(d)) + " was generated in " + SensIDLOutputConfigurationProvider.SENSIDL_GEN)
			}
		}
		filesToGenerate
	}

	/**
	 * Generates the Classes
	 */
	def generateClassBody(String className, DataSet d) {
		'''
			«IF createProject»
			package de.fzi.sensidl.«GenerationUtil.getSensorInterfaceName(d.eContents.filter(Data).get(0).eContainer)»;
			 
			«ENDIF» 
			«IF !bigEndian»
				import java.nio.ByteBuffer;
				import java.nio.ByteOrder;
			«ENDIF» 
			 
			/**
			 * Data transfer object for «className»
			 *
			 * @generated
			 */
			public class «className» {
				
				private static final long serialVersionUID = 1L;
				«generateDataFieldsIncludeParentDataSet(d)»
				
				«generateConstructorIncludeParentDataSet(d, className)»
				
				«IF createEmptyConstructor»
				/**
				 * empty constructor for the Data transfer object
				 */
				public «className»() {
				
				}
				«ENDIF»
				«generateDataMethodsIncludeParentDataSet(d)»
				
				«IF !bigEndian»
				«d.generateConverterMethods»
				«ENDIF»
			}
		 '''
	}

// ------------------------------ Data Fields ------------------------------
	/**
	 * Generates the data fields for this data set including used data sets.
	 */
	def generateDataFieldsIncludeParentDataSet(DataSet d) {
		'''
		«FOR data : d.eContents.filter(NonMeasurementData)»
			
			«generateDataFields(data)»
		«ENDFOR» 
		«FOR data : d.eContents.filter(MeasurementData)»
			«generateDataFields(data)»
		«ENDFOR» 
		«IF d.parentDataSet != null»
		
		/*	
		 * «d.parentDataSet.description»
		 */
		private «GenerationUtil.toNameUpper(d.parentDataSet)» «GenerationUtil.toNameLower(d.parentDataSet)»;
		«ENDIF»
		'''
	}
	
	
	/**
	 * Generates the fields for the measurement data
	 */
	def generateDataFields(MeasurementData d) {
		'''
			/*
			 «IF d.description != null»
			 * «d.description»
			 * 
			 «ENDIF» 
			 * Unit: «d.unit»
			 */
			private «d.toTypeName» «GenerationUtil.toNameLower(d)»;
			
			«IF d.hasLinearDataConversionWithInterval»
			/*
			«IF d.description != null» * «d.description»
			 * 
			«ENDIF» 
			 * Unit: «d.unit»
			 * Adjusted
			 */
			private «d.getDataConversionType.toTypeName» «GenerationUtil.toNameLower(d)»Adjusted;
			«ENDIF»
		'''
	}

	/**
	 * Generates the fields for the non measurement data
	 */
	def generateDataFields(NonMeasurementData d) {
		if (d.constant) {
			'''
				«IF d.description != null»
				/*
				 *«d.description»
				 «IF d.dataType.isUnsigned» 
				 *
				 * Java has no option for unsigned data types, so if the data has an 
				 * unsigned data type the value is calculated by subtracting the maximum 
				 * value from the signed data type and adding it again, if it is used.
				 *
				«ENDIF»
				 */
				«ENDIF» 
				private «d.toTypeName» «d.name.toUpperCase» = «IF d.dataType.isUnsigned»(«d.toSimpleTypeName») («d.value» - «d.toTypeName».MAX_VALUE)«ELSE»«IF d.dataType == DataType.STRING»"«d.value»"«ELSE»«d.value»«ENDIF»«ENDIF»;
			'''
		} else {
			'''
				«IF d.description != null»
				 /*
				  *«d.description»
				  */
				«ENDIF» 
				private «d.toTypeName» «GenerationUtil.toNameLower(d)»«IF !d.value.nullOrEmpty» = «IF d.dataType == DataType.STRING»"«d.value»"«ELSE»«d.value»«ENDIF»«ENDIF»;
			'''
		}

	}
	
	
// ------------------------------ Constructor ------------------------------	
	/**
	 * Generates the constructor for this data set including used data sets.
	 */
	def generateConstructorIncludeParentDataSet(DataSet d,String className) {
		'''
		/**
		 * Constructor for the Data transfer object
		 */
		public «className»(«d.generateConstructorArgumentsIncludeParentDataSets») {
			«FOR data : d.eContents.filter(MeasurementData)»
				«IF data.hasLinearDataConversionWithInterval»
					set«GenerationUtil.toNameUpper(data)»(«GenerationUtil.toNameLower(data)»);
				«ELSE»
					this.«GenerationUtil.toNameLower(data)» = «IF data.dataType.isUnsigned»(«data.toSimpleTypeName») («GenerationUtil.toNameLower(data)» - «data.toTypeName».MAX_VALUE)«ELSE»«GenerationUtil.toNameLower(data)»«ENDIF»;
				«ENDIF»
			«ENDFOR»
			«FOR data : d.eContents.filter(NonMeasurementData)»
				«IF !data.constant»
					this.«GenerationUtil.toNameLower(data)» = «IF data.dataType.isUnsigned»(«data.toSimpleTypeName») («GenerationUtil.toNameLower(data)» - «data.toTypeName».MAX_VALUE)«ELSE»«GenerationUtil.toNameLower(data)»«ENDIF»;
				«ENDIF»
			«ENDFOR»
			«IF d.parentDataSet != null»
				this.«GenerationUtil.toNameLower(d.parentDataSet)» = «GenerationUtil.toNameLower(d.parentDataSet)»;
			«ENDIF»
		}
		'''
	}
	
	/**
	 * Generates the Constructor arguments
	 */
	def generateConstructorArgumentsIncludeParentDataSets(DataSet d) {
		// create an ArrayList with all data that is not a constant NonMeasurementData (which will not be constructor arguments)
		var dataList = new ArrayList<Data>();
		var dataSet = d
		
		for (data : dataSet.eContents.filter(Data)) {
			if (data instanceof NonMeasurementData) {
				var nmdata = data as NonMeasurementData
				if (!nmdata.constant) {
					dataList.add(data)
				}
			} else {
				dataList.add(data)
			}
		}

		if (dataList.size > 0) {
			var firstElement = dataList.get(0).toTypeName + " " + GenerationUtil.toNameLower(dataList.get(0))
			dataList.remove(0)
			if(d.parentDataSet != null){
				'''«firstElement»«FOR data : dataList», «data.toTypeName» «GenerationUtil.toNameLower(data)»«ENDFOR», «GenerationUtil.toNameUpper(d.parentDataSet)» «GenerationUtil.toNameLower(d.parentDataSet)»'''
			} else {
				'''«firstElement»«FOR data : dataList», «data.toTypeName» «GenerationUtil.toNameLower(data)»«ENDFOR»'''
			}
		} else {
			createEmptyConstructor = false;
			''''''
		}
	}

// ------------------------------ Getter and Setter ------------------------------
	
	/**
	 * Generates the getter and setter methods for the data of this data set including used data sets.
	 */
	def generateDataMethodsIncludeParentDataSet(DataSet d) {
		'''
		«FOR data : d.eContents.filter(MeasurementData)»
			
			«generateGetter(data)»
			
			«generateSetter(data)»
		«ENDFOR»
		«FOR data : d.eContents.filter(NonMeasurementData)»
			
			«generateGetter(data)»
			
			«generateSetter(data)»
		«ENDFOR»
		«IF d.parentDataSet != null»
			
			«generateGetter(d.parentDataSet)»
			
			«generateSetter(d.parentDataSet)»
		«ENDIF»
		'''
	}
	
	/**
	 * returns the Getter Name
	 */
	def toGetterName(Data d) {
		'''get«d.name.replaceAll("[^a-zA-Z0-9]", "").toFirstUpper»'''
	}

	
	/**
	 * returns the Setter Name
	 */
	def toSetterName(Data d) {
		'''set«d.name.replaceAll("[^a-zA-Z0-9]", "").toFirstUpper»'''
	}
	
	
// ------------------------------ Getter ------------------------------

	/** 
	 * Generates the Getter Method for the dataset
	 */
	
	def generateGetter(DataSet d){
		'''
		/**
		 * @return the «GenerationUtil.toNameLower(d)»
		 *
		 */
		public «GenerationUtil.toNameUpper(d)» get«GenerationUtil.toNameUpper(d)»() {
			return this.«GenerationUtil.toNameLower(d)»;
		}
		'''
	}

	/** 
	 * Generates the Getter Method for the measurement data
	 */
	def generateGetter(MeasurementData d) {
		'''
			«IF !d.hasLinearDataConversionWithInterval»
				/**
				 «IF d.dataType.isUnsigned» 
				 * Java has no option for unsigned data types, so if the data has an 
				 * unsigned data type the value is calculated by subtracting the maximum 
				 * value from the signed data type and adding it again, if it is used.
				 *
				 «ENDIF»
				 * @return the «GenerationUtil.toNameLower(d)»
				 */
				public «d.toTypeName» «d.toGetterName»() {
					return «IF d.dataType.isUnsigned»(«d.toSimpleTypeName») (this.«GenerationUtil.toNameLower(d)» + «d.toTypeName».MAX_VALUE)«ELSE»this.«GenerationUtil.toNameLower(d)»«ENDIF»;
				}
			«ELSE»
				/**
				 * @return the «GenerationUtil.toNameLower(d)»
				 */
				public «d.dataConversionType.toTypeName» «d.toGetterName»(){
					return this.«GenerationUtil.toNameLower(d)»Adjusted;
				}
				
				/**
				 * @return the «GenerationUtil.toNameLower(d)»
				 */
				public «d.toTypeName» «d.toGetterName»NotAdjusted(){
					return «IF d.dataType.isUnsigned»(«d.toSimpleTypeName») (this.«GenerationUtil.toNameLower(d)» + «d.toTypeName».MAX_VALUE)«ELSE»this.«GenerationUtil.toNameLower(d)»«ENDIF»;
				}
			«ENDIF»
		'''
	}

	/** 
	 * Generates the Getter Method for the non measurement data
	 */
	def generateGetter(NonMeasurementData d) {
		'''
			/**
			 «IF d.dataType.isUnsigned» 
			 * Java has no option for unsigned data types, so if the data has an 
			 * unsigned data type the value is calculated by subtracting the maximum 
			 * value from the signed data type and adding it again, if it is used.
			 *
			 «ENDIF»
			 * @return the «GenerationUtil.toNameLower(d)»
			 */
			public «d.toTypeName» «d.toGetterName»() {
				return «IF d.dataType.isUnsigned»(«d.toSimpleTypeName») («IF d.constant»«d.name.toUpperCase»«ELSE»this.«GenerationUtil.toNameLower(d)»«ENDIF» + «d.toTypeName».MAX_VALUE)«ELSE»this.«IF d.constant»«d.name.toUpperCase»«ELSE»«GenerationUtil.toNameLower(d)»«ENDIF»«ENDIF»;
			}
		'''
	}


// ------------------------------ Setter ------------------------------

	/**
	 * Generates the Setter Method for the dataset
	 */
	def generateSetter(DataSet d){
		'''
		/**
		 * @param info
		 *            the «GenerationUtil.toNameLower(d)» to set
		 */
		public void set«GenerationUtil.toNameUpper(d)»(«GenerationUtil.toNameUpper(d)» «GenerationUtil.toNameLower(d)») {
			this.«GenerationUtil.toNameLower(d)» = «GenerationUtil.toNameLower(d)»;
		}
		'''
	}
	/** 
	 * Generates the Setter Method for the measurement data
	 */
	def generateSetter(MeasurementData d) {
		'''
			«IF d.adjustments.empty == false»
				«FOR dataAdj : d.adjustments»
					«IF dataAdj instanceof DataRange»
						/**
						 * @param «GenerationUtil.toNameLower(d)»
						 *            the «GenerationUtil.toNameLower(d)» to set
						 */
						public void «d.toSetterName»WithRange(«d.toTypeName» «GenerationUtil.toNameLower(d)»){
							if («GenerationUtil.toNameLower(d)» >= «dataAdj.range.lowerBound» && «GenerationUtil.toNameLower(d)» <= «dataAdj.range.upperBound»)
								this.«GenerationUtil.toNameLower(d)» = «GenerationUtil.toNameLower(d)»;
							else
								throw new IllegalArgumentException("value is out of defined range");	
						} 
					«ELSEIF dataAdj instanceof DataConversion»	
						/**
						 * @param «GenerationUtil.toNameLower(d)»
						 *            the «GenerationUtil.toNameLower(d)» to set
						 */
						public void «d.toSetterName»WithDataConversion(«d.toTypeName» «GenerationUtil.toNameLower(d)») {
							try {
								«d.generateSetterBodyForMeasurementData(dataAdj as DataConversion)»
							} catch (IllegalArgumentException e) {
								//Do something
							}
						}
«««						«IF dataAdj instanceof DataConversion»						
«««							«IF dataAdj instanceof LinearDataConversion»
«««								/**
«««								 * @param «GenerationUtil.toNameLower(d)»  
«««								 *			the «GenerationUtil.toNameLower(d)» to set
«««								 */
«««								public void «d.toSetterName»(«d.toTypeName» «GenerationUtil.toNameLower(d)»){
«««						
«««									this.«GenerationUtil.toNameLower(d)» =  «GenerationUtil.toNameLower(d)» * («d.toSimpleTypeName») «dataAdj.scalingFactor» + («d.toSimpleTypeName») «dataAdj.offset»;
«««								} 
«««							«ELSE»
«««							«IF dataAdj instanceof LinearDataConversionWithInterval»
«««								/**
«««								 * @param «GenerationUtil.toNameLower(d)»  
«««								 *			the «GenerationUtil.toNameLower(d)» to set
«««								 */
«««								public void «d.toSetterName»(«d.toTypeName» «GenerationUtil.toNameLower(d)»){
«««									if («GenerationUtil.toNameLower(d)» >= «dataAdj.fromInterval.lowerBound» && «GenerationUtil.toNameLower(d)» <= «dataAdj.fromInterval.upperBound»){												
«««								
«««										«d.toTypeName» oldMin = («d.toSimpleTypeName») «dataAdj.fromInterval.lowerBound»;
«««										«d.toTypeName» oldMax = («d.toSimpleTypeName») «dataAdj.fromInterval.upperBound»;
«««										«d.toTypeName» newMin = («d.toSimpleTypeName») «dataAdj.toInterval.lowerBound»;
«««										«d.toTypeName» newMax = («d.toSimpleTypeName») «dataAdj.toInterval.upperBound»;
«««								
«««										this.«GenerationUtil.toNameLower(d)» = («d.toSimpleTypeName») ((((«GenerationUtil.toNameLower(d)» - oldMin) * (newMax - newMin)) / (oldMax - oldMin)) + newMin);
«««									}
«««									else
«««										throw new IllegalArgumentException("value is out of defined source Interval");
«««								} 		
«««							«ENDIF»
«««						«ENDIF»
					«ENDIF»				
				«ENDFOR»
			«ENDIF»
			/**
			 * @param «GenerationUtil.toNameLower(d)»  
			 *            the «GenerationUtil.toNameLower(d)» to set
			 */
			public void «d.toSetterName»(«d.toTypeName» «GenerationUtil.toNameLower(d)») {
				
				this.«GenerationUtil.toNameLower(d)» = «IF d.dataType.isUnsigned»(«d.toSimpleTypeName») («GenerationUtil.toNameLower(d)» - «d.toTypeName».MAX_VALUE)«ELSE»«GenerationUtil.toNameLower(d)»«ENDIF»;
			} 
			
		''' 

	}
	
	dispatch def generateSetterBodyForMeasurementData(MeasurementData data, LinearDataConversion conversion) {
		'''
			final double offset = «conversion.offset»;
			final double scalingFactor = «conversion.scalingFactor»;
			
			this.«GenerationUtil.toNameLower(data)» = («data.toSimpleTypeName») «GenerationUtil.getSensorInterfaceName(data.eContainer)»«SensIDLConstants.UTILITY_CLASS_NAME».«SensIDLConstants.LINEAR_CONVERSION_METHOD_NAME»(«GenerationUtil.toNameLower(data)», scalingFactor, offset);
		'''
	}
	
	dispatch def generateSetterBodyForMeasurementData(MeasurementData data, LinearDataConversionWithInterval conversion) {
		'''
			«conversion.dataType.toTypeName» oldMin = («conversion.dataType.toSimpleTypeName») «conversion.fromInterval.lowerBound»;
			«conversion.dataType.toTypeName» oldMax = («conversion.dataType.toSimpleTypeName») «conversion.fromInterval.upperBound»;
			«conversion.dataType.toTypeName» newMin = («conversion.dataType.toSimpleTypeName») «conversion.toInterval.lowerBound»;
			«conversion.dataType.toTypeName» newMax = («conversion.dataType.toSimpleTypeName») «conversion.toInterval.upperBound»;
			
			this.«GenerationUtil.toNameLower(data)» = «IF data.dataType.isUnsigned»(«data.toSimpleTypeName») («GenerationUtil.toNameLower(data)» - «data.toTypeName».MAX_VALUE)«ELSE»«GenerationUtil.toNameLower(data)»«ENDIF»;
			this.«GenerationUtil.toNameLower(data)»Adjusted = («conversion.dataType.toSimpleTypeName») «GenerationUtil.getSensorInterfaceName(data.eContainer)»«SensIDLConstants.UTILITY_CLASS_NAME».«SensIDLConstants.LINEAR_CONVERSION_WITH_INTERVAL_METHOD_NAME»(«GenerationUtil.toNameLower(data)», oldMin, oldMax, newMin, newMax);
		'''
	}

	/** 
	 * Generates the Setter Method for the non measurement data
	 */
	def generateSetter(NonMeasurementData d) {
		if (d.constant) {
			'''
				// no setter for constant value
			'''
		} else {
			'''
				/**
				 «IF d.dataType.isUnsigned» 
				 * Java has no option for unsigned data types, so if the data has an 
				 * unsigned data type the value is calculated by subtracting the maximum 
				 * value from the signed data type and adding it again, if it is used.
				 *
				 «ENDIF»
				 * @param «GenerationUtil.toNameLower(d)»
				 *            the «GenerationUtil.toNameLower(d)» to set
				 */
				public void «d.toSetterName»(«d.toTypeName» «GenerationUtil.toNameLower(d)»){
					this.«GenerationUtil.toNameLower(d)» = «IF d.dataType.isUnsigned»(«d.toSimpleTypeName») («GenerationUtil.toNameLower(d)» - «d.toTypeName».MAX_VALUE)«ELSE»«GenerationUtil.toNameLower(d)»«ENDIF»;
				} 
			'''
		}
	}

// ------------------------------ Other Methods ------------------------------
	/**
	 * Maps to the corresponding language data type.
	 * @see IDTOGenerator#toTypeName(Data)
	 */
	override toTypeName(Data d) {
		return switch (d.dataType) {
			case INT8: Byte.name
			case UINT8: Byte.name
			case INT16: Short.name
			case UINT16: Short.name
			case INT32: Integer.name
			case UINT32: Integer.name
			case INT64: Long.name
			case UINT64: Long.name
			case FLOAT: Float.name
			case DOUBLE: Double.name
			case BOOLEAN: Boolean.name
			case STRING: String.name
		}
	}
	
	/**
	 * returns the appropriate type name for DataType
	 */
	def toTypeName(DataType d) {
		return switch (d) {
			case INT8: Byte.name
			case UINT8: Byte.name
			case INT16: Short.name
			case UINT16: Short.name
			case INT32: Integer.name
			case UINT32: Integer.name
			case INT64: Long.name
			case UINT64: Long.name
			case FLOAT: Float.name
			case DOUBLE: Double.name
			case BOOLEAN: Boolean.name
			case STRING: String.name
		}
	}
	
	/**
	 * returns true if the DataType is an unsigned DataType
	 */
	def isUnsigned(DataType d){
		if (d == DataType.UINT8 || d == DataType.UINT16 || d == DataType.UINT32 ||d == DataType.UINT64 ){
			return true
		}
		return false
	}

	/**
	 * returns the appropriate simple type name suitable for casting
	 */
	def toSimpleTypeName(Data d){
		return switch (d.dataType) {
			case INT8: "byte"
			case UINT8: "byte"
			case INT16: "short"
			case UINT16: "short"
			case INT32: "int"
			case UINT32: "int"
			case INT64: "long"
			case UINT64: "long"
			case FLOAT: "float"
			case DOUBLE: "double"
			case BOOLEAN: "boolean"
			case STRING: "String"
		}
	}
	
	/**
	 * returns the appropriate simple type name suitable for casting for dataType
	 */
	def toSimpleTypeName(DataType d){
		return switch (d) {
			case INT8: "byte"
			case UINT8: "byte"
			case INT16: "short"
			case UINT16: "short"
			case INT32: "int"
			case UINT32: "int"
			case INT64: "long"
			case UINT64: "long"
			case FLOAT: "float"
			case DOUBLE: "double"
			case BOOLEAN: "boolean"
			case STRING: "String"
		}
	}
	
	/** 
	 * returns the Data Conversion type (at the moment 
	 * only LinearDataConversionWithIntervall has a type)
	 * 
	 */
	def getDataConversionType(MeasurementData d){
		if (d.adjustments.empty){
			return null
		}
		var conversion = d.adjustments.filter(LinearDataConversionWithInterval);
		if (conversion.empty || conversion == null){
			return null
		}
		return conversion.head.dataType
	}
	
	/**
	 * @return true if the MeasurementData is adjusted 
	 * with linear data conversion with interval
	 */
	def hasLinearDataConversionWithInterval(MeasurementData data){
		return !data.adjustments.filter(LinearDataConversionWithInterval).empty
	}
	

	override addFileExtensionTo(String ClassName) {
		return ClassName + SensIDLConstants.JAVA_EXTENSION
	}
	
// ------------------------------ Little Endian Converter Methods ------------------------------

def generateConverterMethods(DataSet d) {
	'''
		«IF d.parentDataSet != null»
		/**
		 * Converts a big endian «GenerationUtil.toNameUpper(d.parentDataSet)» Object into a little endian «GenerationUtil.toNameUpper(d.parentDataSet)» Object
		 *	
		 * @param the «GenerationUtil.toNameUpper(d.parentDataSet)» Object to convert
		 * @return «GenerationUtil.toNameUpper(d.parentDataSet)» the converted «GenerationUtil.toNameUpper(d.parentDataSet)» Object
		 *
		 */
		public «GenerationUtil.toNameUpper(d.parentDataSet)» convertToLittleEndian(«GenerationUtil.toNameUpper(d.parentDataSet)» «GenerationUtil.toNameLower(d.parentDataSet)»){
			//TODO: implement Method
			return null;
		}
		«ENDIF»
		
		«convertAllToTLitteEndian(d)»
		
	'''
	}
	
def convertAllToTLitteEndian(DataSet d){
		'''
		public void «SensIDLConstants.JAVA_CONVERT_ALL_TO_LITTLE_ENDIAN_METHOD_NAME»(){
			«FOR data : d.eContents.filter(MeasurementData)»
				«GenerationUtil.toNameLower(data)» = «GenerationUtil.getSensorInterfaceName(data.eContainer)»«SensIDLConstants.UTILITY_CLASS_NAME».convertToLittleEndian(«GenerationUtil.toNameLower(data)»);
			«ENDFOR»
			«FOR data : d.eContents.filter(NonMeasurementData)»
				«IF data.isConstant»
					«data.name.toUpperCase» = «GenerationUtil.getSensorInterfaceName(data.eContainer)»«SensIDLConstants.UTILITY_CLASS_NAME».convertToLittleEndian(«data.name.toUpperCase»);
				«ELSE»
					«GenerationUtil.toNameLower(data)» = «GenerationUtil.getSensorInterfaceName(data.eContainer)»«SensIDLConstants.UTILITY_CLASS_NAME».convertToLittleEndian(«GenerationUtil.toNameLower(data)»);
				«ENDIF»
			«ENDFOR»
			«IF d.parentDataSet != null»
				«GenerationUtil.toNameLower(d.parentDataSet)» = convertToLittleEndian(«GenerationUtil.toNameLower(d.parentDataSet)»);
			«ENDIF»
		}
		
		'''
	}
}