/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved.visitor;

import java.util.ArrayList;
import java.util.List;

import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.resolved.*;
import org.ow2.dsrg.fm.tbplib.resolved.events.*;

/**
 * Copies tree of code nodes.
 * 
 * @author caitt3am
 *
 */
public class CopyVisitor 
	extends ImperativeVisitor<TBPResolvedImperativeNode> {

	@Override
	public TBPResolvedImperativeNode visitResolvedEmit(
			TBPResolvedEmit resolvedEmit) {
		
		TBPResolvedImperativeNode result =  new TBPResolvedEmit(resolvedEmit.getBinding());
		result.setAnnotation(resolvedEmit.getAnnotation());
		return result;
	}
	
    @Override
	public TBPResolvedImperativeNode visitResolvedUndefinedEmit(
			TBPResolvedUndefinedEmit resolvedEmit) {
		TBPResolvedUndefinedEmit result = 
			new TBPResolvedUndefinedEmit(resolvedEmit.getMethodCall().clone());
		result.setAnnotation(resolvedEmit.getAnnotation());
		return result;
	}
	

	@Override
	public TBPResolvedImperativeNode visitResolvedAssignment(
			TBPResolvedAssignment resolvedAssignment) {
		
		TBPResolvedNode child = (TBPResolvedNode) resolvedAssignment.getChild();
		TBPResolvedValue value_copy = (TBPResolvedValue) child.visit(this);
		TBPResolvedImperativeNode result =  new TBPResolvedAssignment( 
				resolvedAssignment.getIdf(), value_copy);
		result.setAnnotation(resolvedAssignment.getAnnotation());
		return result;
	}


	@Override
	public TBPResolvedImperativeNode visitResolvedIf(TBPResolvedIf resolvedIf) {
		TBPResolvedCondition cond = resolvedIf.getCondition().clone();
		TBPResolvedImperativeNode ifb = (TBPResolvedImperativeNode) resolvedIf.getChild(0);
		ifb = ifb.visit(this);
		TBPResolvedImperativeNode elseb = null;
		if(resolvedIf.hasElse()){
			elseb = (TBPResolvedImperativeNode) resolvedIf.getChild(1);
			elseb = elseb.visit(this);
		}
		
		TBPResolvedImperativeNode result =  new TBPResolvedIf(cond, ifb, elseb);
		result.setAnnotation(resolvedIf.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedImperativeNull(
			TBPResolvedImperativeNull resolvedImperativeNull) {
		TBPResolvedImperativeNode result =  new TBPResolvedImperativeNull();
		result.setAnnotation(resolvedImperativeNull.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedImperativeSequence(
			TBPResolvedImperativeSequence resolvedImperativeSequence) {
		TBPResolvedImperativeNode left = (TBPResolvedImperativeNode) resolvedImperativeSequence.getLeft();
		TBPResolvedImperativeNode right= (TBPResolvedImperativeNode) resolvedImperativeSequence.getRight();
		return new TBPResolvedImperativeSequence(left.visit(this), right.visit(this));
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedReturn(
			TBPResolvedReturn resolvedReturn) {
		TBPResolvedImperativeNode result =  new TBPResolvedReturn(resolvedReturn.getReturnValue());
		result.setAnnotation(resolvedReturn.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedSwitch(
			TBPResolvedSwitch resolvedSwitch) {
		
		List<TBPResolvedImperativeNode> branches = new ArrayList<TBPResolvedImperativeNode>();
		
		for (TBPNode b : resolvedSwitch.getChildren()) {
			branches.add(((TBPResolvedImperativeNode)b).visit(this));
		}
		
		TBPResolvedImperativeNode result;
		if(resolvedSwitch.isNondeterministic()){
			result = new TBPResolvedSwitch(branches);
		} else {
		    result = new TBPResolvedSwitch(resolvedSwitch.getCases(),
		    		(TBPResolvedValue) resolvedSwitch.getValue().visit(this),
		    		branches);
		}
		result.setAnnotation(resolvedSwitch.getAnnotation());
		return result;
	}
	
	@Override
	public TBPResolvedImperativeNode visitResolvedSync(
			TBPResolvedSync resolvedSync) {
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) resolvedSync.getChild();
		TBPResolvedImperativeNode result =  new TBPResolvedSync(resolvedSync.getRef(), code.visit(this));
		result.setAnnotation(resolvedSync.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedValue(
			TBPResolvedValue resolvedValue) {
		TBPResolvedImperativeNode result;
		if(resolvedValue.isReference()){
			result =  new TBPResolvedValue(resolvedValue.getReference());	 
		} else {
			TBPResolvedImperativeNode child = (TBPResolvedImperativeNode) resolvedValue.getChild(0);
			result =  new TBPResolvedValue((TBPResolvedEmit)child.visit(this));
		}
		result.setAnnotation(resolvedValue.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedWhile(
			TBPResolvedWhile resolvedWhile) {
		TBPResolvedCondition cond = resolvedWhile.getCondition().clone();
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) resolvedWhile.getChild();
		TBPResolvedImperativeNode result =  new TBPResolvedWhile(cond, code.visit(this));
		result.setAnnotation(result.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedImperativeBinaryNode(
			TBPResolvedImperativeBinaryNode node) {
		return null;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedImperativeLeafNode(
			TBPResolvedImperativeLeafNode node) {
		return null;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedImperativeNaryNode(
			TBPResolvedImperativeNaryNode node) {
		return null;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedImperativeUnaryNode(
			TBPResolvedImperativeUnaryNode node) {
		return null;
	}

	@Override
	public TBPResolvedImperativeNode visitResolvedMangledReaction(
			TBPResolvedMangledReaction node) {
		
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) node.getChild();
		TBPResolvedImperativeNode result =  new TBPResolvedMangledReaction(node.getBinding(),code.visit(this));
		result.setAnnotation(node.getAnnotation());
		return result;
	}
	
	@Override
	public TBPResolvedImperativeNode visitResolvedThreadContainerNode(
			TBPResolvedThreadContainerNode node) {
		
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) node.getChild();
		TBPResolvedImperativeNode result =  new TBPResolvedThreadContainerNode(node.getName(), code.visit(this));
		result.setAnnotation(node.getAnnotation());
		return result;
	}
}
