/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved.util;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Describes signature of a method.
 * 
 * @author caitt3am
 *
 */

public class MethodSignature {

	private final String iface;
	private final String method_name;
	private final String fullname;

	private final List<String> param_names;
	private final Map<String, Typedef> param_types;
	private final Typedef return_type; 
	
	
	public MethodSignature(String iface, String method_name, List<String> param_names, 
			Map<String, Typedef> param_types, Typedef return_type) {
		this.iface = iface;
		this.method_name = method_name;
		this.return_type = return_type;
		this.fullname = iface + "." + method_name;
		this.param_names = param_names;
		this.param_types = param_types; 
		
		if(param_names.size() != param_types.size()){
			throw new TBPResolvingException("Number of parameters and their types are not the same.");
		}
	}


	public String getInterface() {
		return iface;
	}
	
	public String getMethod() {
		return method_name;
	}
	
	public String getFullname() {
		return fullname;
	}
	
	public List<String> getParamNames() {
		return param_names;
	}
	
	public Map<String, Typedef> getParamTypes() {
		return param_types;
	}
	
	/**
	 * Return type of method; <code>null</code> if type is void.
	 */
	public Typedef getReturnType() {
		return return_type;
	}
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append(getFullname()).append('(');
		boolean first = true;
		for(Entry<String, Typedef> entry : getParamTypes().entrySet()){
			if(!first)
				sb.append(", ");
			sb.append(entry.getValue().getName());
			sb.append(" ");
			sb.append(entry.getKey());
			first = false;
		}
		sb.append(')');
		if(return_type != null){
			sb.append(" : ");
			sb.append(return_type.getName());
		}
		return sb.toString();
	}


	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((fullname == null) ? 0 : fullname.hashCode());
		result = prime * result
				+ ((param_types == null) ? 0 : param_types.hashCode());
		return result;
	}


	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof MethodSignature))
			return false;
		final MethodSignature other = (MethodSignature) obj;
		if (fullname == null) {
			if (other.fullname != null)
				return false;
		} else if (!fullname.equals(other.fullname))
			return false;
		if (param_types == null) {
			if (other.param_types != null)
				return false;
		} else if (!param_types.equals(other.param_types))
			return false;
		return true;
	}
	
}

