/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved.util;

import java.util.*;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.FPRef;
import org.ow2.dsrg.fm.tbplib.resolved.Reference;

/**
 * Binding between formal parameters and their actual values.
 * 
 * @author caitt3am
 *
 */
public class Binding {

	private final MethodSignature ms;
	private final Map<String, Reference> actual_values;
	
	public Binding(MethodSignature ms) {
		this.ms = ms;
		this.actual_values = new HashMap<String, Reference>();
	}
	
	public String getFullname(){
		return ms.getFullname();
	}
	
	public MethodSignature getMethodSignature() {
		return ms;
	}
	
	public List<String> getParameterNamesNotBoundToConstant() {
		List<String> result = new ArrayList<String>();
		for (String param_name : ms.getParamNames()){
			if(!(actual_values.get(param_name) instanceof ConstantRef))
				result.add(param_name);
		}
		return result;
	}

	public Binding extractConstantBinding(){
		Binding result = new Binding(ms);
		for (String param_name : ms.getParamNames()){
			Reference reference = actual_values.get(param_name);
			if(reference instanceof ConstantRef)
				result.bindParameter(param_name, reference); 
		}
		return result;
	}
	
	/**
	 * Binds formal parameter to value. 
	 * @param param_name - name of formal parameter
	 * @param value - binded value
	 * @throws TBPResolvingException if param_name is already bound or there is a type mismatch
	 */
	public void bindParameter(String param_name, Reference value) {
		if(actual_values.containsKey(param_name)){
			throw new TBPResolvingException("Method parameter " + param_name + " is already bound.");
		}
		
		if(!ms.getParamTypes().get(param_name).equals(value.getType())){
			throw new TBPResolvingException("Method parameter " + param_name + " has type " 
					+ ms.getParamTypes().get(param_name).getName() + " while " + value.getName()
					+ " has type " + value.getType().getName());
		}
		actual_values.put(param_name, value);
	}
	
	/**
	 * Binding of formal parameter.
	 * @param param_name - name of a formal parameter
	 * @return {@link Reference} to value to which is formal parameter bound
	 */
	public Reference getValue(String param_name) {
		Reference result = actual_values.get(param_name);
		assert result != null;
		return result;
	}
	
	/**
	 * List of actual parameters. 
	 */
	public List<Reference> getValues() {
		List<Reference> result = new ArrayList<Reference>(actual_values.size());
		for(String name : ms.getParamNames()){
			result.add(actual_values.get(name));
		}
		return result;
	}
	
	
	/**
	 * 
	 * @return <code>true</code> if all parameters are bound.
	 */
	public boolean isBound(){
		return actual_values.size() == ms.getParamNames().size();
	}
	

	/**
	 * Rebinds formal parameters. Creates new binding. Does not this object.
	 *   
	 * @param b Binding for method to which refers this objects formal parameters.
	 * @return {@link Binding} where formal parameters were substituted.
	 */
	public Binding rebind(Binding b) {
		Binding result = new Binding(ms);
		assert b.isBound();
		
		for(Entry<String, Reference> entry : actual_values.entrySet()){
			Reference ref = entry.getValue();
			if (ref instanceof FPRef) {
				ref = b.actual_values.get(ref.getName());
			}
			result.bindParameter(entry.getKey(), ref);
		}
		
		return result;
	}
	
	@Override
	public String toString() {
		
		int s = actual_values.size();
		if(s > 0){
			StringBuilder bld = new StringBuilder();
			List<String> names= ms.getParamNames();
			bld.append(this.getMethodSignature().getFullname());
			bld.append("(");
			for(int i = 0; i < s-1; ++i){
				String name = names.get(i);
				Reference ref = actual_values.get(name);
				bld.append(name).append(" = ").append(ref.getName()).append(", ");
			}
			String last_name = names.get(s-1);
			bld.append(last_name).append(" = ").append(actual_values.get(last_name).getName()).append(")");
			return bld.toString();
		} else {
			return getMethodSignature().toString();
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		//result = prime * result + actual_values.size();
		result = prime * result + actual_values.hashCode();
		result = prime * result + ms.hashCode();
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof Binding))
			return false;
		final Binding other = (Binding) obj;
		if (!actual_values.equals(other.actual_values))
			return false;
		if (!ms.equals(other.ms))
			return false;
		return true;
	}
}
