/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.parsed.visitor;

import java.util.List;
import java.util.Map;

import org.ow2.dsrg.fm.tbplib.Annotation;
import org.ow2.dsrg.fm.tbplib.EventTable;
import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.parsed.MethodCall;
import org.ow2.dsrg.fm.tbplib.parsed.TBPLimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAccept;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAlternative;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedParallel;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedParallelOr;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionContainerNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionNull;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedRepetition;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSequence;
import org.ow2.dsrg.fm.tbplib.parsed.TBPUnlimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedAlternative;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedLimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedParallel;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedParallelOr;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedProvisionContainerNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedProvisionNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedProvisionNull;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedProvisionSequence;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedRepetition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedAcceptRequest;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmitResponse;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;
import org.ow2.dsrg.fm.tbplib.resolved.util.MethodSignature;
import org.ow2.dsrg.fm.tbplib.resolved.util.UtilClass;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Translates trees of parsed provision nodes into resolved provision nodes. 
 * @author caitt3am
 *
 */
public class ProvisionResolvingVisitor extends TBPParsedCheckingVisitor<TBPResolvedProvisionNode> {
	
	private final List<Typedef> types;
	private Map<String, MethodSignature> sigs;
	private final EventTable et;

	public ProvisionResolvingVisitor(List<Typedef> types,
			Map<String, TBPResolvedVardef> vardefs,
			Map<String, MethodSignature> sigs,
			EventTable et){
		this.types = types;
		this.sigs = sigs;
		this.et = et;
	}
	
	@Override
	public TBPResolvedProvisionNode visitParsedAlternative(TBPParsedAlternative node) {
		return new TBPResolvedAlternative(
					((TBPParsedNode)node.getLeft()).visit(this),
					((TBPParsedNode)node.getRight()).visit(this)
				);
	}

	@Override
	public TBPResolvedProvisionNode visitParsedParallel(TBPParsedParallel node) {
		return new TBPResolvedParallel(
				((TBPParsedNode)node.getLeft()).visit(this),
				((TBPParsedNode)node.getRight()).visit(this)
			);
	}

	@Override
	public TBPResolvedProvisionNode visitParsedParallelOr(TBPParsedParallelOr node) {
		return new TBPResolvedParallelOr(
				((TBPParsedNode)node.getLeft()).visit(this),
				((TBPParsedNode)node.getRight()).visit(this)
			);
	}

	@Override
	public TBPResolvedProvisionNode visitParsedSequence(TBPParsedSequence node) {
		return new TBPResolvedProvisionSequence(
				((TBPParsedNode)node.getLeft()).visit(this),
				((TBPParsedNode)node.getRight()).visit(this)
			);
	}

	@Override
	public TBPResolvedProvisionNode visitParsedRepetition(TBPParsedRepetition node) {
		return new TBPResolvedRepetition(((TBPParsedNode)node.getChild()).visit(this));
	}

	
	@Override
	public TBPResolvedProvisionNode visitLimitedReentrancy(TBPLimitedReentrancy node) {
		TBPParsedNode ch = (TBPParsedNode) node.getChild();
		return new TBPResolvedLimitedReentrancy(ch.visit(this), node.getLimit());
	}

	
	@Override
	public TBPResolvedProvisionNode visitUnlimitedReentrancy(
			TBPUnlimitedReentrancy node) {
		throw new TBPResolvingException("There should be no unlimited " 
				+ "reentrancy ops during resolving");
	}

	@Override
	public TBPResolvedProvisionNode visitParsedProvisionContainerNode(
			TBPParsedProvisionContainerNode node) {
		
		return new TBPResolvedProvisionContainerNode( 
				((TBPParsedProvisionNode) node.getChild()).visit(this), 
				node.getName());
	}


	@Override
	public TBPResolvedProvisionNode visitParsedNull(TBPParsedProvisionNull node) {
		return new TBPResolvedProvisionNull();
	}

	private TBPResolvedProvisionSequence createPairOfEvents(int req_code, int resp_code, Annotation ann){
		TBPResolvedProvisionNode acceptRequest = new TBPResolvedAcceptRequest(req_code);
		TBPResolvedProvisionNode emitResponse = new TBPResolvedEmitResponse(resp_code);
		
		acceptRequest.setAnnotation(ann);
		emitResponse.setAnnotation(ann);
		
		return new TBPResolvedProvisionSequence(acceptRequest, emitResponse);
	}
	
	@Override
	public TBPResolvedProvisionNode visitParsedAccept(TBPParsedAccept node) {
	
		Annotation annotation = node.getAnnotation();
		MethodCall mc = node.getMethodCall();
		String returnValue = node.getReturnValue();
		String fullname = mc.getFullname();
		Binding binding = UtilClass.makeConstantBinding(sigs, mc, types);
		Typedef return_type = binding.getMethodSignature().getReturnType();
		
		final int req_code = et.assignIdToRequest(binding);
		
		if(return_type == null){ // procedure
			
			if(returnValue != null){
				throw new TBPResolvingException("Return value specified, even when method"
						+ fullname + " returns nothing.");
			}
			int resp_code = et.assignIdToResponse(fullname, null);
			return createPairOfEvents(req_code, resp_code, annotation);
			
		} else if(returnValue != null) { // return value specified
			
			ConstantRef ref = UtilClass.makeConstantReference(returnValue, types);
			if(!ref.getType().equals(return_type)){
				throw new TBPResolvingException("Return value " + returnValue +
						" is not part of type " + return_type.getName());
			}
			int resp_code = et.assignIdToResponse(fullname, ref);
			return createPairOfEvents(req_code, resp_code, annotation);
			
		}
		
		TBPResolvedProvisionNode n = null, o = null;
		
		// foldl list of sequences into n using alternative operator
		for(String ret : return_type.getEnums()){ 
			ConstantRef cref = UtilClass.makeConstantReference(ret, types);
			int resp_code= et.assignIdToResponse(fullname,cref);
			
			o = createPairOfEvents(req_code, resp_code, annotation);
			
			if(n == null)
				n = o;
			else 
				n = new TBPResolvedAlternative(n,o);
		}
		
		return n;
	}
	
}