/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.parsed.visitor;

import java.util.List;

import org.ow2.dsrg.fm.tbplib.parsed.MethodCall;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAssignment;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedCondition;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedEmit;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedIf;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSwitch;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSync;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedValue;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedWhile;

public class PrefixingVisitor extends TBPParsedDefaultVisitor<Object> {
	
	private final String prefix;
	private final List<String> varnames;
	
	public PrefixingVisitor(String prefix, List<String> varnames) {
		this.prefix = prefix;
		this.varnames = varnames;
	}

	private final String prefixName(String name){
		if(varnames.contains(name))
			return prefix + name;
		else 
			return name;
	}
	
	@Override
	public Object visitParsedSwitch(TBPParsedSwitch node) {
		if(!node.isNondeterministic())
			node.getValue().visit(this);
		return null; 
	}
	
	@Override
	public Object visitParsedWhile(TBPParsedWhile node) {
		visitCondition(node.getCondition());
		return null;
	}
	
	@Override
	public Object visitParsedIf(TBPParsedIf node) {
		visitCondition(node.getCondition());
		return null;
	}
	
	@Override
	public Object visitParsedSync(TBPParsedSync node) {
		node.setMutex(prefixName(node.getMutex()));
		return null;
	}
	
	@Override
	public Object visitParsedAssignment(TBPParsedAssignment node) {
		node.setIdf(prefixName(node.getIdf()));
		return null;
	}
	
	@Override
	public Object visitParsedValue(TBPParsedValue node) {
		String varname = node.getVarname();
		if(varname != null){
			node.setVarname(prefixName(varname));
		} else {
			visitMethodCall(node.getMethodCall());
		}
		return null;
	}

	@Override
	public Object visitParsedEmit(TBPParsedEmit node) {
		visitMethodCall(node.getMethodCall());
		return null;
	}
	
	private void visitMethodCall(MethodCall methodCall) {
		List<String> paramDecl = methodCall.getParamDecl();
		final int s = paramDecl.size();
		
		for(int i = 0; i < s; i++){
			paramDecl.set(i, prefixName(paramDecl.get(i)));
		}
		
	}

	private void visitCondition(TBPParsedCondition condition) {
		if(condition.getLeft() != null){
			condition.setLeft(prefixName(condition.getLeft()));
			condition.setRight(prefixName(condition.getRight()));
		}
	}
	
}