/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.parsed;

import java.util.ArrayList;
import java.util.List;

/**
 * Helper class holding call to a reaction.
 * 
 * @author caitt3am
 *
 */
public class MethodCall implements Cloneable {
	public static final String SIGN_ACCEPT = "?";
	public static final String SIGN_RETURN_VALUE = ":";
	
	private String ifaceName;
	private String methodName;
	private String fullname;
	private String returnValue;
	private List<String> paramDecl;
	
	public MethodCall(String ifaceName, String methodName,
			List<String> paramDecl, String returnType) {

		this.ifaceName = ifaceName;
		this.methodName = methodName;
		this.paramDecl = paramDecl;
		this.returnValue = returnType;
		recompute_fullname();
	}
	
	public MethodCall(List<String> nameParts, List<String> paramDecl, String returnType) {
		assert nameParts.size() >= 2;
		this.ifaceName = interfaceName(nameParts);
		this.methodName = nameParts.get(nameParts.size()-1); // Last element
		this.paramDecl = paramDecl;
		this.returnValue = returnType;
		recompute_fullname();
	}
	
	/**
	 * Concatenate list of package names in  the name parts and gets name of the interface. 
	 *  Last entry is considered to be method name a thus is not used in the interface name.
	 * 
	 * <p> Ex. ["cz","cuni", "mff", "d3s", "alf", "MyMethod1"] -> "cz.cuni.mff.d3s.alf"
	 * @param nameParts List of package name parts.
	 * @return Concatenated name of the interface
	 */
	static public String interfaceName(List<String> nameParts) {
		assert nameParts != null;
		assert nameParts.size() >= 1;
		
		int parts = nameParts.size();
		if (parts == 1) {
			return ""; // Only method name
		}
		
		StringBuffer result = new StringBuffer();
		result.append(nameParts.get(0));
		for(int i = 1; i < nameParts.size()-1; i++) {
			result.append('.');
			result.append(nameParts.get(i));
		}
		return result.toString();
	}
	
	public List<String> getParamDecl() {
		return paramDecl;
	}

	public String getReturnValue() {
		return returnValue;
	}
	
	public String getInterface() {
		return ifaceName;
	}


	public void setInterface(String ifaceName) {
		this.ifaceName = ifaceName;
		recompute_fullname();
	}
	
	public String getMethod() {
		return methodName;
	}

	public void setMethod(String methodName) {
		this.methodName = methodName;
		recompute_fullname();
	}
	
	public String getFullname() {
		return fullname;
	}
	
	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append(SIGN_ACCEPT);
		result.append(fullname);
		// process parameters
		result.append('(');

		boolean first = true;
		for(String param : paramDecl) {
			if (param == null || param.trim().isEmpty()) {
				continue;
			}
			
			if (first) {
				first = false;
			} else {
				result.append(", ");
			}
			result.append(param);
		}

		result.append(')');

		// Any (nonenmpty) return value is set
		if (returnValue != null && !returnValue.trim().isEmpty()) {
			result.append(SIGN_RETURN_VALUE);
			result.append(returnValue);
		}
		return result.toString();
		
	}

	@Override
	public MethodCall clone() {
		try {
			MethodCall clone = (MethodCall) super.clone();
			clone.paramDecl = new ArrayList<String>(paramDecl);
			return clone;
		} catch (CloneNotSupportedException e) {
			throw new RuntimeException(e);
		}
	}
	
	private void recompute_fullname() {
		fullname = ifaceName + "." + methodName;
	}
}
