package spec.benchmarks._225_shock;
//------------------------------------------------------------------------
// Shock1DNG - a Java standalone, Non GUI program which computes 1-D
// ideal gas shock waves.  Uses a 2nd-Order Godunov Method Finite Difference
// computation.
//
// James C. Liu, Ph.D., Dept. of Nuclear Engineering, University of California
// at Berkeley.
//
// (c) 1997.
//
// Written originally in FORTRAN and ported to Java by:
// James C. Liu, X.M. Chen, PF Peterson, P. Collela, V.E. Schrock
// Dept. of Nuclear Engineering and Dept. of Mech. Engineering, 
// University of California at Berkeley.
//
// Adopted from TSUNAMI (Transient Shockwave Upwind Numerical Analysis Method
// for ICF) code courtesy of James Liu, and UC Berkeley Department of Nuclear
// Engineering.
//
// Algorithms based on CFD methods published by Colella, et al., Dept.
// of Mechanical Engineering, UC Berkeley.
//
// Research Performed by UCB Dept. of Nuclear Engineering are under the
// auspices of the US Dept. of Energy for the Lawrence Livermore National
// Laboratory under contract W-7405-ENG-48.
//------------------------------------------------------------------------
// Permission to use, modify and distribute this source is hereby granted for
// educational and benchmarking purposes provided that this header is included
// in subsequent modifications.
//
// Neither the author(s) nor the University of California and LLNL/DOE make
// any claims of computational accuracy or fitfullness for engineering uses
// or take any responsibility for any damages to users incurred directly or
// indirectly by the usage of this source.  Users accept all the consequences.
//------------------------------------------------------------------------
//-------------------------end header-------------------------------------

//import java.applet.*;
import java.util.*;

// standalone version for shock
public class Shock1DNG /*implements Runnable*/ {


    //---------------------------------------------------------------------
    // data types and declarations for this class
    //---------------------------------------------------------------------
    private Thread t1 = null;

    private double rho[];
    private double u[];
    private double p[];
    private double drho[];
    private double du[];
    private double dp[];
    private double c[];
    private double temprho[];
    private double tempu[];
    private double tempp[];
    private double tempc[];
    private BoundaryThread F[] = null;

    public static final int sleepTicks = 500;
    public static final int ncell = 100;
    public static final double courant = 0.9;
    public static final double zero = 0.0;
    public static final double half = 0.5;
    public static final double one = 1.0;
    public static final double two = 2.0;

    private int timestep;
    private double time;
    private double dtime;
    private double dtout;
    private double toutlast;
    private double dx;
    private double maxu;
    private double maxc;
    private double gamma;

    private ComputeTracker tracker = null;

    //---------------------------------------------------------------------
    // main program - enables this to run as both applet and standalone
    //---------------------------------------------------------------------
    public static void main(String args[]) {
	Shock1DNG shock = new Shock1DNG();
	shock.setInitialConditions();
	shock.run();
    }

    //---------------------------------------------------------------------
    // start() - starts the the main thread of this applet
    //---------------------------------------------------------------------
/*    
    public void start() {
	if (t1==null) t1 = new Thread(this);
	t1.start();
    }
*/

    //---------------------------------------------------------------------
    // run() - the main thread of this applet
    //---------------------------------------------------------------------
    public void run() {
	int limit = 5*spec.harness.Context.getSpeed();
	spec.harness.Context.out.println("Shock1D limit is "+limit );
	while(timestep < limit) {
	    nextTimeStep();
	    if(timestep%10 == 0) spec.harness.Context.out.println("Computing: "+((timestep*100)/500) +"% complete.");
	}
	spec.harness.Context.out.println("Done.");
    }

    //---------------------------------------------------------------------
    // set initial conditions for problem
    //---------------------------------------------------------------------
    protected void setInitialConditions() {

	//----we set some initial conditions--------------
	gamma = 1.4;
	dx = 1.0;
	time = 0.0;
	timestep = 0;

	tracker = new ComputeTracker(ncell+1);
	F = new BoundaryThread[ncell+1];

	rho = new double[ncell+2];
	u = new double[ncell+2];
	p = new double[ncell+2];
	drho = new double[ncell+2];
	du = new double[ncell+2];
	dp = new double[ncell+2];
	c = new double[ncell+2];
	temprho = new double[ncell+2];
	tempu = new double[ncell+2];
	tempp = new double[ncell+2];
	tempc = new double[ncell+2];

 	//---some temporary hard-coded initialization------
	int i;
	for(i=1;i<=ncell/2+1;i++) {
	    rho[i] = temprho[i] = 2.0;
	    u[i] = tempu[i] = 0.0;
	    p[i] = tempp[i] = 1000.0;
	    c[i] = tempc[i] = Math.sqrt(gamma*p[i]/rho[i]);
	}
	for(i=ncell/2+1;i<=ncell;i++) {
	    rho[i] = temprho[i] = 1.0;
	    u[i] = tempu[i] = 0.0;
	    p[i] = tempp[i] = 1.0;
	    c[i] = tempc[i] = Math.sqrt(gamma*p[i]/rho[i]);
	}

	dtout = dx/c[1];
	toutlast = -dtout;

	//---boundary conditions---------------------
	rho[0] = rho[1];
	u[0] = -u[1];
	p[0] = p[1];

	rho[ncell+1] = rho[ncell];
	u[ncell+1] = -u[ncell];
	p[ncell+1] = p[ncell];

	//---get 2nd-Order quantities from VanLeer's MUSCL scheme
	computeVanLeer();

    }

    //---------------------------------------------------------------------
    // execute one time step of the calculation
    //---------------------------------------------------------------------
    public void nextTimeStep() {
	double r, dt;
	long lastTime,diffTime;
	dtime = computeTimeStep();
	computeFluxes();
	tracker.waitForCompletion();
	computeQuantities();
	
	if(maxVelocityOkay()) {
	    System.arraycopy(temprho,0,rho,0,ncell+2);
	    System.arraycopy(tempu,0,u,0,ncell+2);
	    System.arraycopy(tempp,0,p,0,ncell+2);
	    System.arraycopy(tempc,0,c,0,ncell+2);
	    time += dtime;
	    timestep++;

	    r = Math.IEEEremainder(time,dtout);
	    dt = time - toutlast;
	    if(dt > r) {
		toutlast = time;
	    }
	} 
	cleanThreads();
    }

    //---------------------------------------------------------------------
    // Compute a valid timestep based on Courant number and max velocity
    //---------------------------------------------------------------------
    protected double computeTimeStep() {
	double maxv = getMaxSpeed();
	double dt = courant*dx/maxv;
	return dt;
    }

    //---------------------------------------------------------------------
    // check velocity to see if Courant violation has occurred
    //---------------------------------------------------------------------
    protected boolean maxVelocityOkay() {
	double maxv = getMaxSpeed();
	double maxCourant = dtime*maxv/dx;
	return (maxCourant < 1.0);
    }

    //---------------------------------------------------------------------
    // get the fastest speed of any characteristic wave
    //---------------------------------------------------------------------
    protected double getMaxSpeed() {
	double maxv = 0.0;
	for(int i=1;i<=ncell;i++) {
	    if((Math.abs(tempu[i]) + tempc[i]) > maxv) {
		maxu = Math.abs(tempu[i]);
		maxc = tempc[i];
		maxv = maxu + maxc;
	    }
	}
	return maxv;
    }

    //---------------------------------------------------------------------
    // dispatch threads to compute Fluxes via Riemann Solver
    //---------------------------------------------------------------------
    protected void computeFluxes() {
	for(int i=0;i<=ncell;i++) {
	    F[i] = new BoundaryThread(tracker);
	    F[i].setLRConditions(rho[i],rho[i+1],u[i],u[i+1],
		p[i],p[i+1],c[i],c[i+1],drho[i],drho[i+1],
		du[i],du[i+1],dp[i],dp[i+1],dtime,dx,gamma);
	    F[i].setPriority(Thread.MIN_PRIORITY);
	    F[i].start();
	}
    }

    //---------------------------------------------------------------------
    // clean threads which compute Fluxes via Riemann Solver
    //---------------------------------------------------------------------
    protected void cleanThreads() {
	for(int i=0;i<=ncell;i++) {
	    F[i].stop();
	    F[i] = null;
	}
	System.gc();
    }

    //---------------------------------------------------------------------
    // compute new conserved Quantities
    //---------------------------------------------------------------------
    protected void computeQuantities() {

	double dtdx = dtime/dx;
	double gm1 = gamma - 1.0;

	double mass, momentum, energy;

	int i;

	for(i=1; i<=ncell; i++) {

	    mass = rho[i];
	    momentum = rho[i]*u[i];
	    energy = p[i]/gm1 + rho[i]*u[i]*u[i]*half;

	    mass-=(dtdx*(F[i].getMassFlux() - F[i-1].getMassFlux()));
	    momentum-=(dtdx*(F[i].getMomentumFlux()-F[i-1].getMomentumFlux()));
	    energy-=(dtdx*(F[i].getEnergyFlux() - F[i-1].getEnergyFlux()));

	    temprho[i] = Math.max(mass,1.0e-6);
	    tempu[i] = momentum/mass;
	    tempp[i] = Math.max(((energy/mass - tempu[i]*tempu[i]*half)*mass*gm1),1.0e-6);
	    tempc[i] = Math.sqrt(gamma*tempp[i]/temprho[i]);

	}

	//-----adjust for reflective boundary conditions---------------
	temprho[0] = temprho[1];
	tempu[0] = -tempu[1];
	tempp[0] = tempp[1];
	tempc[0] = tempc[1];

	temprho[ncell+1] = temprho[ncell];
	tempu[ncell+1] = -tempu[ncell];
	tempp[ncell+1] = tempp[ncell];
	tempc[ncell+1] = tempc[ncell];

	computeVanLeer();
    }

    //---------------------------------------------------------------------
    // compute 2nd order flux corrected transport from Van Leer's
    // MUSCL scheme, applied in the next timestep if executed
    //---------------------------------------------------------------------
    protected void computeVanLeer() {

	for(int i=1; i<=ncell;i++) {
	    drho[i] = computeFCT(rho,i);
	    du[i] = computeFCT(u,i);
	    dp[i] = computeFCT(p,i);
	}
	drho[0] = du[0] = dp[0] = 0.0;
	drho[ncell+1] = du[ncell+1] = dp[ncell+1] = 0.0;
    }

    //---------------------------------------------------------------------
    // compute 2nd order Flux Corrected Transport
    //---------------------------------------------------------------------
    protected double computeFCT(double q[], int i) {

	double meandiffsquare = (q[i-1]-q[i]) * (q[i]-q[i+1]);

	if(meandiffsquare <= zero) {
	    return zero;
	}

	double minfct, fctl, fctr, fctm;
	int signfct; 

	fctm = Math.abs(half*(q[i-1]-q[i+1]));
	fctl = Math.abs(two*(q[i-1]-q[i]));
	fctr = Math.abs(two*(q[i]-q[i+1]));
	minfct = Math.min(fctl,fctr);
	minfct = Math.min(minfct,fctm);

	signfct = ((q[i+1]-q[i-1]) > zero ) ? 1 : -1;

	return (signfct == 1) ? minfct : -minfct;
	
    }

}
