/**
 * The Dining Philosophers Problem.
 * <P> 
 * The following program is a Java benchmark that will test how quickly a Java
 * virtual machine can context switch between threads. The main thread creates
 * the five philosophers and forks and then yields itself a number of times. As
 * all six threads have the same priority the main thread should get to run
 * 1/6th of the time and therefore the program must complete.
 * <p>
 * This technique would also allow the program to complete of there was a 
 * a deadlock accessing the forks. This however does not occur because
 * the algorithm has each philosopher pick up the lowest numbered fork first and
 * this prevents circular deadlock forming.
 * <p>
 * This program is freely and unconditionally donated to SPEC organization to do
 * with as it sees fit.
 * 
 * @author A well-wisher who wants to remain anonymous.
 */
 
package spec.benchmarks._223_diners;
import spec.harness.*;
 
public class DiningPhilosophersProblem {
    /**
     * The number of active philosophers.
     */
    static int runners = 5;

    /**
     * Decrement the number of runners. This is a static synchronszed 
     * method to insure that there is no possibility of any non-atomic
     * update being performed
     */
     
    public static synchronized void decrementRunners() {
//	Context.out.println("Philosopher "+id+" stopping");
	DiningPhilosophersProblem.runners--;    
    } 

    /**
     * The main routine 
     */
    public static void run(int count) {
    
        Fork fork1 = new Fork(1);
        Fork fork2 = new Fork(2);
        Fork fork3 = new Fork(3);
        Fork fork4 = new Fork(4);
        Fork fork5 = new Fork(5);
        
        Philosopher philosopher1 = new Philosopher(1, fork1, fork2);
        Philosopher philosopher2 = new Philosopher(2, fork2, fork3);
        Philosopher philosopher3 = new Philosopher(3, fork3, fork4);
        Philosopher philosopher4 = new Philosopher(4, fork4, fork5);
        Philosopher philosopher5 = new Philosopher(5, fork5, fork1);

        philosopher1.start();
        philosopher2.start();
        philosopher3.start();
        philosopher4.start();
        philosopher5.start();
        
        for (int i = 0 ; i < count ; i++) {
            Thread.yield();          
        }

        philosopher1.running = false;
        philosopher2.running = false;
        philosopher3.running = false;
        philosopher4.running = false;
        philosopher5.running = false;
	
	while (runners > 0) {
            Thread.yield();	
	}

	spec.io.PrintStream ps = (spec.io.PrintStream)Context.out;
        ps.println('1',"main() count = "+count);
        ps.println("Philosopher one   count = "+philosopher1.cycles+" wait = "+philosopher1.pauses);
        ps.println("Philosopher two   count = "+philosopher2.cycles+" wait = "+philosopher2.pauses);
        ps.println("Philosopher three count = "+philosopher3.cycles+" wait = "+philosopher3.pauses);
        ps.println("Philosopher four  count = "+philosopher4.cycles+" wait = "+philosopher4.pauses);
        ps.println("Philosopher five  count = "+philosopher5.cycles+" wait = "+philosopher5.pauses);        
    }
}

/**
 * The Philosopher class.
 * <p>
 * The deadlock and indefinite postponement problem is solved
 * by always picking up the fork with the lowest id first.
 */
 
class Philosopher extends Thread {
    /**
     * Philosopher id
     */
    int id;
    
    /**
     * Running flag
     */
    boolean running = true;
    	    
    /**
     * Count of the number of times we went around the loop
     */
    int cycles = 0;
    
    /**
     * Count of the number of times we waited
     */
    int pauses = 0;

    /**
     * The first fork to pick up
     */
    Fork first;

    /**
     * The second fork to pick up
     */
    Fork second;

    /**
     * Constructor
     * @param left the fork to the left of the philosopher.
     * @param right the fork to the right of the philosopher.     
     */
    public Philosopher( int id, Fork left, Fork right ) {
	this.id = id;
        if (left.id < right.id) {
            first  = left;
            second = right;
        } else {
            first  = right;
            second = left;        
        }
    }

    /**
     * Run method
     */
    public void run() {
        while( running ) {
            think();
            getForks();
            eat();
            putForks();
        }
	
	DiningPhilosophersProblem.decrementRunners();
    }
    

    /**
     * Get both forks
     */
    public void getForks() {
        first.get(this);
        second.get(this);
    }

    /**
     * Put both forks
     */
    public void putForks() {
        first.put();
        second.put();
    }
    
    /**
     * Think a bit
     */
    public void think() {
        Thread.yield();        
    }

    /**
     * Eat a bit
     */
    public void eat() {
        cycles++;
    }
}

/**
 * The Fork class. This demonstrates the use of wait() and notify() to
 * schedule access to the fork.
 */
 
class Fork {
    /**
     * The id of the fork
     */
    int id;
    
    /**
     * The owner of the fork
     */
    Philosopher owner = null;

    /**
     * Constructor
     * @param id the id of this fork
     */
    public Fork(int id) {
        this.id = id;
    }

    /**
     * Wait indefinitely for the fork to be free.
     * @param hungryPhilosopher the philosopher waiting for the fork.
     */
    synchronized void get( Philosopher hungryPhilosopher ) {
        try {
            while (owner != null) {
                wait();
                hungryPhilosopher.pauses++;            
            }
            owner = hungryPhilosopher;   
        } catch (InterruptedException ex) {
            System.out.println( "InterruptedException" );
            System.exit(1);
        }
    }

    /**
     * Free the fork and notify the other waiter if there is one
     */
    synchronized void put() {
        owner = null;
        notify();
    }
}

