package eu.qimpress.samm.visualisation;

import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;

import ch.randelshofer.tree.TreeNode;
import de.fzi.gast.core.File;
import de.fzi.gast.types.GASTClass;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.CompositeComponent;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.samm.staticstructure.Repository;
import eu.qimpress.samm.staticstructure.SubcomponentInstance;
import eu.qimpress.samm.staticstructure.util.StaticstructureSwitch;
import eu.qimpress.sourcecodedecorator.ComponentImplementingClassesLink;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * Switch implementation for the repository visualization. Only composite 
 * and primitive components are taken care of. The found repository is 
 * considered to be the entry point.
 * @author Henning Hager
 *
 */
public class RepositoryVizSammSwitch extends StaticstructureSwitch<TreeNode> {
	
	private ArrayList<Object> myRep = new ArrayList<Object>();
	private HashMap<String, EObject> idToEObject  = null;
	private int PCNum = 0, CCNum = 0;
	
	private SourceCodeDecoratorRepository sourceCodeDecorator;
	
	private boolean defaultNaming = false;
	
	public RepositoryVizSammSwitch(HashMap<String, EObject> idMap, SourceCodeDecoratorRepository sourceCodeDecorator) {
		super();
		
		this.sourceCodeDecorator = sourceCodeDecorator;
		
		this.idToEObject = idMap;
		
		if(idMap == null || idMap.size() == 0){
			defaultNaming = true;
			System.out.println("INFO: Using fallback behavior for naming");
		}
	}

	/**
	 * Takes care of all found composites. Creates a {@link RepositoryVizNode}
	 * and adds all subcomponents.
	 */
	@Override
	public TreeNode caseCompositeComponent(CompositeComponent component) {
		int thisCCNum = 0, thisPCNum = 0, linesOfCode = 0;
		CCNum++;

		RepositoryVizNode node = new RepositoryVizNode();
		node.setXmiId(component.getId());
		node.setName(component.getName());
		
		addInterfaces(component, node);
		addComponentClasses(component, node);
		
		myRep.remove(component);
		
		//calling all children
		for (SubcomponentInstance sub : component.getSubcomponents()) {
			RepositoryVizNode temp = (RepositoryVizNode) doSwitch(sub.getRealizedBy());
			node.addChild(temp);

			if(temp.isPrimitive()){
				thisPCNum++;
			}else{
				int[] childInfo = temp.getChildInfo();
				thisCCNum += childInfo[0] + childInfo[1] + 1;
			}
			linesOfCode += node.getLinesOfCode();
		}
		
		node.setLinesOfCode(linesOfCode);
		node.setChildInfo(thisPCNum, thisCCNum);
		return node;
	}
	
	/**
	 * Handles found primitive components. Creates a {@link RepositoryVizNode}
	 * and sets it as a primitive.
	 */
	@Override
	public TreeNode casePrimitiveComponent(PrimitiveComponent component) {
		PCNum++;
		RepositoryVizNode node = new RepositoryVizNode();
		node.setXmiId(component.getId());
		node.setPrimitive(true);
		
		if(!defaultNaming){
			File pcFile = (File) idToEObject.get(component.getId());
			if(pcFile != null){
				String pcPath = pcFile.getPathName();
				if(pcPath != null && pcPath != ""){
					try{
						node.setName(pcPath.substring(pcPath.lastIndexOf("\\") + 1, pcPath.indexOf(".")));
					}catch (IndexOutOfBoundsException  e) {
						node.setName(component.getName());
					}
				}else{
					node.setName(component.getName());
				}
				node.setLinesOfCode(pcFile.getLinesOfCode());
			}else{
				node.setName(component.getName());
			}
		}else{
			node.setName(component.getName());
		}
		addInterfaces(component, node);
		addComponentClasses(component, node);
		
		myRep.remove(component);
		return  node;
	}
	
	private void addComponentClasses(ComponentType component, RepositoryVizNode node) {
		if(sourceCodeDecorator == null) {
			node.addComponentClasses("component classes not available (source code decorator not loaded)");
		} else {
			// derive names from source code decorator
			for(ComponentImplementingClassesLink link : sourceCodeDecorator.getComponentImplementingClassesLink()) {
				if(link.getComponent().equals(component)) {
					for(GASTClass gastClass : link.getImplementingClasses()) {
						node.addComponentClasses(gastClass.getSimpleName());
					}
				}
			}
		}
	}
	
	private void addInterfaces(ComponentType object, RepositoryVizNode node) {
		for (InterfacePort providedInterface : object.getProvided()) {
			if(defaultNaming){
				node.addProvidedInterface(providedInterface.getName());
				continue;
			}
			
			GASTClass interfaceClass = (GASTClass) idToEObject.get(providedInterface.getInterfaceType().getId());
			if(interfaceClass != null && interfaceClass.isAbstract()){
				node.addProvidedInterface(interfaceClass.getSurroundingPackage().getQualifiedName() + "." + interfaceClass.getSimpleName());
			}
		}
		
		for (InterfacePort requiredInterface : object.getRequired()) {
			if(defaultNaming){
				node.addRequiredInterface(requiredInterface.getName());
				continue;
			}
			
			GASTClass interfaceClass = (GASTClass) idToEObject.get(requiredInterface.getInterfaceType().getId());
			if(interfaceClass != null && interfaceClass.isAbstract()){
				node.addRequiredInterface(interfaceClass.getSurroundingPackage().getQualifiedName() + "." + interfaceClass.getSimpleName());
			}		
		}
	}

	/**
	 * The handling of the repository is the entry point for the switch. 
	 * The tree is successively build by processing the nodes with the most
	 * depth. 
	 */
	@Override
	public TreeNode caseRepository(Repository object) {
		int depth = 0, thisDepth = 0;
		
		CompositeComponent root = null;
		RepositoryVizNode rootNode = new RepositoryVizNode();
		rootNode.setName(object.eResource().getURI().lastSegment());
		rootNode.setRoot(true);
		
		setupRepository(object);
		
		//keep on building tree until all relevant components are used
		while(!myRep.isEmpty()){
			depth = 0; thisDepth = 0;
			root = null;
			
			//calling all remaining nodes - getting the most nested node
			for (Object comp : myRep) {
				if(comp instanceof CompositeComponent){
					thisDepth = getDepth((CompositeComponent) comp);
					if(depth < thisDepth){
						root = (CompositeComponent) comp;
						depth = thisDepth;
					}
				}
			}
			
			//Adding all components, which are not used in composite hierarchy 
			if(root == null){
				//copying repository to avoid inconsistencies during iteration and removing of components
				myRep.trimToSize();
				Object[] tempRep = myRep.toArray();

				for (int i = 0; i < tempRep.length; i++) {
					if(tempRep[i] instanceof PrimitiveComponent){
						rootNode.addChild((RepositoryVizNode) doSwitch((PrimitiveComponent) tempRep[i]));
					}
				}
			}else{
				//building tree based on most nested composites
				rootNode.addChild((RepositoryVizNode) doSwitch(root));
			}
		}
		rootNode.setChildInfo(PCNum, CCNum);
		return rootNode;
	}
	
	/**
	 * Calculates the max depth of a component by calling getDepth recursively.
	 * @param comp Component to analyze
	 * @return the depth of the given component
	 */
	public int getDepth(CompositeComponent comp){
		int depth = 0;
		TreeIterator<EObject> treeIt = comp.eAllContents();
		
		//calling all children
		for (; treeIt.hasNext();) {
			EObject subComp = treeIt.next();
			if(subComp instanceof SubcomponentInstance){
				Object temp = ((SubcomponentInstance) subComp).getRealizedBy();
				if(temp instanceof CompositeComponent){
					if(depth < getDepth((CompositeComponent) temp) + 1){
						depth = getDepth((CompositeComponent) temp) + 1;
					}
				}
			}
		}
		
		if(depth == 0)
			return 1;
		else
			return depth;
	}
	/**
	 * Trims the complete repository to the needed components for visualization. 
	 * (only Primitive and CompositeComponents are used. 
	 * @param comp
	 */
	public void setupRepository(Repository comp){
		TreeIterator<EObject> treeIt = comp.eAllContents();
		for (; treeIt.hasNext();) {
			EObject relevantComp = treeIt.next();
			if(relevantComp instanceof PrimitiveComponent
			|| relevantComp instanceof CompositeComponent){
				myRep.add(relevantComp);
			}
		}
	}
}
