/**
 * 
 */
package eu.qimpress.transformations.sofa2.workflow;

import java.io.File;
import java.util.ArrayList;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.URI;
import org.objectweb.dsrg.sofa.repository.model.RepositoryData;

import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import eu.qimpress.transformations.sofa2.actions.Generate;
import eu.qimpress.transformations.sofa2.actions.Transform;
import eu.qimpress.transformations.sofa2.actions.Upload;
import eu.qimpress.transformations.sofa2.qvt.SOFA2Library;
import eu.qimpress.transformations.sofa2.ui.TransformConfig;

/**
 * Workflow job transforming the SAMM model instance to SOFA2 instance.
 * 
 * @author Viliam Simko
 */
public class SammSofaJob implements IJob {

	private Logger logger = Logger.getLogger(SammSofaJob.class);

	private TransformConfig config;

	public SammSofaJob(TransformConfig config) {
		this.config = config;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.uka.ipd.sdq.workflow.IJob#getName()
	 */
	@Override
	public String getName() {
		return "SAMM-SOFA2 QVT Transformation job";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seede.uka.ipd.sdq.workflow.IJob#execute(org.eclipse.core.runtime.
	 * IProgressMonitor)
	 */
	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException,
			UserCanceledException {

		logger.info(getName());
		logger.info("Using SAMM instance    : "
				+ config.getSammInstanceFile().getAbsolutePath());
		logger.info("Using SOFA2 repository : "
				+ config.getSofaRepoAgent().getNodeConfiguration()
						.getSofaNodeName());

		try {

			// total number of tasks
			monitor.beginTask(
					"Running the SAMM model to SOFA2 model transformation", 4);

			// =================================================================
			// TASK
			// =================================================================

			monitor
					.subTask("Testing the connection to the SOFA2 repository...");
			config.getSofaRepoAgent().getQuery().getRepositoryDataObject();
			logger
					.info("Connection to the SOFA2 repository seems to be ok, let's proceed");

			// =================================================================
			// TASK
			// =================================================================

			monitor.subTask("Applying the QVT-O transformation...");

			// HACK: using static variables within the QVT black-box
			SOFA2Library
			.setupQvtBlackbox(config.getSofaRepoAgent().getFacade());

			// the transformation requires a running SOFA2 repository
			Transform transform = new Transform();
			RepositoryData data = transform.doTransform(config
					.getSammInstanceFile());

			// HACK: using static variable within the QVT black-box to get error
			// messages
			if (!SOFA2Library.blackBoxErrors.isEmpty()) {
				logger.error("Transformation failed. Cleaning up the repository might help.");

				// logging the aggregated error messages
				for (String errorMessage : SOFA2Library.blackBoxErrors) {
					logger.error("SOFA2 BLACKBOX:" + errorMessage);
				}

				throw new JobFailedException(
						"Errors detected during the execution of the SOFA2 blackbox within the QVT-O transformation");
			}

			logger.info("Transformation successful");
			monitor.worked(1);
			if (monitor.isCanceled())
				throw new UserCanceledException();

			// =================================================================
			// TASK
			// =================================================================

			monitor.subTask("Generating the SOFA2 package...");

			File folder = config.getTransformTmpDir();

			// we assume that the URI is a local file
			URI sammInstanceUri = URI.createFileURI(config
					.getSammInstanceFile().getAbsolutePath());

			Generate generate = new Generate(sammInstanceUri, folder,
					new ArrayList<String>());

			generate.doGenerate(new BasicMonitor());
			
			logger.info("Code generation successful");
			monitor.worked(1);
			if (monitor.isCanceled())
				throw new UserCanceledException();

			// =================================================================
			// TASK
			// =================================================================

			monitor.subTask("Uploading the package to the SOFA2 repository...");

			Upload export = new Upload(config.getSofaRepoAgent(), folder);
			export.doUpload(data);

			logger.info("Upload successful");
			monitor.worked(1);
			if (monitor.isCanceled())
				throw new UserCanceledException();

			monitor.done();

		} catch (UserCanceledException e) {
			throw e;
		} catch (Exception e) {
			throw new JobFailedException(e.getMessage(), e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seede.uka.ipd.sdq.workflow.IJob#rollback(org.eclipse.core.runtime.
	 * IProgressMonitor)
	 */
	@Override
	public void rollback(IProgressMonitor monitor)
			throws RollbackFailedException {
		try {
			cleanupTransformDir();
		} catch (Exception e) {
			throw new RollbackFailedException(e.getMessage());
		}
	}

	/**
	 * Helper method that cleans the content of the temporary directory.
	 * Generated java and jar files are stored there. Note: we assume that the
	 * directory always contains regular files.
	 */
	private void cleanupTransformDir() {

		File folder = config.getTransformTmpDir();

		for (File entry : folder.listFiles()) {
			entry.delete();
		}
		folder.delete();

		logger.info("Job cleanup finished");
	}
}
