package eu.qimpress.transformations.sofa2.qvt;

import java.util.ArrayList;

import org.apache.log4j.Logger;
import org.eclipse.m2m.qvt.oml.blackbox.java.Operation;
import org.objectweb.dsrg.sofa.repository.RepositoryFacade;
import org.objectweb.dsrg.sofa.repository.model.Architecture;
import org.objectweb.dsrg.sofa.repository.model.Frame;
import org.objectweb.dsrg.sofa.repository.model.InterfaceType;

/**
 * This class represents the QVT-O black-box used in the transformation. It
 * converts SAMM names into SOFA2 names by automatically creating the entities
 * within the SOFA2 repository.
 * 
 * Note: we use a nasty hack here
 * {@link SOFA2Library#setupQvtBlackbox(RepositoryFacade)}
 * 
 * @author Viliam Simko
 */
public class SOFA2Library {

	final private static Logger logger = Logger.getLogger(SOFA2Library.class);

	/**
	 * External reference to the SOFA2 repository. Usually passed to the
	 * black-box from the external transformation job.
	 */
	private static RepositoryFacade facade;

	/**
	 * List of all error messages aggregated during the transformation. Empty
	 * list indicates successful transformation.
	 * 
	 * @see {@link SOFA2Library#setupQvtBlackbox(RepositoryFacade)}
	 */
	final public static ArrayList<String> blackBoxErrors = new ArrayList<String>();

	/**
	 * Some hacks to prepare the QVT-O black-box.
	 * 
	 * HACK #1: We use static variables to allow communication between the
	 * black-box and the job that runs the transformation.
	 * 
	 * A reference to the SOFA2 repository (facade) is required and all error
	 * messages that appeared during the transformation need to be stored. This
	 * is because the transformation mechanism wouldn't propagate exceptions to
	 * the running job.
	 * 
	 * HACK #2: We had to use the Object type as a method parameter rather than
	 * RepositoryFacade, because the QVT-O framework wouldn't find the class in
	 * the CLASSPATH.
	 * 
	 * @param sofaRepoFacade
	 */
	final public static void setupQvtBlackbox(Object sofaRepoFacade) {
		logger.info("Setting up the SOFA2 QVT-O blackbox");

		// empty list indicates successful transformation
		SOFA2Library.blackBoxErrors.clear();

		// facade for later use during the transformation
		SOFA2Library.facade = (RepositoryFacade) sofaRepoFacade;
	}

	/**
	 * Creates new interface type.
	 * 
	 * @param name
	 *            interface type name
	 * @return interface type
	 */
	@Operation
	public InterfaceType createInterfaceType(String name) {
		try {
			logger.info("Creating Interface Type : " + name);
			return facade.createInterfaceType(String.format("I%s", name), name);
		} catch (Exception e) {
			blackBoxErrors.add(e.getMessage());
		}
		return null;
	}

	/**
	 * Creates new frame.
	 * 
	 * @param name
	 *            frame name
	 * @return frame
	 */
	@Operation
	public Frame createFrame(String name) {
		try {
			logger.info("Creating Frame : " + name);
			return facade.createFrame(String.format("F%s", name));
		} catch (Exception e) {
			blackBoxErrors.add(e.getMessage());
		}
		return null;
	}

	/**
	 * Creates new architecture.
	 * 
	 * @param name
	 *            architecture name
	 * @param _implements
	 *            implemented frame
	 * @return architecture
	 */
	@Operation
	public Architecture createArchitecture(String name, Frame _implements) {
		try {
			logger.info("Creating Architecture : " + name);
			return facade.createArchitecture(String.format("A%s", name),
					_implements);
		} catch (Exception e) {
			blackBoxErrors.add(e.getMessage());
		}
		return null;
	}

}
