package eu.qimpress.transformations.sofa2.actions;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;
import javax.tools.ToolProvider;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.objectweb.dsrg.sofa.repository.CodeBundleHelper;
import org.objectweb.dsrg.sofa.repository.RepositoryAgent;
import org.objectweb.dsrg.sofa.repository.RepositoryFacade;
import org.objectweb.dsrg.sofa.repository.RepositoryPersister;
import org.objectweb.dsrg.sofa.repository.model.Architecture;
import org.objectweb.dsrg.sofa.repository.model.CodeBundle;
import org.objectweb.dsrg.sofa.repository.model.Frame;
import org.objectweb.dsrg.sofa.repository.model.InterfaceType;
import org.objectweb.dsrg.sofa.repository.model.RepositoryData;

public class Upload {
	
	/** Entity's code suffix. */
	public static String CODE_SUFFIX = ".code";
	
	/** Default buffer size. */
	private static final int IO_BUFFER_SIZE = 4 * 1024;

	private RepositoryAgent agent;
	private File files;

	public Upload(RepositoryAgent agent, File files) {
		this.agent = agent;
		this.files = files;
	}
	
	/**
	 * Upload entities into repository.
	 * @param data data to upload
	 * @throws Exception
	 */
	public void doUpload(RepositoryData data) throws Exception {
		RepositoryPersister persister = agent.getPersister();
		RepositoryFacade facade = agent.getFacade();
		
		ResourceSet resourceSet = new ResourceSetImpl();
		File output = new File(files, "repositoryData.xml");
		Resource outResource = resourceSet.createResource(
				URI.createURI(output.toURI().toString(), false));
		outResource.getContents().add(data);
		outResource.save(Collections.emptyMap());
		
		for (InterfaceType interfaceType: data.getInterfaceType()) {
			File bundleFile = processBundle(interfaceType);
			if (bundleFile != null) {
				CodeBundle bundle = facade.createCodeBundle(interfaceType.getName() + CODE_SUFFIX, interfaceType);
				persister.save(bundle);
				CodeBundleHelper.uploadFile(bundleFile, bundle);
				interfaceType.setCodeBundle(bundle);
			}
			persister.save(interfaceType);
		}
		
		for (Frame frame: data.getFrame()) {
			persister.save(frame);
		}
		
		for (Architecture architecture: data.getArchitecture()) {
			persister.save(architecture);
		}
	}
	
	/**
	 * Create code bundle for given entity.
	 * @param interfaceType entity to create bundle for
	 * @return code bundle file
	 * @throws IOException
	 */
	private File processBundle(InterfaceType interfaceType) throws IOException {
		File sourceFile = new File(files, interfaceType.getSignature() + ".java");
		if (!sourceFile.exists()) {
			return null;
		}
		File classFile = compile(sourceFile);
		File jarFile = new File(files, interfaceType.getSignature() + ".jar");
		
		OutputStream output = new BufferedOutputStream(new FileOutputStream(jarFile));
		JarOutputStream jarStream = new JarOutputStream(output, new Manifest());
		
		jarStream.putNextEntry(new JarEntry(classFile.getName()));
		copy(new BufferedInputStream(new FileInputStream(classFile)), jarStream);
		jarStream.closeEntry();
		
		jarStream.close();
		
		return jarFile;
	}
	
	/**
	 * Compile given source file.
	 * @param file the source file
	 * @return binary output file
	 */
	private File compile(File file) {
		JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
		StandardJavaFileManager fileManager = compiler.getStandardFileManager(null, null, null);
		Iterable<? extends JavaFileObject> compilationUnits = fileManager.getJavaFileObjects(file);
		compiler.getTask(null, fileManager, null, null, null, compilationUnits).call();
		return new File(files, file.getName().replace(".java", ".class"));
	}
	
	/**
     * Copy the content of the input stream into the output stream, using a temporary
     * byte array buffer whose size is defined by {@link #IO_BUFFER_SIZE}.
     * @param in the input stream to copy from
     * @param out the output stream to copy to
     * @throws IOException if any error occurs during the copy
     */
	private static void copy(InputStream in, OutputStream out) throws IOException {
		byte[] b = new byte[IO_BUFFER_SIZE];
		int read;
		while ((read = in.read(b)) != -1) {
			out.write(b, 0, read);
		}
	}

}
