package eu.qimpress.transformations.samm2pcm.multiple;

import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import de.uka.ipd.sdq.pipesandfilters.framework.recorder.launch.IRecorderConfiguration;
import de.uka.ipd.sdq.pipesandfilters.framework.recorder.sensorframework.launch.SensorFrameworkConfig;
import de.uka.ipd.sdq.sensorframework.SensorFrameworkDataset;
import de.uka.ipd.sdq.sensorframework.entities.Experiment;
import de.uka.ipd.sdq.sensorframework.entities.ExperimentRun;
import de.uka.ipd.sdq.sensorframework.entities.Measurement;
import de.uka.ipd.sdq.sensorframework.entities.Sensor;
import de.uka.ipd.sdq.sensorframework.entities.SensorAndMeasurements;
import de.uka.ipd.sdq.sensorframework.entities.TimeSpanMeasurement;
import de.uka.ipd.sdq.sensorframework.entities.dao.IDAOFactory;
import de.uka.ipd.sdq.workflow.IBlackboardInteractingJob;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.MDSDBlackboard;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.ResourceSetPartition;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.samm.usagemodel.UsageModel;
import eu.qimpress.samm.usagemodel.UsageRepository;
import eu.qimpress.samm.usagemodel.UsageScenario;
import eu.qimpress.transformations.common.jobs.LoadSAMMIntoBlackboardJob;
import eu.qimpress.transformations.samm2pcm.Samm2PcmUtil;
import eu.qimpress.transformations.samm2pcm.ui.SAMM2PCMConfiguration;

public class StorePcmResultsToCsvJob implements IJob, IBlackboardInteractingJob<MDSDBlackboard> {

	private MDSDBlackboard blackboard;
	private SAMM2PCMConfiguration config;
	
	private static final Logger logger = Logger.getLogger(StorePcmResultsToCsvJob.class);

	public StorePcmResultsToCsvJob(SAMM2PCMConfiguration config) {
		super();
		this.config = config;
	}

	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException, UserCanceledException {
		if ((config.getSimuComWorkflowConfiguration() == null) || (config.getSimuComWorkflowConfiguration().getSimuComConfiguration() == null)) {
			throw new JobFailedException("Failed to store results. No data source specified!");
		}
		IDAOFactory daoFactory = null;
		IRecorderConfiguration recorderConfiguration =  config.getSimuComWorkflowConfiguration().getSimuComConfiguration().getRecorderConfig();
		if (recorderConfiguration != null) {
			if (recorderConfiguration instanceof SensorFrameworkConfig) {
				long dataSourceId =  ((SensorFrameworkConfig)recorderConfiguration).getDatasourceID();
				daoFactory = SensorFrameworkDataset.singleton().getDataSourceByID(dataSourceId);
			}
		}
		if (daoFactory == null) {
			throw new JobFailedException("Failed to load data source.");
		}
		
		Experiment experiment = null;
		if (daoFactory.createExperimentDAO().findByExperimentName(config.getSimuComWorkflowConfiguration().getSimuComConfiguration().getNameExperimentRun()).size() == 1) {
			experiment = daoFactory.createExperimentDAO().findByExperimentName(config.getSimuComWorkflowConfiguration().getSimuComConfiguration().getNameExperimentRun()).iterator().next();
		} else {
			throw new JobFailedException("More than one experiment available.");
		}
		if (experiment == null) {
			throw new JobFailedException("Cannot load experiment.");
		}
		UsageRepository sammUsageRepository = getUsageRepositoryFromBlackboard();
		if (sammUsageRepository == null) {
			throw new JobFailedException("No Usage Repository available.");
		}
		UsageModel sammUsage = null;
		if (config.getUsageModelId() == null) {
			sammUsage = sammUsageRepository.getUsageModels().get(0);
		} else {
			sammUsage = getUsageModelFromUsageRepository(sammUsageRepository, config.getUsageModelId());
		}
		if (sammUsage == null) {
			throw new JobFailedException("SAMM usage model not found.");
		}
		
		// Get the latest experiment run from the experiment
		ExperimentRun experimentRun = getLatestExperimentRun(experiment.getExperimentRuns());
		for (Sensor sensor : getResponseTimeSensors(experiment)) {
			String fileName = sensor.getSensorName();
			storeTimeSpanSensorResultsToCsv(experimentRun.getMeasurementsOfSensor(sensor), Samm2PcmUtil.getFileNameForSensor(sensor.getSensorName()));
		}
		/*for (UsageScenario scenario : sammUsage.getUsageScenarios()) {
			Sensor sensor = getSensorForUsageScenario(experiment, scenario);
			if (sensor == null) {
				logger.warn("No sensor found for usage scenario " + scenario.getName());
			} else {
				storeTimeSpanSensorResultsToCsv(experimentRun.getMeasurementsOfSensor(sensor), scenario);
			}
		}*/
		//SensorFrameworkDataset.singleton().removeDataSource(daoFactory);
		daoFactory.finalizeAndClose();
	}

	@Override
	public String getName() {
		return "Store PCM Results to Result Model";
	}
	
	
	private void storeTimeSpanSensorResultsToCsv(SensorAndMeasurements sensorAndMeasurements, UsageScenario usageScenario) {
		String sFileName = config.getAlternative().getRepository().getQProject().getProject().getLocation().toOSString();
		sFileName = sFileName + System.getProperty("file.separator") + config.getAlternative().getInfo().getId() + "_" + config.getAlternative().getInfo().getDescription() + "_" + Samm2PcmUtil.getUsageScenarioSimulationName(usageScenario) + ".csv";
		storeTimeSpanSensorResultsToCsv(sFileName, sensorAndMeasurements);		    
	}
	
	private void storeTimeSpanSensorResultsToCsv(SensorAndMeasurements sensorAndMeasurements, String sensorName) {
		String sFileName = config.getAlternative().getRepository().getQProject().getProject().getLocation().toOSString();
		sFileName = sFileName + System.getProperty("file.separator") + config.getAlternative().getInfo().getId() + "_" + config.getAlternative().getInfo().getDescription() + "_" + sensorName + ".csv";
		storeTimeSpanSensorResultsToCsv(sFileName, sensorAndMeasurements);		    
	}
	
	
	private void storeTimeSpanSensorResultsToCsv(String fileName, SensorAndMeasurements sensorAndMeasurements) {
		try
		{
		    FileWriter writer = new FileWriter(fileName);
 
			writer.append("Event Time");
			writer.append(';');
			writer.append("Time Span");
			writer.append('\n');
			
			for (Measurement measurement : sensorAndMeasurements.getMeasurements()) {
				if (measurement instanceof TimeSpanMeasurement) {
					writer.append(Double.toString(((TimeSpanMeasurement)measurement).getEventTime()));
					writer.append(';');
					writer.append(Double.toString(((TimeSpanMeasurement)measurement).getTimeSpan()));
					writer.append('\n');
				}
			}
  
			writer.flush();
			writer.close();
		}
		catch(IOException e)
		{
			logger.error("Failed to write results to csv file.", e);
		}
	}


	@Override
	public void rollback(IProgressMonitor monitor) throws RollbackFailedException {
	}

	@Override
	public void setBlackboard(MDSDBlackboard blackboard) {
		this.blackboard = blackboard;
	}

	/**
	 * Retrieve the latest experiment run (= the experiment run with the highest ID) from a collection of experiment runs 
	 * @param experimentRuns
	 * @return
	 */
	private ExperimentRun getLatestExperimentRun(Collection<ExperimentRun> experimentRuns) {
		Iterator<ExperimentRun> experimentRunIterator = experimentRuns.iterator();
		ExperimentRun latestExperimentRun = null;
		while (experimentRunIterator.hasNext()) {
			if (latestExperimentRun != null) {
				ExperimentRun tmpExperimentRun = experimentRunIterator.next();
				if (latestExperimentRun.getExperimentRunID() < tmpExperimentRun.getExperimentRunID()) {
					latestExperimentRun = tmpExperimentRun;
				}
			} else {
				latestExperimentRun = experimentRunIterator.next();
			}
		}
		return latestExperimentRun;
	}
	
	private UsageRepository getUsageRepositoryFromBlackboard() {
		ResourceSetPartition partition = blackboard.getPartition(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID);
		ResourceSet resourceSet = partition.getResourceSet();
		EList<Resource> resources = resourceSet.getResources();
		for (Resource r : resources) {
			if (r.getURI().fileExtension().equals(RepositoryModels.USAGE_MODEL_EXT)) {
				for (EObject resourceObject : r.getContents()) {
					if (resourceObject instanceof UsageRepository) {
						return (UsageRepository)resourceObject;
					}
				}
			}
		}
		return null;
	}
	
	private UsageModel getUsageModelFromUsageRepository(UsageRepository usageRepository, String usageModelId) {
			if ((usageRepository == null) || (usageRepository.getUsageModels() == null)) {
				return null;
			}
			for (UsageModel model : usageRepository.getUsageModels()){
				if (model.getId().equals(usageModelId)) {
					return model;
				}
			}		
			return null;
	}
	
		
	private Sensor getSensorForUsageScenario(Experiment experiment, UsageScenario scenario) {
		Collection<Sensor> sensors = experiment.getSensors();
		String usageScenarioSimulationName = Samm2PcmUtil.getUsageScenarioSimulationName(scenario);
		for (Sensor sensor : sensors) {
			if (sensor.getSensorName().equals("Response Time of " + usageScenarioSimulationName)) {
				logger.info("Found sensor " + sensor.getSensorName() + " for result model.");
				return sensor; 
			}
		}	
		return null;
		
	}
	
	private List<Sensor> getResponseTimeSensors(Experiment experiment) {
		ArrayList<Sensor> responseTimeSensors = new ArrayList<Sensor>();
		Collection<Sensor> sensors = experiment.getSensors();
		for (Sensor sensor : sensors) {
			//System.out.println(sensor.getSensorName());
			if (sensor.getSensorName().startsWith("Response Time")) {
				logger.info("Found sensor " + sensor.getSensorName() + " for result model.");
				responseTimeSensors.add(sensor);
			}
		}
		return responseTimeSensors;
	}
	
	

}
