package eu.qimpress.transformations.samm2pcm.jobs;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import de.uka.ipd.sdq.pipesandfilters.framework.recorder.launch.IRecorderConfiguration;
import de.uka.ipd.sdq.pipesandfilters.framework.recorder.sensorframework.launch.SensorFrameworkConfig;
import de.uka.ipd.sdq.sensorframework.SensorFrameworkDataset;
import de.uka.ipd.sdq.sensorframework.entities.Experiment;
import de.uka.ipd.sdq.sensorframework.entities.ExperimentRun;
import de.uka.ipd.sdq.sensorframework.entities.Measurement;
import de.uka.ipd.sdq.sensorframework.entities.Sensor;
import de.uka.ipd.sdq.sensorframework.entities.SensorAndMeasurements;
import de.uka.ipd.sdq.sensorframework.entities.StateMeasurement;
import de.uka.ipd.sdq.sensorframework.entities.TimeSpanMeasurement;
import de.uka.ipd.sdq.sensorframework.entities.dao.IDAOFactory;
import de.uka.ipd.sdq.workflow.IBlackboardInteractingJob;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.MDSDBlackboard;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.ResourceSetPartition;
import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.ide.backbone.core.operations.SaveQModelUIOperation;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.CpuResourceUtilization;
import eu.qimpress.resultmodel.HddResourceUtilization;
import eu.qimpress.resultmodel.PerformancePredictionResult;
import eu.qimpress.resultmodel.ResponseTime;
import eu.qimpress.resultmodel.ResultDistribution;
import eu.qimpress.resultmodel.ResultModelFactory;
import eu.qimpress.resultmodel.Throughput;
import eu.qimpress.samm.deployment.targetenvironment.Container;
import eu.qimpress.samm.deployment.targetenvironment.ExecutionResource;
import eu.qimpress.samm.deployment.targetenvironment.Node;
import eu.qimpress.samm.deployment.targetenvironment.StorageResource;
import eu.qimpress.samm.deployment.targetenvironment.TargetEnvironment;
import eu.qimpress.samm.usagemodel.UsageModel;
import eu.qimpress.samm.usagemodel.UsageRepository;
import eu.qimpress.samm.usagemodel.UsageScenario;
import eu.qimpress.transformations.common.jobs.LoadSAMMIntoBlackboardJob;
import eu.qimpress.transformations.samm2pcm.Samm2PcmUtil;
import eu.qimpress.transformations.samm2pcm.ui.SAMM2PCMConfiguration;

public class StorePcmResultsToResultModelJob implements IJob, IBlackboardInteractingJob<MDSDBlackboard> {

	private MDSDBlackboard blackboard;
	private SAMM2PCMConfiguration config;
	private IQRepository repository = null;

	private static final Logger logger = Logger.getLogger(StorePcmResultsToResultModelJob.class);

	public StorePcmResultsToResultModelJob(SAMM2PCMConfiguration config) {
		super();
		this.config = config;
	}

	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException, UserCanceledException {
		if ((config.getSimuComWorkflowConfiguration() == null) || (config.getSimuComWorkflowConfiguration().getSimuComConfiguration() == null)) {
			throw new JobFailedException("Failed to store results. No data source specified!");
		}
		IDAOFactory daoFactory = null;
		IRecorderConfiguration recorderConfiguration =  config.getSimuComWorkflowConfiguration().getSimuComConfiguration().getRecorderConfig();
		if (recorderConfiguration != null) {
			if (recorderConfiguration instanceof SensorFrameworkConfig) {
				long dataSourceId =  ((SensorFrameworkConfig)recorderConfiguration).getDatasourceID();
				daoFactory = SensorFrameworkDataset.singleton().getDataSourceByID(dataSourceId);
			}
		}
		if (daoFactory == null) {
			throw new JobFailedException("Failed to load data source.");
		}
		Experiment experiment = null;
		if (daoFactory.createExperimentDAO().findByExperimentName(config.getSimuComWorkflowConfiguration().getSimuComConfiguration().getNameExperimentRun()).size() == 1) {
			experiment = daoFactory.createExperimentDAO().findByExperimentName(config.getSimuComWorkflowConfiguration().getSimuComConfiguration().getNameExperimentRun()).iterator().next();
		} else {
			throw new JobFailedException("More than one experiment available.");
		}
		if (experiment == null) {
			throw new JobFailedException("Cannot load experiment.");
		}
		UsageRepository sammUsageRepository = getUsageRepositoryFromBlackboard();
		if (sammUsageRepository == null) {
			throw new JobFailedException("No Usage Repository available.");
		}
		UsageModel sammUsage = null;
		sammUsage = getUsageModelFromUsageRepository(sammUsageRepository, config.getUsageModelId());
		if (sammUsage == null) {
			throw new JobFailedException("SAMM usage model not found.");
		}
		AlternativeEvaluation alternativeEvaluation = getAlternativeEvaluation();
		if (alternativeEvaluation == null) {
			throw new JobFailedException("Result Model alternative evaluation not found.");
		}
		PerformancePredictionResult performancePredictionResult = ResultModelFactory.eINSTANCE.createPerformancePredictionResult();
		// Get the latest experiment run from the experiment
		ExperimentRun experimentRun = getLatestExperimentRun(experiment.getExperimentRuns());
		for (UsageScenario scenario : sammUsage.getUsageScenarios()) {
			Sensor sensor = getSensorForUsageScenario(experiment, experimentRun, scenario);
			if (sensor == null) {
				logger.warn("No sensor found for usage scenario " + scenario.getName());
			} else {
				storeTimeSpanSensorResultsToResultModel(experimentRun.getMeasurementsOfSensor(sensor), performancePredictionResult, scenario);
			}
		}
		TargetEnvironment sammTargetEnvironment = getTargetEnvironmentFromBlackboard();
		if (sammTargetEnvironment == null) {
			throw new JobFailedException("SAMM target environment not found.");
		}
		for (Node node : sammTargetEnvironment.getNodes()) {
			for (Container container : node.getContainers()) {
				for (ExecutionResource executionResource : container.getExecutionResources()) {
					Sensor sensor = getSensorForExecutionResource(experiment, experimentRun, container, executionResource);
					if (sensor == null) {
						logger.warn("No sensor found for CPU in container " + container.getName());
					} else {
						storeUtilizationSensorResultsToResultModel(experimentRun.getMeasurementsOfSensor(sensor), performancePredictionResult, executionResource);
					}
				}

				for (StorageResource storageResource : container.getStorageResources()) {
					Sensor sensor = getSensorForStorageResource(experiment, experimentRun, container, storageResource);
					if (sensor == null) {
						logger.warn("No sensor found for HDD in container " + container.getName());
					} else {
						storeUtilizationSensorResultsToResultModel(experimentRun.getMeasurementsOfSensor(sensor), performancePredictionResult, storageResource);
					}
				}

			}
		}
		alternativeEvaluation.getAnalysisResults().add(performancePredictionResult);
		try {
			new SaveQModelUIOperation(repository.getResultModel()).run(monitor);
		} catch (RepositoryException e) {
			throw new JobFailedException("Failed to save result model.");
		} catch (InvocationTargetException e) {
			throw new JobFailedException("Failed to save result model.");
		} catch (InterruptedException e) {
			throw new JobFailedException("Failed to save result model.");
		}
	}

	@Override
	public String getName() {
		return "Store PCM Results to Result Model";
	}

	private void storeUtilizationSensorResultsToResultModel(SensorAndMeasurements sensorAndMeasurements,
			PerformancePredictionResult performancePredictionResult, ExecutionResource executionResource) {
		ResultDistribution resultDistribution = calculateUtilization(sensorAndMeasurements);
		CpuResourceUtilization cpuResourceUtilization = ResultModelFactory.eINSTANCE.createCpuResourceUtilization();
		cpuResourceUtilization.setResultDistribution(resultDistribution);
		cpuResourceUtilization.setExecutionResource(executionResource);
		performancePredictionResult.getResourceUtilizations().add(cpuResourceUtilization);
	}
	
	private void storeUtilizationSensorResultsToResultModel(SensorAndMeasurements sensorAndMeasurements,
			PerformancePredictionResult performancePredictionResult, StorageResource storageResource) {
		ResultDistribution resultDistribution = calculateUtilization(sensorAndMeasurements);
		HddResourceUtilization hddResourceUtilization = ResultModelFactory.eINSTANCE.createHddResourceUtilization();
		hddResourceUtilization.setResultDistribution(resultDistribution);
		hddResourceUtilization.setStorageResource(storageResource);
		performancePredictionResult.getHddResourceUtilizations().add(hddResourceUtilization);
	}
	
	private ResultDistribution calculateUtilization(SensorAndMeasurements sensorAndMeasurements) {
		double lastTime = 0.0;
		// 0 - nothing, 1 = idle, 2 = not idle
		int lastTimeSensorState = 0;
		double timeOfIdle = 0.0;
		double timeOfNotIdle = 0.0;
		for (Measurement measurement : sensorAndMeasurements.getMeasurements()) {
			if (measurement instanceof StateMeasurement) {
				if (lastTimeSensorState == 1) {
					timeOfIdle += (measurement.getEventTime() - lastTime);
				} else if (lastTimeSensorState == 2) {
					timeOfNotIdle += (measurement.getEventTime() - lastTime);
				}
				if (((StateMeasurement) measurement).getSensorState().getStateLiteral().toLowerCase().startsWith("idle")) {
					lastTimeSensorState = 1;
				} else {
					lastTimeSensorState = 2;

				}
				lastTime = measurement.getEventTime();
			}
		}
		double percentageIdle = (timeOfIdle / (timeOfIdle + timeOfNotIdle)) * 100;
		double result = 0.0;
		if (percentageIdle <= 0.0) {
			result = 100.0;
		} else if (percentageIdle < 100.0) {
			result = (100.0 - percentageIdle);
		}
		ResultDistribution resultDistribution = ResultModelFactory.eINSTANCE.createResultDistribution();
		resultDistribution.setArithmeticMean(result);
		resultDistribution.setMedian(result);
		resultDistribution.set_10PercentQuantile(result);
		resultDistribution.set_90PercentQuantile(result);
		return resultDistribution;
	}

	private void storeTimeSpanSensorResultsToResultModel(SensorAndMeasurements sensorAndMeasurements, PerformancePredictionResult performancePredictionResult,
			UsageScenario usageScenario) {
		List<TimeSpanMeasurement> timeSpanMeasurements = new ArrayList<TimeSpanMeasurement>();
		for (Measurement measurement : sensorAndMeasurements.getMeasurements()) {
			if (measurement instanceof TimeSpanMeasurement) {
				timeSpanMeasurements.add((TimeSpanMeasurement) measurement);
			}
		}
		if (timeSpanMeasurements.size() == 0) {
			logger.warn("No sensor data available for response time sensor " + sensorAndMeasurements.getSensor().getSensorName());
		}
		ResponseTime responseTime = ResultModelFactory.eINSTANCE.createResponseTime();
		ResultDistribution resultDistribution = ResultModelFactory.eINSTANCE.createResultDistribution();
		// arithmetic mean
		resultDistribution.setArithmeticMean(calculateArithmeticMean(timeSpanMeasurements));
		// sort list by time spans
		timeSpanMeasurements = sortByTimeSpan(timeSpanMeasurements);
		// median
		resultDistribution.setMedian(calculateMedian(timeSpanMeasurements));
		resultDistribution.set_10PercentQuantile(calculate10Quantile(timeSpanMeasurements));
		resultDistribution.set_90PercentQuantile(calculate90Quantile(timeSpanMeasurements));
		responseTime.setResultDistribution(resultDistribution);
		responseTime.setUsageScenario(usageScenario);
		performancePredictionResult.getResponseTimes().add(responseTime);
		
		// calculate throughput
		List<TimeSpanMeasurement> timeSpanMeasurementsSortedByEventTime = sortByEventTime(timeSpanMeasurements);
		ResultDistribution resultDistributionThroughput = ResultModelFactory.eINSTANCE.createResultDistribution();
		double firstEventTime = getStartTimeOfEarliestEvent(timeSpanMeasurementsSortedByEventTime);
		double lastEventTime = getEndTimeOfLatestEvent(timeSpanMeasurementsSortedByEventTime);
		double throughputValue =   0.0;
		if ((lastEventTime-firstEventTime) > 0) {
			throughputValue = timeSpanMeasurementsSortedByEventTime.size()/(lastEventTime-firstEventTime);
		}
		resultDistributionThroughput.set_10PercentQuantile(throughputValue);
		resultDistributionThroughput.set_90PercentQuantile(throughputValue);
		resultDistributionThroughput.setArithmeticMean(throughputValue);
		resultDistributionThroughput.setMedian(throughputValue);
		Throughput throughput = ResultModelFactory.eINSTANCE.createThroughput();
		throughput.setResultDistribution(resultDistributionThroughput);
		throughput.setUsageScenario(usageScenario);
		performancePredictionResult.getThroughputs().add(throughput);
	}
	
	private double getEndTimeOfLatestEvent(List<TimeSpanMeasurement> timeSpanMeasurements) {
		// Assume that list is sorted by event times
		// Assume that the event time is recorded after the demand has finished,
		// i.e. the event end time = event time  
		if ((timeSpanMeasurements == null) || (timeSpanMeasurements.size() == 0)) {
			return 0.0;
		}
		return timeSpanMeasurements.get(timeSpanMeasurements.size()-1).getEventTime();
	}
	
	private double getStartTimeOfEarliestEvent(List<TimeSpanMeasurement> timeSpanMeasurements) {
		// Assume that list is sorted by event times
		// Assume that the event time is recorded after the demand has finished,
		// i.e. the event start time = event time - time span
		if ((timeSpanMeasurements == null) || (timeSpanMeasurements.size() == 0)) {
			return 0.0;
		}
		double earliestTime = -1;
		double evTimeStart = 0.0;
		for (TimeSpanMeasurement measurement : timeSpanMeasurements) {
			evTimeStart = measurement.getEventTime()-measurement.getTimeSpan();
			if (earliestTime < 0) {
				earliestTime = evTimeStart;
			} else {
				if (evTimeStart < earliestTime) {
					earliestTime = evTimeStart;
				}
			}
		}
		return earliestTime;
	}
	
	

	/**
	 * Get the AlternativeEvaluation object from the backbone that has been specified for the run.
	 * 
	 * @return
	 */
	private AlternativeEvaluation getAlternativeEvaluation() {
		if ((config.getAlternativeEvaluationId() == null) || (config.getAlternativeEvaluationProjectName() == null)) {
			logger.warn("Run does not contain a set alternative evaluation.");
			return null;
		}
		IQProject project = QImpressCore.getQProject(config.getAlternativeEvaluationProjectName());
		if (project == null) {
			logger.warn("Project of alternative evaluation is not available.");
			return null;
		}
		try {
			repository = project.getRepository();
			return (getAlternativeEvaluationForId(repository.getAllAlternativeEvaluations(), config.getAlternativeEvaluationId()));
		} catch (RepositoryException e) {
			logger.warn("Failed to access alternative evaluation from repository.");
			return null;
		}
	}

	/**
	 * Get the AlternativeEvaluation object from the backbone for a given id. specified for the run.
	 * 
	 * @return
	 */
	private AlternativeEvaluation getAlternativeEvaluationForId(List<AlternativeEvaluation> alternativeEvaluations, String alternativeEvaluationId) {
		if ((alternativeEvaluationId == null) || (alternativeEvaluations == null) || (alternativeEvaluations.size() == 0)) {
			return null;
		}
		for (AlternativeEvaluation alternativeEvaluation : alternativeEvaluations) {
			if (alternativeEvaluation.getId().equals(alternativeEvaluationId)) {
				return alternativeEvaluation;
			}
		}
		return null;
	}

	private double calculateArithmeticMean(List<TimeSpanMeasurement> timeSpanMeasurements) {
		double tmp = 0.0;
		if ((timeSpanMeasurements == null) || (timeSpanMeasurements.size() == 0)) {
			return tmp;
		}
		for (TimeSpanMeasurement timeSpanMeasurement : timeSpanMeasurements) {
			tmp += timeSpanMeasurement.getTimeSpan();
		}
		tmp = tmp / timeSpanMeasurements.size();
		return tmp;
	}

	private List<TimeSpanMeasurement> sortByTimeSpan(List<TimeSpanMeasurement> timeSpanMeasurements) {
		Collections.sort(timeSpanMeasurements, new Comparator<TimeSpanMeasurement>() {

			@Override
			public int compare(TimeSpanMeasurement o1, TimeSpanMeasurement o2) {
				if (o1 == null)
					return -1;
				else {
					try {
						return (new Double(o1.getTimeSpan()).compareTo(new Double(o2.getTimeSpan())));
					} catch (Exception e) {
						return 0;
					}
				}
			}

		});
		return timeSpanMeasurements;
	}
	
	private List<TimeSpanMeasurement> sortByEventTime(List<TimeSpanMeasurement> timeSpanMeasurements) {
		Collections.sort(timeSpanMeasurements, new Comparator<TimeSpanMeasurement>() {

			@Override
			public int compare(TimeSpanMeasurement o1, TimeSpanMeasurement o2) {
				if (o1 == null)
					return -1;
				else {
					try {
						return (new Double(o1.getEventTime()).compareTo(new Double(o2.getEventTime())));
					} catch (Exception e) {
						return 0;
					}
				}
			}

		});
		return timeSpanMeasurements;
	}

	// Assume the list is sorted
	private double calculateMedian(List<TimeSpanMeasurement> timeSpanMeasurements) {
		if ((timeSpanMeasurements == null) || (timeSpanMeasurements.size() == 0)) {
			return 0.0;
		} else if (timeSpanMeasurements.size() % 2 == 1) {
			// odd-length, return middle element
			return timeSpanMeasurements.get(timeSpanMeasurements.size() / 2).getTimeSpan();
		} else {
			// even-length, return average of middle two elements
			// be sure to divide by 2.0 and not by 2!
			return (timeSpanMeasurements.get(timeSpanMeasurements.size() / 2).getTimeSpan() + timeSpanMeasurements.get((timeSpanMeasurements.size() / 2) - 1).getTimeSpan()) / 2.0;
		}
	}

	// Assume the list is sorted
	private double calculate10Quantile(List<TimeSpanMeasurement> timeSpanMeasurements) {
		if ((timeSpanMeasurements == null) || (timeSpanMeasurements.size() == 0)) {
			return 0.0;
		}
		return timeSpanMeasurements.get(timeSpanMeasurements.size() / 10).getTimeSpan();
	}

	// Assume the list is sorted
	private double calculate90Quantile(List<TimeSpanMeasurement> timeSpanMeasurements) {
		if ((timeSpanMeasurements == null) || (timeSpanMeasurements.size() == 0)) {
			return 0.0;
		}
		if (((timeSpanMeasurements.size() / 10) * 9) > timeSpanMeasurements.size()) {
			return timeSpanMeasurements.get(timeSpanMeasurements.size()).getTimeSpan();
		} else {
			return timeSpanMeasurements.get((timeSpanMeasurements.size() / 10) * 9).getTimeSpan();
		}
	}

	@Override
	public void rollback(IProgressMonitor monitor) throws RollbackFailedException {
	}

	@Override
	public void setBlackboard(MDSDBlackboard blackboard) {
		this.blackboard = blackboard;
	}

	/**
	 * Retrieve the latest experiment run (= the experiment run with the highest ID) from a collection of experiment runs
	 * 
	 * @param experimentRuns
	 * @return
	 */
	private ExperimentRun getLatestExperimentRun(Collection<ExperimentRun> experimentRuns) {
		Iterator<ExperimentRun> experimentRunIterator = experimentRuns.iterator();
		ExperimentRun latestExperimentRun = null;
		while (experimentRunIterator.hasNext()) {
			if (latestExperimentRun != null) {
				ExperimentRun tmpExperimentRun = experimentRunIterator.next();
				if (latestExperimentRun.getExperimentRunID() < tmpExperimentRun.getExperimentRunID()) {
					latestExperimentRun = tmpExperimentRun;
				}
			} else {
				latestExperimentRun = experimentRunIterator.next();
			}
		}
		return latestExperimentRun;
	}

	private UsageRepository getUsageRepositoryFromBlackboard() {
		ResourceSetPartition partition = blackboard.getPartition(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID);
		ResourceSet resourceSet = partition.getResourceSet();
		EList<Resource> resources = resourceSet.getResources();
		for (Resource r : resources) {
			if (r.getURI().fileExtension().equals(RepositoryModels.USAGE_MODEL_EXT)) {
				for (EObject resourceObject : r.getContents()) {
					if (resourceObject instanceof UsageRepository) {
						return (UsageRepository) resourceObject;
					}
				}
			}
		}
		return null;
	}

	private UsageModel getUsageModelFromUsageRepository(UsageRepository usageRepository, String usageModelId) {
		if ((usageRepository == null) || (usageRepository.getUsageModels() == null)) {
			return null;
		}
		for (UsageModel model : usageRepository.getUsageModels()) {
			if (model.getId().equals(usageModelId)) {
				return model;
			}
		}
		return null;
	}

	private TargetEnvironment getTargetEnvironmentFromBlackboard() {
		ResourceSetPartition partition = blackboard.getPartition(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID);
		ResourceSet resourceSet = partition.getResourceSet();
		EList<Resource> resources = resourceSet.getResources();
		for (Resource r : resources) {
			if (r.getURI().fileExtension().equals(RepositoryModels.TARGET_ENVIRONMENT_MODEL_EXT)) {
				for (EObject resourceObject : r.getContents()) {
					if (resourceObject instanceof TargetEnvironment) {
						return (TargetEnvironment) resourceObject;
					}
				}
			}
		}
		return null;
	}

	private Sensor getSensorForUsageScenario(Experiment experiment, ExperimentRun experimentRun, UsageScenario scenario) {
		Collection<Sensor> sensors = experiment.getSensors();
		for (Sensor sensor : sensors) {
			if (sensor.getSensorName().equals("Response Time of " + Samm2PcmUtil.getUsageScenarioSimulationName(scenario))) {
				if (experimentRun.getMeasurementsOfSensor(sensor) != null) {
					if ((experimentRun.getMeasurementsOfSensor(sensor).getMeasurements() != null) && (experimentRun.getMeasurementsOfSensor(sensor).getMeasurements().size() > 0)) {
						logger.info("Found sensor " + sensor.getSensorName() + " for result model.");
						return sensor;
					}
				}
			}
		}
		return null;

	}

	private Sensor getSensorForExecutionResource(Experiment experiment, ExperimentRun experimentRun, Container container, ExecutionResource executionResource) {
		Collection<Sensor> sensors = experiment.getSensors();
		for (Sensor sensor : sensors) {
			//ATTENTION: Currently, we assume only one execution resource per container exists!
			//if (sensor.getSensorName().startsWith(experiment.getExperimentName() + ": Utilisation of " + container.getName() + " [CPU]")) {
			if (sensor.getSensorName().startsWith("Overall Utilisation of " + container.getName() + " [CPU]")) {
				if (experimentRun.getMeasurementsOfSensor(sensor) != null) {
					if ((experimentRun.getMeasurementsOfSensor(sensor).getMeasurements() != null) && (experimentRun.getMeasurementsOfSensor(sensor).getMeasurements().size() > 0)) {
						logger.info("Found sensor " + sensor.getSensorName() + " for result model.");
						return sensor;
					}
				}
			}
		}
		return null;
	}

	private Sensor getSensorForStorageResource(Experiment experiment, ExperimentRun experimentRun, Container container, StorageResource storageResource) {
		Collection<Sensor> sensors = experiment.getSensors();
		for (Sensor sensor : sensors) {
			//ATTENTION: Currently, we assume only one storage resource per container exists!
			//if (sensor.getSensorName().startsWith(experiment.getExperimentName() + ": Utilisation of " + container.getName() + " [HDD]")) {
			if (sensor.getSensorName().startsWith("Overall Utilisation of " + container.getName() + " [HDD]")) {
				if (experimentRun.getMeasurementsOfSensor(sensor) != null) {
					if ((experimentRun.getMeasurementsOfSensor(sensor).getMeasurements() != null) && (experimentRun.getMeasurementsOfSensor(sensor).getMeasurements().size() > 0)) {
						logger.info("Found sensor " + sensor.getSensorName() + " for result model.");
						return sensor;
					}
				}
			}
		}
		return null;
	}

}
