package eu.qimpress.transformations.rpg2sam.sam;

import java.util.List;

import org.dom4j.Element;

import eu.qimpress.samm.behaviour.BehaviourFactory;
import eu.qimpress.samm.behaviour.SeffBehaviourStub;
import eu.qimpress.samm.deployment.hardware.MemoryDescriptor;
import eu.qimpress.samm.deployment.hardware.ProcessorDescriptor;
import eu.qimpress.samm.deployment.targetenvironment.Container;
import eu.qimpress.samm.deployment.targetenvironment.ExecutionResource;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.seff.AcquireAction;
import eu.qimpress.seff.InternalAction;
import eu.qimpress.seff.LoopAction;
import eu.qimpress.seff.ProbabilisticBranchTransition;
import eu.qimpress.seff.ReleaseAction;
import eu.qimpress.seff.ResourceDemandingSEFF;
import eu.qimpress.transformations.rpg2sam.ide.Configuration;
import eu.qimpress.transformations.rpg2sam.rpg.Architecture;
import eu.qimpress.transformations.rpg2sam.rpg.Measurements;

/**
 * The model importer.
 *
 * <p> The structure of the model importer is as follows. The input to the import is provided by two source
 * classes, {@link Architecture} and {@link Measurements}. The output of the import is stored in multiple
 * model classes, each taking care of a particular aspect of the model. The classes are strictly separated,
 * the role of the importer is to interconnect the classes as necessary to facilitate the import.  
 *  
 * <p> The import process creates multiple elements of the constituent models at various locations of the architecture,
 * and therefore also at various places of the recursive import code. There are two contexts in which this occurs:
 * 
 * <p> - A model needs to create an element that conceptually belongs to it, but is created by a foreign factory.
 *       When this is the case, the code simply calls the foreign factory directly.
 *       
 * <p> - A model needs to create an element that conceptually belongs to another model.
 * 		 When this is the case, the code calls the importer callback, which calls the foreign model, which calls the foreign factory.
 * 
 * <p> This helps factor out the code that couples the constituent models together from the code that builds the models themselves.
 * 
 * <p> Some rules for the design of the source classes and the model classes follow:
 * <p> - The source classes understand the semantics of the input.
 *       For example, {@code getMethodDuration} is preferred to {@code getElementAttribute}. 
 * <p> - The model classes do not care what is happening in callbacks.  
 *       For example, {@code importer.primitiveActionCreated} is preferred to {@code importer.addTimeToPrimitiveAction}.
 * <p> - The model classes call the source classes directly to obtain the input.
 *       For example, {@code module.getChildren} is preferred to {@code importer.getChildren (module)}.
 * <p> - Both the source classes and the model classes understand the importer configuration.
 *       For example, {@code getMethodDuration} is preferred to {@code getRandomMethodDuration}, {@code getSharedMethodDuration}, {@code getIsolatedMethodDuration}.
 */
public class Importer
{
	protected final Configuration configuration;

	protected final Architecture architecture;
	protected final Measurements measurements;
	
	public final AnnotationsModel annotationsModel;
	public final ArchitectureModel architectureModel;
	public final BehaviorModel behaviorModel;
	public final DeploymentModel deploymentModel;
	public final DomainModel domainModel;
	public final HardwareModel hardwareModel;
	public final WorkloadModel usageModel;

	/**
	 * The constructor imports an architecture.
	 *
	 * @param architectureName Name of the architecture file to import.
	 * @param measurementName Name of the measurement file to import.
	 */
	public Importer (String architectureName, String measurementName, Configuration configurationContext) throws Exception  
	{
		// The configuration is potentially used by the other model constructors. 
		configuration = configurationContext;
		
		architecture = new Architecture (architectureName, configuration);
    	measurements = new Measurements (measurementName, configuration);

    	annotationsModel = new AnnotationsModel (this);
    	architectureModel = new ArchitectureModel (this);
    	behaviorModel = new BehaviorModel (this);
    	deploymentModel = new DeploymentModel (this);
    	domainModel = new DomainModel (this);
    	hardwareModel = new HardwareModel (this);
    	usageModel = new WorkloadModel (this);

    	// Some parameters are obtained from measurements.
		int userCount = (int) Math.round (measurements.getGlobalAverage (Measurements.RPG_MEASUREMENT_ATTRIBUTE_CLIENTS));
		int threadCount = (int) Math.round (measurements.getGlobalAverage (Measurements.RPG_MEASUREMENT_ATTRIBUTE_THREADS));
		double thinkTime = measurements.getGlobalAverage (Measurements.RPG_MEASUREMENT_ATTRIBUTE_THINKTIME) / (double) 1000000;

		// The component type structure is fetched as one composite component.
		// Another component type is the thread pool component.
    	List<Element> moduleInstances = architecture.getAllModules ();
    	ComponentType rootComponentType = architectureModel.createComponentTypes (moduleInstances);
    	ComponentType poolComponentType = architectureModel.createThreadPoolType (threadCount);

		// The service instance is created from the composite component and the thread pool component.
		deploymentModel.createServices (rootComponentType, poolComponentType);
		
		// The usage scenario is created from the number of clients.
		InterfacePort systemProvidedPort = deploymentModel.model.getProvided ().get (0);
		usageModel.createUsageScenario (systemProvidedPort, userCount, thinkTime);
	}
	
	//----------------------------------------------------------------------
	
	protected void callbackProcessingActionCreated (Element context, InternalAction action) throws Exception
	{
		// When a processing action is created, its timing needs to be specified.
		annotationsModel.annotateActionTime (context, action);
		// When a processing action is created, its reliability needs to be specified.
		annotationsModel.annotateActionReliability (context, action);
	}

	protected void callbackAcquireReleaseActionsCreated (Element context, PrimitiveComponent component, AcquireAction acquire, ReleaseAction release)
	{
		// When a pair of acquire and release actions is created, its resource needs to be specified.
		annotationsModel.annotateActionMonitorResource (context, component, acquire, release);
	}
	
	protected void callbackThreadPoolActionsCreated (PrimitiveComponent component, AcquireAction acquire, ReleaseAction release)
	{
		// When a thread pool is created, its resource needs to be specified.
		annotationsModel.annotateActionThreadPoolResource (component, acquire, release);
	}
	
	protected void callbackLoopingActionCreated (Element context, LoopAction action) throws Exception
	{
		// When a looping action is created, its loop count needs to be specified.
		annotationsModel.annotateLoopCount (context, action);
	}

	protected void callbackBranchingTransitionCreated (Element context, ProbabilisticBranchTransition branchTransition, int branchCount) throws Exception
	{
		// When a branching action is created, its branch probabilities need to be specified.
		annotationsModel.annotateBranchTransitionProbability (context, branchTransition, branchCount);
	}

	protected void callbackPrimitiveComponentTypeCreated (Element context, PrimitiveComponent component) throws Exception
	{
		// When a primitive component type is created, its behavior needs to be specified.
		// The behavior of architectural modules is the architectural behavior.
		// The behavior of leaf modules is the processing behavior.

		ResourceDemandingSEFF behavior;

		// The distinction between architectural modules and leaf modules is the presence of children.
		// The coalescing process can decide to turn an architectural module into a leaf
		// module by removing its children, the type is therefore not always important.
		
		if (component.getRequired ().isEmpty ())
		{
			behavior = behaviorModel.getProcessingBehavior (context, component);	
		}
		else
		{
			String type = architecture.getModuleType (context);
			
			if (type.equals (Architecture.RPG_MODULE_CLASS_LOOP))
			{
				// A loop component whose behavior is looping behavior.
				behavior = behaviorModel.getLoopingBehavior (context, component);
			}
			else if (type.equals (Architecture.RPG_MODULE_CLASS_BRANCH))
			{
				// A branch component whose behavior is branching behavior.
				behavior = behaviorModel.getBranchingBehavior (context, component);
			}
			else if (type.equals (Architecture.RPG_MODULE_CLASS_SEQUENCE))
			{
				// A sequence component whose behavior is sequential behavior.
				behavior = behaviorModel.getSequentialBehavior (context, component);
			}
			else throw new Exception ("Unknown architectural module.");
		}

		SeffBehaviourStub stub = BehaviourFactory.eINSTANCE.createSeffBehaviourStub ();
		stub.setOperation (architectureModel.getModuleOperationType ());
		behavior.setSeffBehaviourStub (stub);
		component.getOperationBehaviour ().add (stub);
	}

	protected void callbackThreadPoolComponentTypeCreated (PrimitiveComponent component)
	{
		// When a thread pool component type is created, its behavior needs to be specified.

		ResourceDemandingSEFF behavior = behaviorModel.getThreadPoolBehavior (component);
		SeffBehaviourStub stub = BehaviourFactory.eINSTANCE.createSeffBehaviourStub ();
		stub.setOperation (architectureModel.getModuleOperationType ());
		behavior.setSeffBehaviourStub (stub);
		component.getOperationBehaviour ().add (stub);
	}
	
	/*
	 * Some trivial singleton instances are also provided through callbacks.
	 * 
	 */

	protected Operation callbackGetOperationType () { return (architectureModel.getModuleOperationType ()); }
	protected Container callbackGetContainerSingleton () { return (hardwareModel.getContainerInstance ()); }
	protected ExecutionResource callbackGetExecutionResourceSingleton () { return (hardwareModel.getExecutionResourceInstance ()); }
	protected ProcessorDescriptor callbackGetProcessorDescriptorSingleton () { return (domainModel.getProcessorDescriptorInstance ()); }
	protected MemoryDescriptor callbackGetMemoryDescriptorSingleton () { return (domainModel.getMemoryDescriptorInstance ()); }
}
