package eu.qimpress.transformations.rpg2sam.sam;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.dom4j.Element;

import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.seff.AbstractAction;
import eu.qimpress.seff.AcquireAction;
import eu.qimpress.seff.BranchAction;
import eu.qimpress.seff.ExternalCallAction;
import eu.qimpress.seff.InternalAction;
import eu.qimpress.seff.LoopAction;
import eu.qimpress.seff.ProbabilisticBranchTransition;
import eu.qimpress.seff.ReleaseAction;
import eu.qimpress.seff.ResourceDemandingSEFF;
import eu.qimpress.seff.SeffRepository;
import eu.qimpress.seff.StartAction;
import eu.qimpress.seff.StopAction;
import eu.qimpress.seff.seffFactory;

public class BehaviorModel
{
	/**
	 * The model container is accessible as a field.
	 */
	public final SeffRepository model = seffFactory.eINSTANCE.createSeffRepository ();

	/**
	 * The import context for this model.
	 * 
	 * The import context provides links to other models that will be created from the same input.
	 */
	private Importer importer;
	
	/**
	 * The model factory is invoked through a proxy 
	 * that takes care of adding all model elements
	 * to the model container.
	 */
	class FactoryProxy implements InvocationHandler
	{
		@Override
		public Object invoke (Object self, Method method, Object[] arguments) throws Throwable
		{
			Object element = method.invoke (seffFactory.eINSTANCE, arguments);
			if (element instanceof ResourceDemandingSEFF) model.getResourceDemandingSeff ().add ((ResourceDemandingSEFF) element);
			return (element);			
		}
	}
	
	private final seffFactory factory = (seffFactory) Proxy.newProxyInstance (
		seffFactory.eINSTANCE.getClass ().getClassLoader (),
		new Class [] { seffFactory.class },
		new FactoryProxy ());

	//----------------------------------------------------------------------

	private enum BehaviorContainment { LOCAL, GLOBAL };
	
	/**
	 * Guess what. A constructor. 
	 */
	protected BehaviorModel (Importer importContext)
	{
		importer = importContext;
	}

	/**
	 * Connects the given actions into a sequence
	 * by setting the predecessor and successor
	 * properties.
	 * 
	 * @param actions
	 */
	private void connectActions (List<AbstractAction> actions)
	{
		AbstractAction lastAction = null;
		for (AbstractAction thisAction : actions)
		{
			if (lastAction != null)
			{
				lastAction.setSuccessor_AbstractAction (thisAction);
				thisAction.setPredecessor_AbstractAction (lastAction);
			}
			lastAction = thisAction;
		}
	}

	/**
	 * Creates a behavior describing external invocations.
	 * 
	 * This method can create either a local or a global behavior.
	 * Unlike global behaviors, local behaviors are meant to be
	 * contained by another behavior, and therefore are not
	 * registered with the model container.
	 *  
	 * @param ports The list of the external invocations.
	 * @param containment Local or global behavior.
	 * @return The behavior description.
	 */
	private ResourceDemandingSEFF getExternalInvocationSequenceBehavior (List<InterfacePort> ports, BehaviorContainment containment)
	{
		// Create the actions to describe external invocations.
		
		List<AbstractAction> allActions = new LinkedList<AbstractAction> ();
		
		StartAction startAction = factory.createStartAction ();
		startAction.setName ("Start");
		allActions.add (startAction);

		for (InterfacePort port : ports)
		{
			ExternalCallAction externalAction = factory.createExternalCallAction ();
			externalAction.setCalledInterfacePort (port);
			externalAction.setCalledService (importer.architectureModel.getModuleOperationType ());
			externalAction.setName ("External");
			
			allActions.add (externalAction);
		}
		
		StopAction stopAction = factory.createStopAction ();
		stopAction.setName ("Stop");
		allActions.add (stopAction);
	
		// Connect the actions into a sequence.
		
		connectActions (allActions);

		// Create either local or global behavior.

		ResourceDemandingSEFF behavior = null;
		switch (containment)
		{
			case LOCAL:
				behavior = seffFactory.eINSTANCE.createResourceDemandingSEFF ();
				break;
			case GLOBAL:
				behavior = factory.createResourceDemandingSEFF ();
				break;
		}
		behavior.getSteps ().addAll (allActions);

		return (behavior);
	}
	
	/**
	 * Creates a processing behavior.
	 * 
	 * The processing behavior consists of an internal action with attached duration.
	 * Synchronized components acquire and release monitor lock too.
	 */
	protected ResourceDemandingSEFF getProcessingBehavior (Element context, PrimitiveComponent component) throws Exception
	{
		// Determine whether the component is synchronized.
		
		boolean monitor = !component.getPassiveResources ().isEmpty ();
		
		// Create the actions to describe processing.
		
		List<AbstractAction> actions = null; 
		
		StartAction startAction = factory.createStartAction ();
		startAction.setName ("Start");

		InternalAction processingAction = factory.createInternalAction ();
		processingAction.setName ("Processing");
		
		StopAction stopAction = factory.createStopAction ();
		stopAction.setName ("Stop");

		if (monitor)
		{
			AcquireAction acquireAction = null;
			ReleaseAction releaseAction = null;

			acquireAction = factory.createAcquireAction ();
			acquireAction.setName ("Acquire");
			
			releaseAction = factory.createReleaseAction ();
			releaseAction.setName ("Release");
			
			actions = Arrays.asList (new AbstractAction [] { startAction, acquireAction, processingAction, releaseAction, stopAction });
			
			// Notify the importer about the important elements that were created.
			
			importer.callbackAcquireReleaseActionsCreated (context, component, acquireAction, releaseAction);
		}
		else
		{
			actions = Arrays.asList (new AbstractAction [] { startAction, processingAction, stopAction });
		}
		
		// Connect the actions into a sequence and set the sequence as the behavior.
		
		ResourceDemandingSEFF behavior = factory.createResourceDemandingSEFF ();
		connectActions (actions);
		behavior.getSteps ().addAll (actions);

		// Notify the importer about the important elements that were created.
		
		importer.callbackProcessingActionCreated (context, processingAction);

		return (behavior);
	}

	/**
	 * Creates a thread pool behavior.
	 * 
	 * The processing behavior consists of acquire and release actions surrounding an external action.
	 */
	protected ResourceDemandingSEFF getThreadPoolBehavior (PrimitiveComponent component)
	{
		// Create the actions to describe thread pool.
		
		StartAction startAction = factory.createStartAction ();
		startAction.setName ("Start");

		AcquireAction acquireAction = factory.createAcquireAction ();
		acquireAction.setName ("Acquire");
		
		ExternalCallAction externalAction = factory.createExternalCallAction ();
		externalAction.setCalledInterfacePort (component.getRequired ().get (0));
		externalAction.setCalledService (importer.architectureModel.getModuleOperationType ());
		externalAction.setName ("External");

		ReleaseAction releaseAction = factory.createReleaseAction ();
		releaseAction.setName ("Release");

		StopAction stopAction = factory.createStopAction ();
		stopAction.setName ("Stop");

		// Connect the actions into a sequence and set the sequence as the behavior.

		List<AbstractAction> actions = Arrays.asList (new AbstractAction [] { startAction, acquireAction, externalAction, releaseAction, stopAction });
			
		ResourceDemandingSEFF behavior = factory.createResourceDemandingSEFF ();
		connectActions (actions);
		behavior.getSteps ().addAll (actions);

	    // Notify the importer about the important elements that were created.
			
		importer.callbackThreadPoolActionsCreated (component, acquireAction, releaseAction);

		return (behavior);
	}
	
	/**
	 * Creates a looping behavior.
	 * 
	 * The looping behavior consists of a loop action with attached sequence of external actions.
	 */
	protected ResourceDemandingSEFF getLoopingBehavior (Element context, PrimitiveComponent component) throws Exception
	{
		// Create the actions to describe looping.
		
		StartAction startAction = factory.createStartAction ();
		startAction.setName ("Start");
		
		LoopAction loopingAction = factory.createLoopAction ();
		ResourceDemandingSEFF internalBehavior = getExternalInvocationSequenceBehavior (component.getRequired (), BehaviorContainment.LOCAL);
		loopingAction.setBodyBehaviour (internalBehavior);
		loopingAction.setName ("Looping");

		StopAction stopAction = factory.createStopAction ();
		stopAction.setName ("Stop");
	
		// Connect the actions into a sequence.
		
		List<AbstractAction> actions = Arrays.asList (new AbstractAction [] { startAction, loopingAction, stopAction });
		connectActions (actions);

		// Set the sequence as the behavior.
		
		ResourceDemandingSEFF externalBehavior = factory.createResourceDemandingSEFF ();
		externalBehavior.getSteps ().addAll (actions);

		// Notify the importer about the important elements that were created.
		
		importer.callbackLoopingActionCreated (context, loopingAction);

		return (externalBehavior);
	}

	/**
	 * Creates a branching behavior.
	 * 
	 * The branching behavior consists of a branch action with attached forks to external actions.  
	 */
	protected ResourceDemandingSEFF getBranchingBehavior (Element context, PrimitiveComponent component) throws Exception
	{
		// Create the actions to describe branching.
		
		StartAction startAction = factory.createStartAction ();
		startAction.setName ("Start");
		
		BranchAction branchingAction = factory.createBranchAction ();
		branchingAction.setName ("Branching");

		int branchingCount = component.getRequired ().size ();
		
		for (InterfacePort requiredPort : component.getRequired ())
		{
			ProbabilisticBranchTransition transition = factory.createProbabilisticBranchTransition ();
            List<InterfacePort> requiredList = Arrays.asList (new InterfacePort [] { requiredPort });
            ResourceDemandingSEFF internalBehavior = getExternalInvocationSequenceBehavior (requiredList,  BehaviorContainment.LOCAL);
			transition.setResourceDemandingBehaviour (internalBehavior);
			transition.setName ("Branch");
			branchingAction.getAbstractBranchTransition ().add (transition);

			// Notify the importer about the important elements that were created.
			
			importer.callbackBranchingTransitionCreated (context, transition, branchingCount);
		}
		
		StopAction stopAction = factory.createStopAction ();
		stopAction.setName ("Stop");
	
		// Connect the actions into a sequence.
		
		List<AbstractAction> actions = Arrays.asList (new AbstractAction [] { startAction, branchingAction, stopAction });
		connectActions (actions);

		// Set the sequence as the behavior.
		
		ResourceDemandingSEFF externalBehavior = factory.createResourceDemandingSEFF ();
		externalBehavior.getSteps ().addAll (actions);

		return (externalBehavior);
	}

	/**
	 * Creates a sequential behavior.
	 * 
	 * The sequential behavior consists of a sequence of external actions.  
	 */
	protected ResourceDemandingSEFF getSequentialBehavior (Element context, PrimitiveComponent component)
	{
		ResourceDemandingSEFF externalBehavior = getExternalInvocationSequenceBehavior (component.getRequired (), BehaviorContainment.GLOBAL);

		return (externalBehavior);
	}
}
