package eu.qimpress.transformations.rpg2sam.sam;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.AbstractQueue;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Set;

import org.dom4j.Element;

import eu.qimpress.samm.staticstructure.ComponentEndpoint;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.CompositeComponent;
import eu.qimpress.samm.staticstructure.Connector;
import eu.qimpress.samm.staticstructure.Interface;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.samm.staticstructure.PassiveResource;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.samm.staticstructure.Repository;
import eu.qimpress.samm.staticstructure.StaticstructureFactory;
import eu.qimpress.samm.staticstructure.SubcomponentEndpoint;
import eu.qimpress.samm.staticstructure.SubcomponentInstance;

public class ArchitectureModel
{
	/**
	 * The model container is accessible as a field.
	 */
	public final Repository model = StaticstructureFactory.eINSTANCE.createRepository ();

	/**
	 * The import context for this model.
	 * 
	 * The import context provides links to other models that will be created from the same input.
	 */
	private Importer importer;
	
	/**
	 * The model factory is invoked through a proxy 
	 * that takes care of adding all model elements
	 * to the model container.
	 */
	class FactoryProxy implements InvocationHandler
	{
		@Override
		public Object invoke (Object self, Method method, Object[] arguments) throws Throwable
		{
			Object element = method.invoke (StaticstructureFactory.eINSTANCE, arguments);
			if (element instanceof Interface) model.getInterface ().add ((Interface) element);
			else if (element instanceof ComponentType) model.getComponenttype ().add ((ComponentType) element);
			return (element);			
		}
	}
	
	private final StaticstructureFactory factory = (StaticstructureFactory) Proxy.newProxyInstance (
		StaticstructureFactory.eINSTANCE.getClass ().getClassLoader (),
		new Class [] { StaticstructureFactory.class },
		new FactoryProxy ());

	//----------------------------------------------------------------------

	private enum ChildHandling { IGNORE_CHILDREN, INCLUDE_CHILDREN };
	
	/**
	 * Guess what. A constructor. 
	 */
	protected ArchitectureModel (Importer importContext)
	{
		importer = importContext;
		model.setName ("Main");

		// Create the module interface type, which is the same everywhere.
		
		moduleOperationType = factory.createOperation ();
		moduleOperationType.setName ("Work");
		
		moduleInterfaceType = factory.createInterface ();
		moduleInterfaceType.getSignatures ().add (moduleOperationType);
		moduleInterfaceType.setName ("WorkInterface");
	}
	
	/**
	 * The singleton model of a module operation.
	 */
	private Operation moduleOperationType;
	
	protected Operation getModuleOperationType ()
	{
		return (moduleOperationType);
	}
	
	/**
	 * The singleton model of a module interface.
	 */
	private Interface moduleInterfaceType;

    protected Interface getModuleInterfaceType ()
    {
    	return (moduleInterfaceType);
    }
	
	/** 
	 * Creates a primitive component type.
	 * 
	 * @param module The module to create the type from.
	 * @param children Whether to include or ignore children.
	 * @return The new component type.
	 */
	private PrimitiveComponent createPrimitiveType (Element module, ChildHandling children) throws Exception
	{
		String name = importer.architecture.getModuleName (module);
		
		PrimitiveComponent component = factory.createPrimitiveComponent ();
		component.setName (name);
		
		// Synchronized components have a single monitor lock.
		if (importer.measurements.getMeasurementSynchronization (name))
		{
			PassiveResource monitorResource = factory.createPassiveResource ();
			monitorResource.setCapacity (1);
			monitorResource.setName ("Monitor");
			component.getPassiveResources ().add (monitorResource);
		}

		// All components have a single provided port.
		InterfacePort providedPort = factory.createInterfacePort ();
		providedPort.setInterfaceType (moduleInterfaceType);
		providedPort.setName ("WorkProvided");
		component.getProvided ().add (providedPort);

		// Components with children have required ports.
		if (children == ChildHandling.INCLUDE_CHILDREN)
		{
			List<Element> childModules = importer.architecture.getChildModules (module);
			int childCount = childModules.size ();
			for (int childIndex = 0 ; childIndex < childCount ; childIndex ++)
			{
				InterfacePort requiredPort = factory.createInterfacePort ();
				requiredPort.setInterfaceType (moduleInterfaceType);
				requiredPort.setName ("WorkRequired" + Integer.toString (childIndex));
				component.getRequired ().add (requiredPort);
			}
		}
		
		importer.callbackPrimitiveComponentTypeCreated (module, component);
		
		return (component);
	}
	
	//----------------------------------------------------------------------

	private class ComponentHolder
	{
		public int depth = 0;
		
		/** Links to parents of this component. Can contain multiple instances of the same parent. Order is not important. */
		public List<ComponentHolder> parents = new LinkedList<ComponentHolder> ();
		/** Links to children of this component. Can contain multiple instances of the same child. Order is important. */
		public List<ComponentHolder> children = new LinkedList<ComponentHolder> ();

		public Element module;
		public ComponentType component;
		public SubcomponentInstance instance;
		
		public void assignDepthRecursively (int current_depth)
		{
			// Assign current depth.
			if (depth < current_depth) depth = current_depth;
			
			// The depth of the children should be at least current depth plus one.
			int child_depth = depth + 1;
			for (ComponentHolder child : children) child.assignDepthRecursively (child_depth);
		}
	}
	
	private class ComponentHolderComparatorByDepth implements Comparator<ComponentHolder>
	{
		/**
		 * Orders two component holders by depth in reverse.
		 * 
		 * Note that the comparator is not consistent with equals.
		 */
		@Override
		public int compare (ComponentHolder first, ComponentHolder second)
		{
			if (first.depth < second.depth) return (+1);
			if (first.depth > second.depth) return (-1);
			return (0);
		}
	}

	/**
	 * Builds both primitive and composite component types.
	 * 
	 * The method works by repeatedly locating the most deeply nested component and adding its
	 * neighbors until a group of components with no neighbors and one parent is created,
	 * this group of components is then turned into a single composite component.
	 * 
	 * The algorithm does not guarantee locating all potential composite components !
	 *  
	 * @param moduleInstances The modules to build from.
	 * @return The composite component consisting of all the other composite components and all the primitive components.
	 */
	public ComponentType createComponentTypes (List<Element> moduleInstances) throws Exception
	{
		// Collect all modules into holders.
		Map<String, ComponentHolder> componentsByName = new HashMap<String, ComponentHolder> ();
		for (Element moduleInstance : moduleInstances)
		{
			String componentName = importer.architecture.getModuleName (moduleInstance);
			ComponentHolder componentHolder = new ComponentHolder ();
			componentHolder.module = moduleInstance;
			componentsByName.put (componentName, componentHolder);
		}
		
		// Establish connection between parent and child modules.
		for (ComponentHolder componentHolder : componentsByName.values ())
		{
			List<Element> componentChildren = importer.architecture.getChildModules (componentHolder.module);
			for (Element childModule : componentChildren)
			{
				String childName = importer.architecture.getModuleName (childModule);
				ComponentHolder childHolder = componentsByName.get (childName);
				
				componentHolder.children.add (childHolder);
				childHolder.parents.add (componentHolder);
			}
		}

		// Measure depth of all modules.
		Element rootModule = importer.architecture.getRootModule ();
		String rootName = importer.architecture.getModuleName (rootModule);
		ComponentHolder rootHolder = componentsByName.get (rootName);
		rootHolder.assignDepthRecursively (0);

		// The rest of the algorithm needs to look up by depth rather than by name.
		AbstractQueue<ComponentHolder> componentsByDepth = new PriorityQueue <ComponentHolder> (componentsByName.size (), new ComponentHolderComparatorByDepth ());
		componentsByDepth.addAll (componentsByName.values ());
		componentsByName = null;

		// To coalesce the components, we repeatedly pick the component with the largest depth together with its
		// parents and keep adding connected children and parents until the entire group has no external
		// children and at most one component with external parents.
	
		while (componentsByDepth.size () > 1)
		{
			Set<ComponentHolder> coalescedHolders = new HashSet<ComponentHolder> ();
			Set<ComponentHolder> incomingHolders = new HashSet<ComponentHolder> ();

			// We start coalescing at the component with the largest depth.
			ComponentHolder deepestHolder = componentsByDepth.peek ();
			final int deepestDepth = deepestHolder.depth;
			incomingHolders.add (deepestHolder);

			do
			{
				// At this moment, we are either in the first iteration of the cycle,
				// or we are repeating the cycle because our coalescing set has more
				// than one component with external parents.
				//
				// In both cases, we need to extend the coalescing set with the external parents.
				
				Set<ComponentHolder> parentHolders = new HashSet<ComponentHolder> ();
				for (ComponentHolder holder : incomingHolders) parentHolders.addAll (holder.parents);
				incomingHolders.addAll (parentHolders);
				parentHolders = null;
				
				// Now we make the set closed with respect to children.
				// Probably not very efficient, but the architectures
				// should be really small, and we do try to trim
				// unnecessary repetition.
				while (!incomingHolders.isEmpty ())
				{
					coalescedHolders.addAll (incomingHolders);
					Set<ComponentHolder> childHolders = new HashSet<ComponentHolder> ();
					for (ComponentHolder holder : incomingHolders) childHolders.addAll (holder.children);
					childHolders.removeAll (coalescedHolders);
					incomingHolders = childHolders;
				}
				
				// Count the components with external parents.
				// Root counts as having an implicit parent.
				// Remember the holders too.
				for (ComponentHolder holder : coalescedHolders)
				{
					Set<ComponentHolder> externalHolders = new HashSet<ComponentHolder> (holder.parents);
					externalHolders.removeAll (coalescedHolders);
					if ((holder.depth == 0) || !externalHolders.isEmpty ())
					{
						incomingHolders.add (holder);
					}
				}
			}
			while (incomingHolders.size () > 1);

			// We have found a set of related components that can be coalesced.
			// Depending on the maximum depth setting, we replace it with
			// either a primitive or a composite component.
			//
			// Either way, we first identify the top of the set and remove the set except for the top.
			
			if (incomingHolders.size () != 1) throw new AssertionError ("Architecture coalescing algorithm failed.");
			ComponentHolder topHolder = incomingHolders.iterator ().next ();
			incomingHolders = null;

			componentsByDepth.removeAll (coalescedHolders);
			componentsByDepth.add (topHolder);

			final int maximumDepth = importer.configuration.maximumDepth;
			if ((maximumDepth > 0) && (deepestDepth > maximumDepth))
			{
				// The set of coalesced components is below maximum depth.
				// It is therefore replaced with a primitive component.

				if (topHolder.component != null) throw new AssertionError ("Architecture coalescing algorithm failed."); 
				topHolder.component = createPrimitiveType (topHolder.module, ChildHandling.IGNORE_CHILDREN);
				
				topHolder.children.clear ();
			}
			else
			{
				// The set of coalesced components is above maximum depth.
				// It is therefore replaced with a composite component.
				
				CompositeComponent composite = factory.createCompositeComponent ();
				composite.setName ("Composite" + importer.architecture.getModuleName (topHolder.module));
	
				// Create instances.
				for (ComponentHolder subcomponentHolder : coalescedHolders)
				{
					if (subcomponentHolder.component == null) subcomponentHolder.component = createPrimitiveType (subcomponentHolder.module, ChildHandling.INCLUDE_CHILDREN);

					subcomponentHolder.instance = factory.createSubcomponentInstance ();
					subcomponentHolder.instance.setName ("Child" + importer.architecture.getModuleName (subcomponentHolder.module));
					subcomponentHolder.instance.setRealizedBy (subcomponentHolder.component);
					
					composite.getSubcomponents ().add (subcomponentHolder.instance);
				}
				
				// Create connectors.
				for (ComponentHolder subcomponentHolder : coalescedHolders)
				{
					int childIndex = 0;
					for (ComponentHolder childHolder : subcomponentHolder.children)
					{
						Connector connector = factory.createConnector ();
						SubcomponentEndpoint startingEndpoint = factory.createSubcomponentEndpoint ();
						startingEndpoint.setSubcomponent (childHolder.instance);
						startingEndpoint.setPort (childHolder.component.getProvided ().get (0));
						connector.getEndpoints ().add (startingEndpoint);
						SubcomponentEndpoint endingEndpoint = factory.createSubcomponentEndpoint ();
						endingEndpoint.setSubcomponent (subcomponentHolder.instance);
						endingEndpoint.setPort (subcomponentHolder.component.getRequired ().get (childIndex));
						connector.getEndpoints ().add (endingEndpoint);
						composite.getConnector ().add (connector);
						
						childIndex ++;
					}
				}
	
				// Create export.
				InterfacePort providedPort = factory.createInterfacePort ();
				providedPort.setInterfaceType (moduleInterfaceType);
				providedPort.setName ("WorkProvided");
				composite.getProvided ().add (providedPort);
			
				Connector connector = factory.createConnector ();
				SubcomponentEndpoint startingEndpoint = factory.createSubcomponentEndpoint ();
				startingEndpoint.setSubcomponent (topHolder.instance);
				startingEndpoint.setPort (topHolder.component.getProvided ().get (0));
				connector.getEndpoints ().add (startingEndpoint);
				ComponentEndpoint endingEndpoint = factory.createComponentEndpoint ();
				endingEndpoint.setPort (providedPort);
				connector.getEndpoints ().add (endingEndpoint);
				composite.getConnector ().add (connector);
	
				// Return the composite back to the architecture.
	
				// Rather than creating a new holder, the primitive type of the composite
				// root is replaced with the composite type. This makes it possible to
				// keep child references of parent components unchanged ...
				
				topHolder.component = composite;
				topHolder.children.clear ();
			}
		}
		
		// At this moment we should only have one component left.
		
		if (componentsByDepth.size () != 1) throw new AssertionError ("Architecture coalescing algorithm failed.");
		
		return (componentsByDepth.peek ().component);
	}
	
	/**
	 * Builds thread pool component type.
	 * 
	 * @arg capacity The capacity of the thread pool.
	 * @return The component representing the thread pool.
	 */
	public ComponentType createThreadPoolType (int capacity) throws Exception
	{
		PrimitiveComponent component = factory.createPrimitiveComponent ();
		component.setName ("ThreadPool");

		// Thread pool has a semaphore resource.
		PassiveResource monitorResource = factory.createPassiveResource ();
		monitorResource.setCapacity (capacity);
		monitorResource.setName ("Semaphore");
		component.getPassiveResources ().add (monitorResource);

		// Thread pool has a single provided port.
		InterfacePort providedPort = factory.createInterfacePort ();
		providedPort.setInterfaceType (moduleInterfaceType);
		providedPort.setName ("WorkProvided");
		component.getProvided ().add (providedPort);

		// Thread pool has a single required port.
		InterfacePort requiredPort = factory.createInterfacePort ();
		requiredPort.setInterfaceType (moduleInterfaceType);
		requiredPort.setName ("WorkRequired");
		component.getRequired ().add (requiredPort);
		
		importer.callbackThreadPoolComponentTypeCreated (component);
		
		return (component);
	}	
}
