package eu.qimpress.transformations.rpg2sam.sam;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Map;

import org.dom4j.Element;

import eu.qimpress.qualityannotationdecorator.seffdecorator.ActivityFailureProbability;
import eu.qimpress.qualityannotationdecorator.seffdecorator.BranchProbability;
import eu.qimpress.qualityannotationdecorator.seffdecorator.CpuResourceDemand;
import eu.qimpress.qualityannotationdecorator.seffdecorator.LoopCount;
import eu.qimpress.qualityannotationdecorator.seffdecorator.PassiveResourceDemand;
import eu.qimpress.qualityannotationdecorator.seffdecorator.SeffdecoratorFactory;
import eu.qimpress.samm.qosannotation.Annotation;
import eu.qimpress.samm.qosannotation.AnnotationType;
import eu.qimpress.samm.qosannotation.ConstantNumber;
import eu.qimpress.samm.qosannotation.ParametricFormula;
import eu.qimpress.samm.qosannotation.QosAnnotations;
import eu.qimpress.samm.qosannotation.QosannotationFactory;
import eu.qimpress.samm.staticstructure.PassiveResource;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.seff.AcquireAction;
import eu.qimpress.seff.InternalAction;
import eu.qimpress.seff.LoopAction;
import eu.qimpress.seff.ProbabilisticBranchTransition;
import eu.qimpress.seff.ReleaseAction;
import eu.qimpress.transformations.rpg2sam.rpg.Architecture;
import eu.qimpress.transformations.rpg2sam.rpg.Measurements;
import eu.qimpress.transformations.rpg2sam.rpg.Architecture.Parameter;
import eu.qimpress.transformations.rpg2sam.rpg.Measurements.Measurement;

public class AnnotationsModel
{
	/**
	 * The model container is accessible as a field.
	 */
	public final QosAnnotations model = QosannotationFactory.eINSTANCE.createQosAnnotations ();

	/**
	 * The import context for this model.
	 * 
	 * The import context provides links to other models that will be created from the same input.
	 */
	private Importer importer;
	
	/**
	 * The model factory is invoked through a proxy 
	 * that takes care of adding all model elements
	 * to the model container.
	 */
	class FactoryProxy implements InvocationHandler
	{
		@Override
		public Object invoke (Object self, Method method, Object[] arguments) throws Throwable
		{
			Object element = method.invoke (SeffdecoratorFactory.eINSTANCE, arguments);
			if (element instanceof Annotation) model.getAnnotation ().add ((Annotation) element);
			return (element);			
		}
	}
	
	private final SeffdecoratorFactory factory = (SeffdecoratorFactory) Proxy.newProxyInstance (
		SeffdecoratorFactory.eINSTANCE.getClass ().getClassLoader (),
		new Class [] { SeffdecoratorFactory.class },
		new FactoryProxy ());

	//----------------------------------------------------------------------
	
	protected AnnotationsModel (Importer importContext)
	{
		importer = importContext;
		model.setName ("Main");
	}


	/**
	 * Converts a double value to an annotation.
	 * 
	 * @param value The double value to convert.
	 * @return The annotation value object.
	 */
	private Object doubleToAnnotation (double value)
	{
		ConstantNumber constant = QosannotationFactory.eINSTANCE.createConstantNumber ();
		constant.setValue (value);
		return (constant);
	}

	
	/**
	 * Converts a measurement to an annotation.
	 * Uses means or distributions depending on global configuration.
	 * 
	 * @param measurement The measurement to convert.
	 * @param scale A scale to apply on values.
	 * @return The annotation value object.
	 */
	private Object measurementToAnnotation (Measurement measurement, double scale)
	{
		switch (importer.configuration.measurementAnnotationSelection)
		{
			case USE_MEANS:
				
				ConstantNumber constant = QosannotationFactory.eINSTANCE.createConstantNumber ();
				double averageOriginal = (double) measurement.sum / (double) measurement.count;
				double averageScaled = averageOriginal * scale;
				constant.setValue (averageScaled);
				return (constant);
				
			case USE_DISTRIBUTIONS:

				// We create a distribution that produces each measurement with equal probability.
				ParametricFormula formula = QosannotationFactory.eINSTANCE.createParametricFormula ();
				String specification = "DoublePMF";
				double probability = (double) 1 / (double) measurement.count;
				specification += "[";
				// Convert measurements to distribution by removing repeated measurements.
				// Standard iteration must not be used here because array can be longer than count.
				Map<Double, Integer> values = new HashMap<Double, Integer> (measurement.count);
				for (int i = 0 ; i < measurement.count ; i ++)
				{
					Double value = measurement.values [i];
					Integer count = values.get (value);
					if (count == null) count = 0;
					values.put (value, count + 1);
				}
				for (Double value : values.keySet ())
				{
					int count = values.get (value);
					double scaled = value * scale;
					specification += "(" + scaled + ";" + count * probability + ")";
				}
				specification += "]";
				formula.setSpecification (specification);
				return (formula);
				
			default:
				
				throw new AssertionError ("An impossible configuration option value.");
		}
	}

	
	/**
	 * Converts a parameter to an annotation.
	 * @param parameter The parameter to convert. 
	 * @return The annotation value object.
	 */
	private Object parameterToAnnotation (Parameter parameter) throws Exception
	{
		if (parameter.type.equals (Architecture.RPG_PARAMETER_TYPE_INTEGER))
		{
			double primitive = Double.parseDouble (parameter.arguments.get (Architecture.RPG_ATTRIBUTE_NAME_VALUE));
			ConstantNumber value = QosannotationFactory.eINSTANCE.createConstantNumber ();
			value.setValue (primitive);
			return (value);
		}
		else if (parameter.type.equals (Architecture.RPG_PARAMETER_TYPE_UNIFORM))
		{
			double primitiveMin = Double.parseDouble (parameter.arguments.get (Architecture.RPG_ATTRIBUTE_NAME_MIN));
			double primitiveMax = Double.parseDouble (parameter.arguments.get (Architecture.RPG_ATTRIBUTE_NAME_MAX));
			// The distribution values are not yet supported by some transformations.
			// We therefore create a parametric value that corresponds to the distribution.
			ParametricFormula value = QosannotationFactory.eINSTANCE.createParametricFormula ();
			value.setSpecification ("UniDouble(" + Double.toString (primitiveMin) + "," + Double.toString (primitiveMax) + ")");
			return (value);
		}
		else if (parameter.type.equals (Architecture.RPG_PARAMETER_TYPE_GEOMETRIC))
		{
			// Since the model does not support geometric distribution, we
			// replace it with exponential distribution of the same average.
			double primitiveAverage = Double.parseDouble (parameter.arguments.get (Architecture.RPG_ATTRIBUTE_NAME_VALUE));
			double primitiveRate = (double) 1 / primitiveAverage;
			// The distribution values are not yet supported by some transformations.
			// We therefore create a parametric value that corresponds to the distribution.
			ParametricFormula value = QosannotationFactory.eINSTANCE.createParametricFormula ();
			value.setSpecification ("Exp(" + Double.toString (primitiveRate) + ")");
			return (value);
		}
		else throw new Exception ("Unknown parameter class.");
	}

	
	/**
	 * Wraps an annotation value in an annotation object.
	 * Just useful because different values are assigned using different functions which sucks. 
	 */
	protected void setAnnotationValue (Annotation annotation, Object value) throws Exception
	{
		// Each unique value type has a unique setter. 
		if (value instanceof ConstantNumber)
		{
			annotation.setConstantNumber ((ConstantNumber) value);
		}
		else if (value instanceof ParametricFormula)
		{
			annotation.setParametricFormula ((ParametricFormula) value);
		}
		else throw new Exception ("Setting of an annotation type not supported.");
	}

	
	/**
	 * Rounds the given annotation value.
	 */
	protected Object roundAnnotationValue (Object value) throws Exception
	{
		if (value instanceof ConstantNumber)
		{
			ConstantNumber valueConstantNumber = (ConstantNumber) value;
			valueConstantNumber.setValue (Math.round (valueConstantNumber.getValue ()));
			return (valueConstantNumber);
		}
		else if (value instanceof ParametricFormula)
		{
			ParametricFormula valueParametricFormula = (ParametricFormula) value;
			valueParametricFormula.setSpecification ("Round(" + valueParametricFormula.getSpecification () + ")");
			return (valueParametricFormula);
		}
		else throw new Exception ("Rounding of an annotation type not supported.");
	}
	
	//----------------------------------------------------------------------
	
	protected void annotateActionTime (Element context, InternalAction action) throws Exception
	{
		String name = importer.architecture.getModuleName (context);

		// The timing is read from the measurements. The units need to be
		// converted from the measurement frequency to the processor frequency.
		Measurement measurement = importer.measurements.getMeasurementTiming (name);
		double scale = HardwareModel.SAM_HARDWARE_PROCESSOR_FREQUENCY / Measurements.RPG_MEASUREMENT_FREQUENCY;
		Object annotation = measurementToAnnotation (measurement, scale);

		CpuResourceDemand demand = factory.createCpuResourceDemand ();
		setAnnotationValue (demand, annotation);
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		demand.setAnnotationType (AnnotationType.REQUIREMENT);
		demand.setInternalAction (action);
		//Hauck: QoS annotation resource demand does not have reference to execution resource anymore
		//demand.setExecutionResource (importer.callbackGetExecutionResourceSingleton ());
		demand.setName ("Demand" + name);
	}

	protected void annotateActionReliability (Element context, InternalAction action) throws Exception
	{
		String name = importer.architecture.getModuleName (context);

		// The failure probability is read from the architecture.
		double failureProbability = importer.architecture.getParameterFailureProbability (context);
		Object failureAnnotation = doubleToAnnotation (failureProbability);

		ActivityFailureProbability failure = factory.createActivityFailureProbability ();
		setAnnotationValue (failure, failureAnnotation);
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		failure.setAnnotationType (AnnotationType.REQUIREMENT);
		failure.setInternalAction (action);
		failure.setName ("Failure" + name);
	}
	
	protected void annotateActionMonitorResource (Element context, PrimitiveComponent component, AcquireAction acquire, ReleaseAction release)
	{
		String name = importer.architecture.getModuleName (context);
		PassiveResource monitor = component.getPassiveResources ().get (0);
		
		PassiveResourceDemand acquireDemand = factory.createPassiveResourceDemand ();
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		acquireDemand.setAnnotationType (AnnotationType.REQUIREMENT);
		acquireDemand.setPassiveResource (monitor);
		acquireDemand.setPassiveAction (acquire);
		acquireDemand.setName ("Acquire" + name);

		PassiveResourceDemand releaseDemand = factory.createPassiveResourceDemand ();
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		releaseDemand.setAnnotationType (AnnotationType.REQUIREMENT);
		releaseDemand.setPassiveResource (monitor);
		releaseDemand.setPassiveAction (release);
		releaseDemand.setName ("Release" + name);
	}
	
	protected void annotateActionThreadPoolResource (PrimitiveComponent component, AcquireAction acquire, ReleaseAction release)
	{
		PassiveResource monitor = component.getPassiveResources ().get (0);
		
		PassiveResourceDemand acquireDemand = factory.createPassiveResourceDemand ();
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		acquireDemand.setAnnotationType (AnnotationType.REQUIREMENT);
		acquireDemand.setPassiveResource (monitor);
		acquireDemand.setPassiveAction (acquire);
		acquireDemand.setName ("Acquire");

		PassiveResourceDemand releaseDemand = factory.createPassiveResourceDemand ();
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		releaseDemand.setAnnotationType (AnnotationType.REQUIREMENT);
		releaseDemand.setPassiveResource (monitor);
		releaseDemand.setPassiveAction (release);
		releaseDemand.setName ("Release");
	}
	
	protected void annotateLoopCount (Element context, LoopAction action) throws Exception
	{
		String name = importer.architecture.getModuleName (context);

		// The loop count is read from the architecture.
		Parameter valueParameter = importer.architecture.getParameterLoopCount (context);
		Object valueAnnotationReal = parameterToAnnotation (valueParameter);
		Object valueAnnotationInteger = roundAnnotationValue (valueAnnotationReal);
		
		LoopCount count = factory.createLoopCount ();
		setAnnotationValue (count, valueAnnotationInteger);
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		count.setAnnotationType (AnnotationType.REQUIREMENT);
		count.setLoopAction (action);
		count.setName ("Count" + name);
	}

	protected void annotateBranchTransitionProbability (Element context, ProbabilisticBranchTransition branchTransition, int branchCount) throws Exception
	{
		String name = importer.architecture.getModuleName (context);

		// The probability is uniform given by number of branches.
		double branchProbability = (double) 1 / (double) branchCount;
		Object branchAnnotation = doubleToAnnotation (branchProbability);

		// The annotation is associated with the transition.
		BranchProbability branch = factory.createBranchProbability ();
		setAnnotationValue (branch, branchAnnotation);
		// TODO The annotation type is temporarily set to REQUIREMENT rather than MEASURED.
		branch.setAnnotationType (AnnotationType.REQUIREMENT);
		branch.setProbabilisticBranchTransition (branchTransition);
		branch.setName ("Branch" + name);
	}
}
