package eu.qimpress.transformations.rpg2sam.rpg;

import java.io.BufferedReader;
import java.io.FileReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;

import eu.qimpress.transformations.rpg2sam.ide.Configuration;

/**
 * A measurement source.
 */
public class Measurements
{
	/**
	 * The configuration context for this source.
	 */
	private final Configuration configuration;
	
	private String timingContext;
	private String timingAttribute;

	private Random random;
	private Map<Location, Measurement> values;

	// Some constants need to be public to allow the model classes to understand the module measurements.
	
	public static final int RPG_MEASUREMENT_FREQUENCY = 1000000000;
	
	public static final String RPG_MEASUREMENT_ATTRIBUTE_CORES = "cores-count";
	public static final String RPG_MEASUREMENT_ATTRIBUTE_CLIENTS = "client-count";
	public static final String RPG_MEASUREMENT_ATTRIBUTE_THREADS = "thread-count";
	public static final String RPG_MEASUREMENT_ATTRIBUTE_THINKTIME = "client-wait-time-usec-ex";
	
	// Other constants are private to enforce the model classes using the appropriate interfaces.

	// The number of measurements generated with random source.
	private static final int RPG_MEASUREMENT_VALUES = 100;
	// The number of measurements read before array resizing. 
	private static final int RPG_MEASUREMENT_INCREMENTS = 10;
	
	private static final String RPG_MEASUREMENT_MODULE_GLOBAL = "app";
	
	private static final String RPG_MEASUREMENT_CONTEXT_CONFIG = "config";
	private static final String RPG_MEASUREMENT_CONTEXT_SHARED = "shared";
	private static final String RPG_MEASUREMENT_CONTEXT_ISOLATED = "isolated";
	
	private static final String RPG_MEASUREMENT_ATTRIBUTE_REALTIME = "monotonic";
	private static final String RPG_MEASUREMENT_ATTRIBUTE_THREADTIME = "threadtime";
	private static final String RPG_MEASUREMENT_ATTRIBUTE_SYNCHRONIZED = "synchronized";

	private static final String RPG_MEASUREMENT_ATTRIBUTE_RESULT_OK = "result-ok-count";
	private static final String RPG_MEASUREMENT_ATTRIBUTE_RESULT_ERROR_LATENT = "result-error-latent-count";
	private static final String RPG_MEASUREMENT_ATTRIBUTE_RESULT_ERROR_EFFECTIVE = "result-error-effective-count";
	
	/**
	 * A class that holds measurement values for single location.
	 */
	public static class Measurement
	{
		public int count = 0;
		public double sum = 0;
		public double [] values = new double [RPG_MEASUREMENT_INCREMENTS];
		
		public void add (double value)
		{
			// Reallocate values array if necessary, probably could be more efficient with other than constant increments.
			if (values.length == count) values = Arrays.copyOf (values, values.length + RPG_MEASUREMENT_INCREMENTS);
			
			values [count] = value;
			if (count == Integer.MAX_VALUE) throw new AssertionError ("Too many measurements.");
			
			count += 1;
			sum += value;
		}
	}

	/**
	 * A class that holds measurement location.
	 */
	private static class Location
	{
		public String module;
		public String context;
		public String attribute;

		@Override
		public int hashCode ()
		{
			return (module.hashCode () +
					context.hashCode () +
					attribute.hashCode ());
		}
		
		@Override
		public boolean equals (Object objectRaw)
		{
			if (objectRaw instanceof Location)
			{
				Location objectLocation = (Location) objectRaw;
				return (module.equals (objectLocation.module) &&
						context.equals (objectLocation.context) &&
						attribute.equals (objectLocation.attribute));
			}
			return (false);
		}
		
		public Location (String module, String context, String attribute)
		{
			this.module = module;
			this.context = context;
			this.attribute = attribute;
		}
	}
	
	/**
	 * The constructor imports a measurement.
	 *
	 * @param importContext The import context to use.
	 * @param file Name of the measurement file to import.
	 */
	public Measurements (String file, Configuration configurationContext) throws Exception  
	{
		configuration = configurationContext;
		
		// The configuration determines the default measurement context and attribute.
		switch (configuration.timingSourceSelection)
		{
			case USE_ISOLATED:
				timingContext = RPG_MEASUREMENT_CONTEXT_ISOLATED;
				timingAttribute = RPG_MEASUREMENT_ATTRIBUTE_REALTIME;
				break;
			case USE_SHARED:
				timingContext = RPG_MEASUREMENT_CONTEXT_SHARED;
				timingAttribute = RPG_MEASUREMENT_ATTRIBUTE_THREADTIME;
				break;
			default:
				throw new AssertionError ("Unexpected measurement source selection value.");
		}

		// The configuration determines the value source.
		switch (configuration.valueSourceSelection)
		{
			case USE_RANDOM:
		
				// All reported values will be random. Create the random generator.
				random = new Random ();
				
				break;

			case USE_MEASURED:
				
				// All reported values will be measured. Import the measurement file.
				values = new HashMap<Location, Measurement> ();			
			
				BufferedReader reader = new BufferedReader (new FileReader (file));
			
				while (true)
				{
					String line = reader.readLine ();
					if (line == null) break;
					
					String [] fields = line.split (";");
					if (fields.length != 4) throw new Exception ("Invalid measurement format.");
					String moduleName = fields [0];
					String contextName = fields [1];
					String attributeName = fields [2];
					double value = Double.parseDouble (fields [3]);
		
					Location location = new Location (moduleName, contextName, attributeName);
					if (!values.containsKey (location)) values.put (location, new Measurement ());
					Measurement measurement = values.get (location);

					measurement.add (value);
				}
				
				reader.close ();
				
				break;
				
			default:

				throw new AssertionError ("Unexpected timing source selection value.");
		}
	}
	
	/**
	 * Locates a measurement.
	 *
	 * Takes care of generating random values if configured appropriately.
	 */
	private Measurement getMeasurement (String moduleName, String contextName, String attributeName) throws Exception
	{
		Measurement measurement;
		
		if (values != null)
		{
			// Measured values are reported if imported.
			Location location = new Location (moduleName, contextName, attributeName);
			measurement = values.get (location);
		}
		else
		{
			// Random values are reported otherwise.
			measurement = new Measurement ();
			for (int i = 0 ; i < RPG_MEASUREMENT_VALUES ; i ++) measurement.add (Math.abs (random.nextDouble ()));
		}
		
		if (measurement == null) throw new Exception ("Missing measurement: " + moduleName + ";" + contextName + ";" + attributeName + ".");
		
		return (measurement);
	}
	
	/**
	 * Returns the timing as measured in configured context.
	 */
	public Measurement getMeasurementTiming (String moduleName) throws Exception
	{
		Measurement measurement = getMeasurement (moduleName, timingContext, timingAttribute);
		return (measurement);
	}
	
	/**
	 * Returns the failure probability as measured in shared context.
	 * Measurements in isolated context are not done for all components.
	 * Effective errors are counted as failures, latent errors are counted as successes. 
	 */
	public double getMeasurementFailureProbability (String moduleName) throws Exception
	{
		Measurement resultOk = getMeasurement (moduleName, RPG_MEASUREMENT_CONTEXT_SHARED, RPG_MEASUREMENT_ATTRIBUTE_RESULT_OK);
		Measurement resultErrorLantent = getMeasurement (moduleName, RPG_MEASUREMENT_CONTEXT_SHARED, RPG_MEASUREMENT_ATTRIBUTE_RESULT_ERROR_LATENT);
		Measurement resultErrorEffective = getMeasurement (moduleName, RPG_MEASUREMENT_CONTEXT_SHARED, RPG_MEASUREMENT_ATTRIBUTE_RESULT_ERROR_EFFECTIVE);
		double resultCountNegative = resultErrorEffective.sum;
		double resultCountPositive = resultOk.sum + resultErrorLantent.sum;
		double resultCountTotal = resultCountNegative + resultCountPositive;
		double reliability = resultCountNegative / resultCountTotal;
		return (reliability);
	}

	/**
	 * Returns the synchronization configuration as measured in configured context.
	 */
	public boolean getMeasurementSynchronization (String moduleName) throws Exception
	{
		Measurement measurement = getMeasurement (moduleName, RPG_MEASUREMENT_CONTEXT_CONFIG, RPG_MEASUREMENT_ATTRIBUTE_SYNCHRONIZED);
		return (measurement.sum != 0);
	}
	
	/**
	 * Returns a global measurement value.
	 */
	public double getGlobalAverage (String attributeName) throws Exception
	{
		Measurement measurement = getMeasurement (RPG_MEASUREMENT_MODULE_GLOBAL, RPG_MEASUREMENT_CONTEXT_CONFIG, attributeName);
		double average = (double) measurement.sum / (double) measurement.count;
		return (average);
	}
}
