package eu.qimpress.transformations.rpg2sam.rpg;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import eu.qimpress.transformations.rpg2sam.ide.Configuration;

/**
 * An architecture source.
 */
public class Architecture
{
	/**
	 * The configuration context for this source.
	 */
	private final Configuration configuration;
	
	private Document rootDocument;
	private Element rootElement;

	// Some constants need to be public to allow the model classes to understand the module elements.
	
	public static final String RPG_MODULE_CLASS_SEQUENCE = "sequential";
	public static final String RPG_MODULE_CLASS_BRANCH = "branch";
	public static final String RPG_MODULE_CLASS_LOOP = "loop";
	
	public static final String RPG_PARAMETER_TYPE_COIN_TOSS= "random-coin-toss";
	public static final String RPG_PARAMETER_TYPE_FLOAT = "float";
	public static final String RPG_PARAMETER_TYPE_BOOLEAN = "boolean";
	public static final String RPG_PARAMETER_TYPE_INTEGER = "integer";
	public static final String RPG_PARAMETER_TYPE_UNIFORM = "random-range";
	public static final String RPG_PARAMETER_TYPE_GEOMETRIC = "random-geometric";

	public static final String RPG_ATTRIBUTE_NAME_MIN = "min";
	public static final String RPG_ATTRIBUTE_NAME_MAX = "max";
	public static final String RPG_ATTRIBUTE_NAME_VALUE = "value";
	
	// Other constants are private to enforce the model classes using the appropriate interfaces.
	
	private static final String RPG_NODE_ITEM = "item";
	private static final String RPG_NODE_SLOT = "slot";
	private static final String RPG_NODE_MODULE = "module";
	private static final String RPG_NODE_ATTRIBUTE = "attribute";
	private static final String RPG_NODE_PARAMETER = "param";
	private static final String RPG_NODE_PARAMETERS = "params";

	private static final String RPG_SLOT_ATTRIBUTE_TARGET = "target-id";
	
	private static final String RPG_MODULE_ATTRIBUTE_NAME = "id"; 
	private static final String RPG_MODULE_ATTRIBUTE_CLASS = "classname";
	
	private static final String RPG_ATTRIBUTE_ATTRIBUTE_NAME = "name";

	private static final String RPG_ATTRIBUTE_NAME_PROBABILITY = "probability";
	
	private static final String RPG_PARAMETER_ATTRIBUTE_NAME = "name";
	private static final String RPG_PARAMETER_ATTRIBUTE_CLASS = "classname";

	private static final String RPG_PARAMETER_NAME_LOOP_COUNT = "loopcount";
	private static final String RPG_PARAMETER_NAME_FAULT_PROBABILITY = "fault-probability";
	private static final String RPG_PARAMETER_NAME_ERROR_INTERNAL_PROBABILITY = "error-internal-probability";
	private static final String RPG_PARAMETER_NAME_ERROR_INVOCATION_PROBABILITY = "error-invocation-probability";
	private static final String RPG_PARAMETER_NAME_ERROR_TERMINATION_PROBABILITY = "error-termination-probability";
	private static final String RPG_PARAMETER_NAME_FAILURE_INTERNAL_PROBABILITY = "failure-internal-probability";
	private static final String RPG_PARAMETER_NAME_FAILURE_INVOCATION_PROBABILITY = "failure-invocation-probability";
	private static final String RPG_PARAMETER_NAME_FAILURE_TERMINATION_PROBABILITY = "failure-termination-probability";

	
	private static final String RPG_ARCHITECTURE_ATTRIBUTE_ROOT = "root-module-id";
	
	/**
	 * A class that holds an architectural parameter value.
	 */
	public static class Parameter
	{
		public String type;
		public Map<String, String> arguments;
	}
	
	/**
	 * The constructor imports an architecture.
	 *
	 * @param file Name of the architecture file to import.
	 * @param configuration The transformation configuration to use.
	 */
	public Architecture (String file, Configuration configurationContext) throws Exception  
	{
		configuration = configurationContext;

		// Start by parsing the input file.

		File architectureFile = new File (file);
		SAXReader architectureReader = new SAXReader ();
    	rootDocument = architectureReader.read (architectureFile);

		// We just get the top level architecture element.
		
		rootElement = rootDocument.getRootElement ();
		if (rootElement.getName () != "architecture") throw new Exception ("Top level architecture node missing.");
	}

	public Element getModuleByName (String name)
	{
		String moduleQuery = RPG_NODE_MODULE + "[@" + RPG_MODULE_ATTRIBUTE_NAME + "='" + name + "']";
		return ((Element) rootElement.selectSingleNode (moduleQuery));
	}

	public Element getRootModule ()
	{
		String rootName = rootElement.attributeValue (RPG_ARCHITECTURE_ATTRIBUTE_ROOT);
		Element rootModule = getModuleByName (rootName);
		return (rootModule);
	}

	public List<Element> getAllModules ()
	{
		return (rootElement.elements (RPG_NODE_MODULE));
	}

	public List<Element> getChildModules (Element context)
	{
		List<Element> slots = context.elements (RPG_NODE_SLOT);
		List<Element> children = new ArrayList<Element> (slots.size ());
		for (Element slot : slots)
		{
			String child = slot.attributeValue (RPG_SLOT_ATTRIBUTE_TARGET);
			children.add (getModuleByName (child));
		}
		return (children);
	}
	
	public String getModuleName (Element module)
	{
		return (module.attribute (Architecture.RPG_MODULE_ATTRIBUTE_NAME).getValue ());
	}

	public String getModuleType (Element module)
	{
		return (module.attribute (Architecture.RPG_MODULE_ATTRIBUTE_CLASS).getValue ());
	}
	
	/**
	 * Retrieves module parameters. 
	 * 
	 * Module parameters are specified in the architecture description file.
	 * Each parameter has a type, which corresponds to the datasource provider class,
	 * and a value, which is a list of arguments passed to the datasource provider class.
	 */
	private Parameter getParameterValue (Element context, String name) throws Exception
	{
		Parameter parameter = new Parameter ();
		
		// Locate the desired parameter element.
		String parameterQuery =
			RPG_NODE_PARAMETERS + "/" + RPG_NODE_PARAMETER +
			"[@" + RPG_PARAMETER_ATTRIBUTE_NAME + "='" + name + "']";
		Element parameterElement = (Element) context.selectSingleNode (parameterQuery);
		parameter.type = parameterElement.attributeValue (RPG_PARAMETER_ATTRIBUTE_CLASS);

		// Gather parameter attributes. 
		parameter.arguments = new HashMap<String, String> ();
		List<Element> attributesList = parameterElement.elements (RPG_NODE_ATTRIBUTE);
		for (Element attribute : attributesList)
		{
			String attributeName = attribute.attributeValue (RPG_ATTRIBUTE_ATTRIBUTE_NAME);
			String attributeValue = attribute.elementText (RPG_NODE_ITEM);
			parameter.arguments.put (attributeName, attributeValue);
		}
		
		return (parameter);
	}

	/**
	 * Retrieves module parameter as double.
	 * 
	 * The conversion is performed for those parameter types that can
	 * be meaningfully converted to double, an exception is thrown for
	 * other parameter types.
	 */
	private double getParameterValueAsDouble (Element context, String name) throws Exception
	{
		Parameter parameter = getParameterValue (context, name);
		final String type = parameter.type; 
		
		double value;
		
		if (type.equals (Architecture.RPG_PARAMETER_TYPE_FLOAT))
		{
			value = Double.parseDouble (parameter.arguments.get (RPG_ATTRIBUTE_NAME_VALUE));
		}
		else if (type.equals (Architecture.RPG_PARAMETER_TYPE_BOOLEAN))
		{
			value = Double.parseDouble (parameter.arguments.get (RPG_ATTRIBUTE_NAME_VALUE));
		}
		else if (type.equals (Architecture.RPG_PARAMETER_TYPE_INTEGER))
		{
			value = Double.parseDouble (parameter.arguments.get (RPG_ATTRIBUTE_NAME_VALUE));
		}
		else if (type.equals (Architecture.RPG_PARAMETER_TYPE_COIN_TOSS))
		{
			value = Double.parseDouble (parameter.arguments.get (RPG_ATTRIBUTE_NAME_PROBABILITY));
		}
		else throw new Exception ("Unexpected parameter type: " + type + ".");
		
		return (value);
	}
	
	public Parameter getParameterLoopCount (Element context) throws Exception
	{
		return (getParameterValue (context, RPG_PARAMETER_NAME_LOOP_COUNT));
	}
	
	public double getParameterFailureProbability (Element context) throws Exception
	{
		double faultProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_FAULT_PROBABILITY);
		double errorInternalProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_ERROR_INTERNAL_PROBABILITY);
		double errorInvocationProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_ERROR_INVOCATION_PROBABILITY);
		double errorTerminationProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_ERROR_TERMINATION_PROBABILITY);
		double failureInternalProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_FAILURE_INTERNAL_PROBABILITY);
		double failureInvocationProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_FAILURE_INVOCATION_PROBABILITY);
		double failureTerminationProbability = getParameterValueAsDouble (context, RPG_PARAMETER_NAME_FAILURE_TERMINATION_PROBABILITY);
		
		// The failure probability of an individual component is approximated.
		// Since the chance of encountering an externally generated  error is not
		// known, only locally generated errors are considered in the approximation.
		
		// Failure occurs if invocation event is generated and promoted in a single session. 
		double invocationFailureProbability =
			errorInvocationProbability *
			(1.0 - 
				(1.0 - failureInvocationProbability) *
				(1.0 - failureTerminationProbability));
		
		// Failure occurs if termination event is generated and promoted in a single session.
		double terminationFalureProbability =
			errorTerminationProbability *
			failureTerminationProbability;

		// Failure occurs if internal event is generated and promoted.
		// The formula is derived from steady state Markov Chain model.
		double internalFailureProbability = (errorInternalProbability * failureInternalProbability) /
		                                    (errorInternalProbability + failureInternalProbability); 
		
		// All failures are keyed on the component containing a fault.  
		double failureProbability = faultProbability *
			(1.0 -
				(1.0 - internalFailureProbability) *
				(1.0 - invocationFailureProbability) *
				(1.0 - terminationFalureProbability));
		
		return (failureProbability);
	}
}
