package eu.qimpress.transformations.common.jobs;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;

import de.uka.ipd.sdq.workflow.IBlackboardInteractingJob;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.MDSDBlackboard;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.ResourceSetPartition;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;

public class LoadSAMMAlternativeJob implements IJob,
		IBlackboardInteractingJob<MDSDBlackboard> {

	private static final Logger logger = Logger.getLogger(LoadSAMMIntoBlackboardJob.class);
	private MDSDBlackboard blackboard;
	private IQAlternative alternative;
	public static final String[] SAMM_EXTENSIONS = new String[]{
		"samm_repository",
		"samm_servicearchitecturemodel",
		"samm_qosannotation",
		"samm_seff",
		"samm_hardware",
		"samm_targetenvironment",
		"samm_usagemodel"
	};

	public LoadSAMMAlternativeJob(IQAlternative alternative) {
		super();
		this.alternative = alternative;
	}

	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException,
			UserCanceledException {
		ResourceSetPartition sammPartition = this.blackboard.getPartition(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID);
		String alternativeID = "";
		if (alternative.getInfo()!= null) {
			alternativeID = alternative.getInfo().getId();	
		}
		logger.info("Loading SAM Model Files for alternative " + alternativeID);
		IQModel[] elements = alternative.getModels();
		IQModel[] inputElements = new IQModel[SAMM_EXTENSIONS.length];
		int j = 0;
		for (IQModel element: elements) {
			if (isNecessaryModelForTransformation(element)) {
				inputElements[j] = element; 
				j++;
			}
		}
		for (String extension : LoadSAMMAlternativeJob.SAMM_EXTENSIONS) {
			URI location = getResourceURI(inputElements, extension);
			logger.debug("Loading SAM model file "+location);
			sammPartition.loadModel(location);
		}
		sammPartition.resolveAllProxies();
	}
	
	public static boolean isNecessaryModelForTransformation(IQModel element) {
		for (String s : LoadSAMMAlternativeJob.SAMM_EXTENSIONS) {
			if (s.equals(element.getType())) {
				return true;
			}
		}
		return false;
	}

	@Override
	public String getName() {
		return "Loading model files of SAM Alternative";
	}

	@Override
	public void rollback(IProgressMonitor monitor)
			throws RollbackFailedException {

	}

	@Override
	public void setBlackboard(MDSDBlackboard blackboard) {
		this.blackboard = blackboard;
	}
	
	public static URI getResourceURI(IQModel[] elements, String fileextension) throws JobFailedException {
		for (IQModel elem : elements) {
			URI location = getURIForQIElement(elem);
			if (location != null && location.fileExtension().equals(fileextension)) {
				return location;
			}
		}
		logger.error("Unable to find file with extension "+fileextension+". " +
				"Check SAM alternative for completeness.");
		throw new JobFailedException("Unable to find file with extension "+fileextension+". " +
				"Check SAM alternative for completeness.");
	}

	/**
	 * @param elem The element to create an URI for.
	 * @return null if no URI could be created
	 */
	public static URI getURIForQIElement(IQElement elem) {
		URI location = null;
		if(elem != null) {

			// FIXED by Viliam Simko on 2010-09-15
			// related to the Q-Impress bug http://jira.ow2.org/browse/QIMPRESS-256
			// the original code produced absolute paths like file:/home/something
			// instead of relative paths like platform:relative/path/to/resource
			// OLD CODE:
			// String pathName = elem.getCorrespondingResource().getLocationURI().toString();
			// location = URI.createURI(pathName);

			String pathName = elem.getCorrespondingResource().getFullPath().toString();
			location = URI.createPlatformResourceURI(pathName, false);
			
		} else {
			logger.warn("Could not load a model. The location was empty or incorrect.");
		}
		
		return location;
	}
	
	/*public static URI getURIForQIElement(IQElement elem) {
		URI location = null;
		if(elem != null) {
			location = URI.createURI(elem.getCorrespondingResource().getLocationURI().toString());			
		} else {
			logger.warn("Could not load a model. The location was empty or incorrect.");
		}
		return location;

	}*/
}
