package eu.qimpress.ide.analysis.reliability.launch;

import java.util.ArrayList;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;

import de.uka.ipd.sdq.workflow.BlackboardBasedWorkflow;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.OrderPreservingBlackboardCompositeJob;
import de.uka.ipd.sdq.workflow.launchconfig.AbstractWorkflowBasedLaunchConfigurationDelegate;
import de.uka.ipd.sdq.workflow.launchconfig.LoggerAppenderStruct;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.MDSDBlackboard;
import de.uka.ipd.sdq.workflow.ui.UIBasedWorkflow;
import eu.qimpress.ide.analysis.reliability.Activator;
import eu.qimpress.ide.analysis.reliability.jobs.FillResultsModelJob;
import eu.qimpress.ide.analysis.reliability.jobs.SysCallAnalysisJob;
import eu.qimpress.ide.analysis.reliability.jobs.SysCallAnalysisJobConf;
import eu.qimpress.samm.usagemodel.UsageScenario;
import eu.qimpress.transformations.common.jobs.LoadResultsModelJob;
import eu.qimpress.transformations.common.jobs.LoadSAMMIntoBlackboardJob;
import eu.qimpress.transformations.common.jobs.PrepareResultsBlackboardPartionJob;
import eu.qimpress.transformations.common.jobs.SAMMValidationJob;

/**
 * Creator of reliability analysis workflow.
 * @author Mauro Luigi Drago, Andrea Ciancone
 *
 */
public class ReliabilityLaunchConfigurationDelegate extends
	AbstractWorkflowBasedLaunchConfigurationDelegate<
		ReliabilityLaunchConfiguration, 
		BlackboardBasedWorkflow<MDSDBlackboard>> {

	/** Logging properties */
	private static final String LOG_PATTERN = "%-5p: %m\n";
	private Logger logger = Logger.getLogger(AbstractWorkflowBasedLaunchConfigurationDelegate.class);
	public static final String OAW_SAMM_CHECK_FILE = "reliability";

	@Override
	protected IJob createWorkflowJob(ReliabilityLaunchConfiguration config, ILaunch launch) throws CoreException {
		// create the root job
		OrderPreservingBlackboardCompositeJob<MDSDBlackboard> rootJob = 
			new OrderPreservingBlackboardCompositeJob<MDSDBlackboard>();
		rootJob.setName("Reliability Analysis Main Task");
		
		rootJob.add(new LoadSAMMIntoBlackboardJob(config.getAlternative()));
		rootJob.add(new PrepareResultsBlackboardPartionJob());
		rootJob.add(new LoadResultsModelJob(config.getAlternative()));

		rootJob.add(new SAMMValidationJob(config, SAMMValidationJob.OAW_CORE_SAMM_CHECK_FILE, OAW_SAMM_CHECK_FILE));

		// Add a job for each usage scenario
		for(UsageScenario us: config.getUsageModel().getUsageScenarios()) {
			logger.debug("Creating Analysis Job for usage scenario ["+us.getId()+"]");
			SysCallAnalysisJob sysCallJob = new SysCallAnalysisJob(deriveSysCallAnalysisJobConf(config, us.getId()));
			rootJob.add(sysCallJob);
		
			logger.debug("Creating job to fill results model with analysis results...");
			FillResultsModelJob frmJob = new FillResultsModelJob(deriveSysCallAnalysisJobConf(config, us.getId()), sysCallJob);
			rootJob.add(frmJob);
		}
				
		return rootJob;
	}

	public static SysCallAnalysisJobConf deriveSysCallAnalysisJobConf(ReliabilityLaunchConfiguration config, String usageScenarioId) {
		SysCallAnalysisJobConf conf = new SysCallAnalysisJobConf();
		conf.setDefaults();
		conf.setReliabilityConf(config);
		conf.setUsageScenarioId(usageScenarioId);

		return conf;
	}
	
	@Override
	protected UIBasedWorkflow<MDSDBlackboard> createWorkflow( ReliabilityLaunchConfiguration workflowConfiguration, 
			IProgressMonitor monitor, ILaunch launch) throws CoreException {
		return new UIBasedWorkflow<MDSDBlackboard>(
				createWorkflowJob(workflowConfiguration, launch), 
				monitor,
				createExcpetionHandler(workflowConfiguration.isInteractive()),
				new MDSDBlackboard());
	}

	@Override
	protected ReliabilityLaunchConfiguration deriveConfiguration(ILaunchConfiguration configuration, String mode) throws CoreException {
		ReliabilityLaunchConfiguration config = new ReliabilityLaunchConfiguration();
		config.initializeFrom(configuration);
		
		if (!config.isValid()) throwCoreException(config.getErrorMessage());
		return config; 
	}
	
	@Override
	protected ArrayList<LoggerAppenderStruct> setupLogging(Level logLevel)
			throws CoreException {
		
		ArrayList<LoggerAppenderStruct> loggerList = new ArrayList<LoggerAppenderStruct>();

		loggerList.add(setupLogger("de.uka.ipd.sdq.workflow", logLevel,	LOG_PATTERN));
		loggerList.add(setupLogger("eu.qimpress.ide.analysis.reliability.jobs", logLevel, LOG_PATTERN));
		loggerList.add(setupLogger("eu.qimpress.ide.analysis.reliability.jobs.acceleo", logLevel, LOG_PATTERN));
		loggerList.add(setupLogger("eu.qimpress.ide.analysis.reliability.jobs.prism", logLevel, LOG_PATTERN));

		return loggerList;
	}
	
	/**
	 * Useful facility to throw standard <code>CoreExceptions</code>.
	 * @param message the message to embed in the exception.
	 * @throws CoreException the <code>CoreException</code>.
	 */
	private void throwCoreException(String message) throws CoreException {
		throw new CoreException(
		          new Status(
		              IStatus.ERROR,
		              Activator.PLUGIN_ID,
		              IStatus.OK,
		              message,
		              null));
	}
}
