package eu.qimpress.ide.analysis.reliability.launch;

import java.util.HashMap;

import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.emf.common.util.URI;

import de.uka.ipd.sdq.workflow.launchconfig.AbstractWorkflowBasedRunConfiguration;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.ModelLocation;
import eu.qimpress.ide.analysis.reliability.jobs.ReliabilityAnalysis;
import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.ide.backbone.core.ui.tabs.QImpressAlternativeEvaluationSelectionTab;
import eu.qimpress.ide.backbone.core.ui.tabs.QImpressAlternativeSelectionTab;
import eu.qimpress.ide.backbone.core.ui.tabs.QImpressUsageModelSelectionTab;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.samm.staticstructure.ServiceArchitectureModel;
import eu.qimpress.samm.usagemodel.UsageModel;
import eu.qimpress.samm.usagemodel.UsageRepository;
import eu.qimpress.samm.usagemodel.UsageScenario;
import eu.qimpress.transformations.common.jobs.LoadSAMMAlternativeJob;
import eu.qimpress.transformations.common.jobs.LoadSAMMIntoBlackboardJob;
import eu.qimpress.transformations.common.jobs.PrepareResultsBlackboardPartionJob;

/**
 * The implementation class for the configuration of reliability analysis.
 * @author Mauro Luigi Drago, Andrea Ciancone
 *
 */
public class ReliabilityLaunchConfiguration extends AbstractWorkflowBasedRunConfiguration {
	public static final String RESULT_PRECISION = "RESULT_PRECISION";

	/** The value used to indicate that no value has been specified for a property */
	public static String NO_VALUE = "";
	
	/** Friendly names for models to output nice error messages */
	/** Map Key -> model extension, Map Value -> friendly name*/
	private static Map<String,String> modelFriendlyNames;
	
	static {
		modelFriendlyNames = new HashMap<String, String>();
		modelFriendlyNames.put(RepositoryModels.REPOSITORY_MODEL_EXT, "Repository");
		modelFriendlyNames.put(RepositoryModels.SERVICE_ARCHITECTURE_MODEL_EXT, "Service Architecture");
		modelFriendlyNames.put(RepositoryModels.USAGE_MODEL_EXT, "Usage Profile");
		modelFriendlyNames.put(RepositoryModels.SEFF_BEHAVIOR_MODEL_EXT, "SEFF Behavioral");
		modelFriendlyNames.put(RepositoryModels.ANNOTATION_MODEL_EXT, "Annotation");
	}

	private IQModel usageProfileModel;
	private String alternativeId;
	private String usageModelId;
	private String qProjectName;
	private String alternativeEvaluationId;

	private String resultPrecision;

	public void initializeFrom(ILaunchConfiguration configuration) throws CoreException {
		initializeFrom(
				configuration.getAttribute(QImpressAlternativeEvaluationSelectionTab.SELECTED_PROJECT_NAME, NO_VALUE),
				configuration.getAttribute(QImpressAlternativeSelectionTab.SELECTED_ALTERNATIVE_GUID, NO_VALUE),
				configuration.getAttribute(QImpressUsageModelSelectionTab.SELECTED_USAGE_MODEL_ID, NO_VALUE),
				configuration.getAttribute(QImpressAlternativeEvaluationSelectionTab.SELECTED_ALTERNATIVE_EVALUATION_ID, NO_VALUE),
				configuration.getAttribute(RESULT_PRECISION, new Integer(ReliabilityAnalysis.DEFAULT_RESULT_PRECISION).toString())
		);
	}
	public void initializeFrom(String qProjectname, IQAlternative alternative, String usageModelId, String alternativeEvaluationId, String resultPrecision) throws CoreException {
		initializeFrom(qProjectname, (alternative==null?null:alternative.getInfo().getId()), usageModelId, alternativeEvaluationId, resultPrecision);
	}
	public void initializeFrom(String qProjectname, String alternativeId, String usageModelId, String alternativeEvaluationId, String resultPrecision) throws CoreException {
		this.qProjectName = qProjectname;
		this.alternativeId = alternativeId;
		this.alternativeEvaluationId = alternativeEvaluationId;
		this.usageProfileModel = getUsageProfilesModelContainingUsageModelId(qProjectname, alternativeId, usageModelId);
		this.usageModelId = usageModelId;
		this.resultPrecision = resultPrecision;
	}

	public String getProjectName() {
		return qProjectName;
	}
	
	/**
	 * Gets the alternative to analyze.
	 * @return the alternative to analyze.
	 */
	public IQAlternative getAlternative() {
		if(alternativeId==null) return null;
		return QImpressCore.getAlternativeById(alternativeId);
	}
	public AlternativeEvaluation getAlternativeEvaluation() {
		try {
			for(AlternativeEvaluation ae: QImpressCore.getQProject(qProjectName).getRepository().getAllAlternativeEvaluations()) {
				if(alternativeEvaluationId.equals(ae.getId())) return ae;
			}
			return createAlternativeEvaluation();
		} catch (RepositoryException e) {
			e.printStackTrace();
		}
		return null;
	}
	public AlternativeEvaluation createAlternativeEvaluation() throws RepositoryException {
		return QImpressCore.getQProject(qProjectName).getRepository().createAlternativeEvaluation(alternativeEvaluationId);
	}	
	/**
	 * Gets the id of the alternative to analyze.
	 * @return the alternative id.
	 */
	public String getAlternativeId() {
		return alternativeId;
	}
	
	@Override
	public boolean isValid() {
		return getErrorMessage() == null;
	}

	@Override
	public String getErrorMessage() {
		if (getAlternative() == null) return "No alternative specified.";

		if (getUsageModelId() == null || getUsageModelId() == "") return "No usage model selected";

		if (checkModelAvailability(getRepositoryModel())) return buildModelErrorMessage(RepositoryModels.REPOSITORY_MODEL_EXT);
		
		if (checkModelAvailability(getAnnotationsModel())) return buildModelErrorMessage(RepositoryModels.ANNOTATION_MODEL_EXT);
		
		if (checkModelAvailability(getUsageProfilesModel())) return buildModelErrorMessage(RepositoryModels.USAGE_MODEL_EXT);
		
		if (checkModelAvailability(getSeffBehavioralModel())) return buildModelErrorMessage(RepositoryModels.SEFF_BEHAVIOR_MODEL_EXT);
		
		if (checkModelAvailability(getSammModel())) return buildModelErrorMessage(RepositoryModels.SERVICE_ARCHITECTURE_MODEL_EXT);			
		
		IFolder outputFolder = getOutputFolder();
		if (outputFolder == null) return "Invalid output folder specified.";

		try {
			if(resultPrecision() < 1) {
				return "Result precision value must be greater than 0";
			}
		}catch(Exception e) {
			return "Invalid result precision value";
		}
		return null;
	}

	/**
	 * Gets the id of the usage model to analyze.
	 * @return the alternative id.
	 */	
	public String getUsageModelId() {
		return usageModelId;
	}
	public int resultPrecision() {
		return new Integer(this.resultPrecision).intValue();
	}
	/**
	 * Checks that a model is available.
	 * @param model the model to check for availability.
	 * @return an error message if the model is not available, null otherwise.
	 */
	private boolean checkModelAvailability(IQModel model) {
		IResource modelRes = model == null ? 
				null : model.getCorrespondingResource();
		
		return modelRes == null || !(modelRes instanceof IFile) || !modelRes.isAccessible();
	}

	/**
	 * Builds a nice error message for the user when a model is not available.
	 * @param modelType the model type.
	 * @return the error message.
	 */
	private String buildModelErrorMessage(String modelType) {
		return "Invalid alternative selected : " + modelFriendlyNames.get(modelType) + " model not available.";
	}
	
	public IQModel getRepositoryModel() {
		return getModelByType(RepositoryModels.REPOSITORY_MODEL_EXT);
	}
	
	public IQModel getAnnotationsModel() {
		return getModelByType(RepositoryModels.ANNOTATION_MODEL_EXT);
	}
	
	public IQModel getUsageProfilesModel() {
		return usageProfileModel;
	}
	
	public IQModel getSeffBehavioralModel() {
		return getModelByType(RepositoryModels.SEFF_BEHAVIOR_MODEL_EXT);
	}
	public IQModel getSammModel() {
		return getModelByType(RepositoryModels.SERVICE_ARCHITECTURE_MODEL_EXT);
	}
	public IQModel getResultsModel() {
		try {
			return getAlternative().getRepository().getResultModel();
		} catch (RepositoryException e) {
			// TODO report error?
		}
		return null;
	}
	
	/**
	 * Gets the EMF URI of a model.
	 * @param model the model.
	 * @return the URI.
	 */
	public URI getModelURI(IQModel model) {
		return LoadSAMMAlternativeJob.getURIForQIElement(model);
	}
	
	/**
	 * Gets a model of the alternative according to its type.
	 * @param modelExtension the extension of the model.
	 * @return the model if available, otherwise null.
	 */
	public IQModel getModelByType(String modelExtension) {
		IQAlternative alternative = getAlternative();
		if (alternative != null) {
			for (IQModel model : alternative.getModels()) {
				if (model.getType().equals(modelExtension)) return model;
			}
		}
		return null;
	}
	
	@Override
	public void setDefaults() {
		alternativeId = NO_VALUE;
		usageModelId = NO_VALUE;
		qProjectName = NO_VALUE;
		resultPrecision = (new Integer(ReliabilityAnalysis.DEFAULT_RESULT_PRECISION)).toString();
	}
	
	/**
	 * Gets the output folder for the analysis intermediate models and results.
	 * @return the output folder.
	 */
	public IFolder getOutputFolder() {
		IQAlternative alternative = getAlternative();
		return getOutputFolder(alternative);
	}
	static public IFolder getOutputFolder(IQAlternative alternative) {
		if (alternative == null) {
			return null;
		}
		return alternative.getAlternativeFolder().getFolder("reliability-analysis");
	}
	static private IQModel getUsageProfilesModelContainingUsageModelId(String qProjectName, String alternativeId, String usageModelId) {
		if(usageModelId==null) return null;
		IQAlternative[] alternatives = {
				QImpressCore.getAlternativeById(alternativeId)//,
				//getGlobalAlternative(qProjectName)
				};
		for(IQAlternative alternative: alternatives) {
			IQModel model = getUsageProfilesModelOfAlternative(alternative);
			if(model==null) continue;
			if(model.getTopLevelEObject()==null) continue;
			for ( UsageModel um: ((UsageRepository) model.getTopLevelEObject()).getUsageModels()) {
				if(usageModelId.equals(um.getId())) {
					return model;
				}
			}
		}
		return null;
	}
	static private IQModel getUsageProfilesModelOfAlternative(IQAlternative alternative) {
		if(alternative==null) return null;
		try {
			return alternative.getModel(RepositoryModels.USAGE_MODEL_EXT);
		} catch (RepositoryException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;
	}
	static private IQAlternative getGlobalAlternative(String qProjectName) {
		if (qProjectName == null || qProjectName.isEmpty()) {
			return null;
		}
		IQProject project = QImpressCore.getQProject(qProjectName);
		if (project == null) {
			return null;
		}
		try {
			return project.getRepository().getGlobalAlternative();
		} catch (RepositoryException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}		
		return null;
	}
	public ModelLocation getSammModelLocation() {
		return new ModelLocation(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID, getModelURI(getSammModel()));
	}
	public ModelLocation getUsageModelLocation() {
		return new ModelLocation(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID, getModelURI(getResultsModel()));
	}
	public ServiceArchitectureModel getServiceArchitecturalModel() {
		return (ServiceArchitectureModel) getSammModel().getTopLevelEObject();
	}
	public UsageModel getUsageModel() {
		for ( UsageModel um: ((UsageRepository) getUsageProfilesModelContainingUsageModelId(qProjectName, alternativeId, usageModelId).getTopLevelEObject()).getUsageModels()) {
			if(usageModelId.equals(um.getId())) {
				return um;
			}
		}
		return null;
	}	
	@Override
	public boolean isInteractive() {
		return true;
	}
}
