package eu.qimpress.ide.analysis.reliability.jobs.prism;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;

import de.uka.ipd.sdq.workflow.IJobWithResult;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;

/**
 * A job which launches a Prism analysis and returns the values for the specified properties as a Map.
 * The key of the Map is the property while the value is the property value computed by prism.
 * @author Mauro Luigi Drago
 *
 */
public class PrismAnalysisJob 
	implements IJobWithResult<Map<String,String>> {
	
	/** The Class logger */
	private static final Logger logger = Logger.getLogger(PrismAnalysisJob.class);
	
	/** The job name */
	private static final String JOB_NAME = "Prism Analysis";
	
	/** The job configuration */
	private PrismAnalysisJobConf conf;
	
	/** The values of the properties computed by prism */
	private Map<String,String> propsValues = new HashMap<String,String>();
	
	/**
	 * Creates a new <code>PrismAnalysisJob</code>
	 * @param conf the configuration.
	 */
	public PrismAnalysisJob(PrismAnalysisJobConf conf) {
		this.conf = conf;
	}
	
	/**
	 * Checks that the job can be executed. 
	 * @throws JobFailedException if the job could not be executed
	 */
	private void check() throws JobFailedException {
		if (!conf.isValid()) throw new JobFailedException(conf.getErrorMessage());
	}

	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException,
			UserCanceledException {
		check();
	
		logger.debug("Preparing prism process builder");
		List<String> args = new ArrayList<String>();
		
		configPrismExecutableByOs(args);
		try {
			logger.debug("Starting prism...");
			File execDir = getPrismDir();
			if (execDir != null) logger.info("Using exec directory : " + execDir.toString());
			Process proc = Runtime.getRuntime().exec(args.toArray(new String[args.size()]), null, execDir); 
						
			StreamRedirector outRed = new StreamRedirector(
					proc.getInputStream(),
					logger,
					StreamRedirector.Type.NORMAL);
			StreamRedirector errRed = new StreamRedirector(
					proc.getErrorStream(),
					logger,
					StreamRedirector.Type.ERROR);
			outRed.start();
			errRed.start();
			
			try {
				int exitVal = proc.waitFor();
				logger.debug("Prism terminated (exit value =" + exitVal + "), importing results...");
				analyzeResults();
			} catch (InterruptedException e) {
				throw new JobFailedException("Unable to launch Prism analysis", e);
			}
		} catch (IOException e) {
			throw new JobFailedException("Unable to launch Prism analysis", e);
		}
	}
	
	private void configPrismExecutableByOs(List<String> args) throws JobFailedException {
		String osName = System.getProperty("os.name");
		if (osName.contains("Windows")) {
			logger.debug("Detected Windows OS");
			
			logger.debug("Creating .bat file to run prism");
			String batFileName = createBatFile();
			
			args.add("cmd.exe");
			args.add("/C");
			args.add(batFileName);
		} else {
			logger.debug("Linux Based OS");
			args.add(conf.getPrismExecutable().toString());
			args.add(conf.getModelFile().toString());
			args.add(conf.getPropsFile().toString());
			args.add("-fixdl");
			args.add("-exportresults");
			args.add(conf.getPropsFile().toString());
		}
	}
	
	private String createBatFile() throws JobFailedException {
		String batFileName = conf.getTempDir().toString() + File.separatorChar + "analyze.bat";
		try {
			PrintWriter pw = new PrintWriter(new File(batFileName));
			try {
				pw.println("@echo off");
				pw.println("set PRISM_DIR=.");
				pw.println("path=%PRISM_DIR%\\lib;%path%");
				pw.println("set CP=%PRISM_DIR%\\lib\\prism.jar;%PRISM_DIR%\\classes;%PRISM_DIR%;%PRISM_DIR%\\lib\\pepa.zip;%PRISM_DIR%\\lib\\jcommon.jar;%PRISM_DIR%\\lib\\jfreechart.jar;%PRISM_DIR%\\lib\\epsgraphics.jar");
				pw.println("java -Djava.library.path=\"%PRISM_DIR%\\lib\" -classpath \"%CP%\" prism.PrismCL" +
						" \"" + conf.getModelFile().toString() + "\"" +
						" \"" + conf.getPropsFile().toString() + "\"" +
						" -fixdl" +
						" -exportresults" +
						" \"" + conf.getPropsFile().toString() + "\"");
			} finally {
				pw.close();
			}
		} catch (FileNotFoundException e) {
			throw new JobFailedException("Unable to create .bat file to run prism : " + e.getMessage());
		}
		
		return batFileName;
	}
	
	private File getPrismDir() {
		String osName = System.getProperty("os.name");
		if (osName.contains("Windows")) {
			String prismDir = conf.getPrismExecutable().toString();
			prismDir = prismDir.substring(0, prismDir.lastIndexOf('\\'));
			prismDir = prismDir.substring(0, prismDir.lastIndexOf('\\'));
			return new File(prismDir);
		}
		
		return null;
	}
	
	private void analyzeResults() throws JobFailedException {
		try {
			FileReader fr = new FileReader(conf.getPropsFile());
			try {
				BufferedReader br = new BufferedReader(fr);
				try {
					String prop = br.readLine();
					while (prop != null) {
						br.readLine(); // skip next line
						String value = br.readLine();
						propsValues.put(prop, value);
						
						prop = br.readLine();
					}
				} catch (IOException e) {
					//TODO handle me
				} finally {
					try {
						br.close();
					} catch (IOException e) {
						throw new JobFailedException("Unable to close property values file", e);
					}
				}
			} finally {
				try {
					fr.close();
				} catch (IOException e) {
					throw new JobFailedException("Unable to close property values file", e);
				}
			}
		} catch (FileNotFoundException e) {
			throw new JobFailedException("Unable to parse property values file", e);
		}
	}

	@Override
	public String getName() {
		return JOB_NAME;
	}

	@Override
	public void rollback(IProgressMonitor monitor)
			throws RollbackFailedException {
		// not needed for now
	}

	@Override
	public Map<String, String> getResult() {
		return propsValues;
	}

}
