package eu.qimpress.ide.analysis.reliability.jobs.acceleo;

import java.io.File;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.acceleo.engine.event.AcceleoTextGenerationEvent;
import org.eclipse.acceleo.engine.event.IAcceleoTextGenerationListener;
import org.eclipse.acceleo.engine.service.AcceleoService;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.Monitor;
import org.eclipse.emf.common.util.URI;

import de.uka.ipd.sdq.workflow.IJobWithResult;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import eu.qimpress.ide.analysis.reliability.jobs.acceleo.AcceleoTransformationJobConf;

/**
 * A job that performs an Acceleo M2T transformation.
 * 
 * @author Mauro Luigi Drago
 * 
 */
public class AcceleoTransformationJob 
	implements IJobWithResult<Set<File>> {

	private static final String JOB_NAME = "Perform Acceleo M2T transformation";
	
	/** The Class logger */
	private static final Logger logger = Logger.getLogger(AcceleoTransformationJob.class);
	
	/** The name of the method in the Acceleo generator class to invoke the transformation */
	private static final String GENERATOR_METHOD_NAME = "doGenerate";
	
	/** A list containing information about the transformation execution */
	private Set<File> result = new HashSet<File>();
	
	/** A reference to the job configuration */
	private AcceleoTransformationJobConf configuration;

	/**
	 * Creates a new <code>AcceleoTransformationJob</code>.
	 * @param conf the configuration for the job.
	 */
	public AcceleoTransformationJob(AcceleoTransformationJobConf configuration) {
		this.configuration = configuration;
	}

	@Override
	public Set<File> getResult() {
		return result;
	}

	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException, UserCanceledException {
		// Invoke the transformation
		Set<String> result = runTransformation(monitor);
		
		logger.debug("Generated files are :");
		for (String filePath : result) {
			logger.debug("* " + filePath);
		}
		
		for (String filePath : result) {
			this.result.add(new File(filePath));
		}
	}

	/**
	 * Runs the transformation.
	 * @return the set of files created during the transformation 
	 * @throws JobFailedException if the tranformation failed.
	 */
	private Set<String> runTransformation(IProgressMonitor monitor) throws JobFailedException {
		logger.info("Starting Transformation...");
		
		// Create an instance of the generator
		logger.debug("Creating a generator class instance...");
		Class<?> genClass = configuration.getAcceleoGeneratorClass();
		Object generator;
		try {
			java.lang.reflect.Constructor<?> constr = genClass.getConstructor(URI.class, File.class, List.class);
			generator = constr.newInstance(
					configuration.getInputModel(),
					new File(configuration.getOutputFolder().getLocation().toString()),
					configuration.getArguments()
					);
		} catch (Exception e) {
			throw new JobFailedException("Unable to instantiate generator class.", e);
		}
		
		// Retrieve the doGenerate method
		Method doGenerateMethod;
		try {
			Class<?>[] params = {
				Monitor.class	
			};
			doGenerateMethod = genClass.getMethod(GENERATOR_METHOD_NAME, params);
		} catch (Exception e) {
			throw new JobFailedException("Invalid generator class.", e);
		}
		
		// Add a listener to the Acceleo service to identify created files
		logger.debug("Attaching listener to AcceleoService...");
		AcceleoTextGenerationListener l = new AcceleoTextGenerationListener();
		AcceleoService.addListener(l);
		
		// Invoke the transformation
		logger.debug("Invoking transformation...");
		try {
			doGenerateMethod.invoke(generator, BasicMonitor.toMonitor(monitor));
		} catch (Exception e) {
			throw new JobFailedException("Unable to execute tranformation.", e);
		}
		
		// Remove the listener
		logger.debug("Detaching listener to AcceleoService...");
		AcceleoService.removeListener(l);
		
		return l.generatedFiles;
	}

	@Override
	public String getName() {
		return JOB_NAME;
	}

	@Override
	public void rollback(IProgressMonitor monitor) throws RollbackFailedException {
		// Not needed yet.
	}
	
	private static class AcceleoTextGenerationListener implements IAcceleoTextGenerationListener {
		private Set<String> generatedFiles = new HashSet<String>();
		
		@Override
		public void textGenerated(AcceleoTextGenerationEvent event) {
		}
		
		@Override
		public void filePathComputed(AcceleoTextGenerationEvent event) {
			generatedFiles.add(event.getText());
		}
	}
}
