package eu.qimpress.ide.analysis.reliability.jobs;

import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;

import de.uka.ipd.sdq.workflow.IJobWithResult;
import de.uka.ipd.sdq.workflow.OrderPreservingBlackboardCompositeJob;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.WorkflowFailedException;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.MDSDBlackboard;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.ModelLocation;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.SavePartitionToDiskJob;
import de.uka.ipd.sdq.workflow.mdsd.emf.qvto.QVTOTransformationJob;
import de.uka.ipd.sdq.workflow.mdsd.emf.qvto.QVTOTransformationJobConfiguration;
import eu.qimpress.ide.analysis.reliability.launch.ReliabilityLaunchConfiguration;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.transformations.common.jobs.LoadSAMMAlternativeJob;
import eu.qimpress.transformations.common.jobs.LoadSAMMIntoBlackboardJob;

/**
 * A job which executes a the reliability analysis for a specified system call in a usage scenario.
 * 
 * @author Mauro Luigi Drago, Andrea Ciancone
 *
 */
public class SysCallAnalysisJob 
	extends OrderPreservingBlackboardCompositeJob<MDSDBlackboard>
	implements IJobWithResult<ReliabilityAnalysisResult> {
	
	/** The Class logger */
	private Logger logger = Logger.getLogger(SysCallAnalysisJob.class);
	
	/** Model transformation inout models information */
	private static final int SAMM2KLAPER_PARAM_NUMBER = 6;
	private static final int KLAPER2DTMC_PARAM_NUMBER = 2;
	private static final int DTMCPREP_PARAM_NUMBER = 2;
	
	/** The name of the reliability partition */
	private String reliabilityPartitionName;
	
//	/** The Prism Analsysis Job*/
//	private PrismAnalysisJob prismAnalysisJob;
	
	/** The configuration */
	private SysCallAnalysisJobConf config;

	private ReliabilityAnalysis reliabilityAnalysis;
	
	/**
	 * Creates a new system call reliability analysis job.
	 * @param config the job configuration.
	 */
	public SysCallAnalysisJob(SysCallAnalysisJobConf config) {
		super();
		
		this.config = config;
	
		// Add a job to prepare the Reliability partition specific to this task
		 PrepareReliabilityBlackboardPartitionConf prbpConf = new PrepareReliabilityBlackboardPartitionConf();
		 prbpConf.setPartitionName(config.getReliabilityConf().getAlternativeId());
		 reliabilityPartitionName = prbpConf.getPartitionName();
		 add(new PrepareReliabiltyBlackboardPartionJob(prbpConf));

		// SAMM2Klaper model transformation job
		QVTOTransformationJob samm2klaperJob = 
			new QVTOTransformationJob(createSamm2KlaperConf(config));
		add(samm2klaperJob);
		
		// save models job
		add(new SavePartitionToDiskJob(reliabilityPartitionName));
		
		// Klaper2Dtmc model transformation job
		QVTOTransformationJob klaper2dtmcJob = 
			new QVTOTransformationJob(createKlaper2DtmcConf(config));
		add(klaper2dtmcJob);
		
		// Dtmc model preparation job
		QVTOTransformationJob dtmcPreparationJob = 
			new QVTOTransformationJob(createDtmcPreparationConf(config));
		add(dtmcPreparationJob);
		
		// save models job
		add(new SavePartitionToDiskJob(reliabilityPartitionName));
		
		reliabilityAnalysis = new ReliabilityAnalysis(getDtmcModelURI(config), config.getReliabilityConf().resultPrecision());
		add(reliabilityAnalysis);
	}
	
	private QVTOTransformationJobConfiguration createSamm2KlaperConf(SysCallAnalysisJobConf config) {
		// Creating and setting transformation configuration
		QVTOTransformationJobConfiguration jobConf = new QVTOTransformationJobConfiguration();
		
		// the model transformation options 
		Map<String, Object> mtOpt = new HashMap<String, Object>();
		logger.debug("Samm2Klaper : setting transformation options...");
		mtOpt.put("usageScenarioId", config.getUsageScenarioId());
		
		ModelLocation[] inoutModels = new ModelLocation[SAMM2KLAPER_PARAM_NUMBER];

		// Add input models
		ReliabilityLaunchConfiguration reliabilityConf = config.getReliabilityConf();
		logger.debug("Samm2Klaper : setting input models...");
		IQModel[] inputModels = {
			reliabilityConf.getSammModel(),
			reliabilityConf.getRepositoryModel(),
			reliabilityConf.getSeffBehavioralModel(),
			reliabilityConf.getAnnotationsModel(),
			reliabilityConf.getUsageProfilesModel()
		};
		for(int i=0; i<inputModels.length; i++) {
			try {
				new EmfValidatorHelper().validateEObject(inputModels[i].getTopLevelEObject());
			} catch (ValidationException e) {
				// new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e);
				logger.debug("Invalid SAMM model ("+inputModels[i]+")");				
				throw new WorkflowFailedException("Invalid SAMM model ("+inputModels[i]+")", e);
			}
			inoutModels[i] = new ModelLocation(LoadSAMMIntoBlackboardJob.SAMM_MODELS_PARTITION_ID, 
					reliabilityConf.getModelURI(inputModels[i]));			
		}
		
		// Add output models
		logger.debug("Samm2Klaper : setting output models...");
		inoutModels[5] = new ModelLocation(
				reliabilityPartitionName, 
				getKlaperModelURI(config));

		logger.debug("Samm2Klaper : saving configuration...");
		jobConf.setInoutModels(inoutModels);
		jobConf.setOptions(mtOpt);
		
		logger.debug("Samm2Klaper : setting traces and script location...");
		URI traceFileURI = LoadSAMMAlternativeJob.getURIForQIElement(reliabilityConf.getAlternative()).
			appendSegment("Samm2Klaper.qvtotrace");
		jobConf.setTraceFileURI(traceFileURI);
		
		URI tScriptURI = URI.createPlatformPluginURI(
				"eu.qimpress.transformations.reliability.m2m.samm2klaper/transforms/samm2klaper.qvto", 
				true);
		jobConf.setScriptFileURI(tScriptURI);
		return jobConf;
	}
	
	private QVTOTransformationJobConfiguration createKlaper2DtmcConf(SysCallAnalysisJobConf config) {
		// Creating and setting transformation configuration
		QVTOTransformationJobConfiguration jobConf = new QVTOTransformationJobConfiguration();
		
		// the model transformation options 
		Map<String, Object> mtOpt = new HashMap<String, Object>();
		logger.debug("Klaper2Dtmc : setting transformation options...");

		ModelLocation[] inoutModels = new ModelLocation[KLAPER2DTMC_PARAM_NUMBER];

		// Add input models
		logger.debug("Klaper2Dtmc : setting input models...");
		inoutModels[0] = new ModelLocation(
				reliabilityPartitionName, 
				getKlaperModelURI(config));

		// Add ouput models
		logger.debug("Klaper2Dtmc : setting output models...");
		inoutModels[1] = new ModelLocation(
				reliabilityPartitionName, 
				getDtmcModelURI(config));

		logger.debug("Klaper2Dtmc : saving configuration...");
		jobConf.setInoutModels(inoutModels);
		jobConf.setOptions(mtOpt);
		
		logger.debug("Klaper2Dtmc : setting traces and script location...");
		URI traceFileURI = LoadSAMMAlternativeJob.getURIForQIElement(config.getReliabilityConf().getAlternative()).
			appendSegment("Klaper2Dtmc.qvtotrace");
		jobConf.setTraceFileURI(traceFileURI);
		
		URI tScriptURI = URI.createPlatformPluginURI(
				"eu.qimpress.transformations.reliability.m2m.klaper2dtmc/transforms/klaper2dtmc.qvto", 
				true);
		jobConf.setScriptFileURI(tScriptURI);
		return jobConf;
	}
	
	private QVTOTransformationJobConfiguration createDtmcPreparationConf(SysCallAnalysisJobConf config) {
		// Creating and setting transformation configuration
		QVTOTransformationJobConfiguration jobConf = new QVTOTransformationJobConfiguration();
		
		// the model transformation options 
		Map<String, Object> mtOpt = new HashMap<String, Object>();
		logger.debug("DtmcPreparation : setting transformation options...");

		ModelLocation[] inoutModels = new ModelLocation[DTMCPREP_PARAM_NUMBER];

		// Add input models
		logger.debug("DtmcPreparation : setting input models...");
		inoutModels[0] = new ModelLocation(
				reliabilityPartitionName, 
				getDtmcModelURI(config));

		// Add ouput models
		logger.debug("DtmcPreparation : setting output models...");
		inoutModels[1] = new ModelLocation(
				reliabilityPartitionName, 
				getDtmcModelURI(config));

		logger.debug("DtmcPreparation : saving configuration...");
		jobConf.setInoutModels(inoutModels);
		jobConf.setOptions(mtOpt);
		
		logger.debug("DtmcPreparation : setting traces and script location...");
		URI traceFileURI = LoadSAMMAlternativeJob.getURIForQIElement(config.getReliabilityConf().getAlternative()).
			appendSegment("Klaper2Dtmc.qvtotrace");
		jobConf.setTraceFileURI(traceFileURI);
		
		URI tScriptURI = URI.createPlatformPluginURI(
				"eu.qimpress.transformations.reliability.m2m.dtmcPreparation/transforms/dtmcPreparation.qvto", 
				true);
		jobConf.setScriptFileURI(tScriptURI);
		return jobConf;
	}
	private IFolder getLocalOutputFolder(SysCallAnalysisJobConf config) {
		return config.getReliabilityConf().getOutputFolder();
	}
	
	private URI getKlaperModelURI(SysCallAnalysisJobConf config) {
		return URI.createFileURI(getLocalOutputFolder(config).getFile("/out.klaper").getLocation().toString());
	}
	
	private URI getDtmcModelURI(SysCallAnalysisJobConf config) {
		return URI.createFileURI(getLocalOutputFolder(config).getFile("/out.dtmc").getLocation().toString());
	}
	
	@Override
	public ReliabilityAnalysisResult getResult() {
		return reliabilityAnalysis.getResult();
	}
	@Override
	public void rollback(IProgressMonitor monitor)
			throws RollbackFailedException {
		// Nothing to do for now
	}
	
	public SysCallAnalysisJobConf getConfiguration() {
		return config;
	}
}