package eu.qimpress.ide.analysis.reliability.jobs;

import java.io.IOException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;

import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import eu.qimpress.dtmc.Dtmc;
import eu.qimpress.ide.analysis.reliability.rmc.core.*;
import eu.qimpress.ide.analysis.reliability.rmc.importer.dtmc.*;
import eu.qimpress.ide.analysis.reliability.rmc.mat.*;

public class ReliabilityAnalysis implements IJob {
	public static int DEFAULT_RESULT_PRECISION = 17;

	URI dtmcModelFile;
	private RMC rmc;
	private Solver ns;

	private int resultDecimalDigitsPrecision;
	public ReliabilityAnalysis(URI dtmcModelFile, int resultDecimalDigitsPrecision) {
		this.dtmcModelFile = dtmcModelFile;
		this.resultDecimalDigitsPrecision = resultDecimalDigitsPrecision;
	}
	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException, UserCanceledException {
		DtmcModel dtmcModel;
		double resultPrecision = java.lang.Math.pow (10, -resultDecimalDigitsPrecision);
		long maximumNumberOfIterations = (long) Math.pow(2, resultDecimalDigitsPrecision);

		long starttime = 0;
		long totaltime = 0;
		
		try {
			dtmcModel = new DtmcModel(
					(Dtmc) new ResourceHelper().fromFile(dtmcModelFile).get(0));
		} catch (IOException e1) {
			e1.printStackTrace();
			throw new JobFailedException(e1.getLocalizedMessage());
		}
		try {
			rmc=new Transfomation().generateRMC(dtmcModel);
			rmc.computeEquations();
			System.out.println("RMC validation : "+rmc.validate());
			System.out.println("Equations: {");
			for(String s : rmc.getEquations()){
				System.out.println(s);
			}
			System.out.println("}");

			if(rmc.validate()==false){
				throw new JobFailedException("Invalid RMC equations");
			}

				System.out.println("RMC Solver");
				RmcVariables inputVariables = new RmcVariables();
				ns = new RmcSolver(inputVariables, rmc.getRmcEquations(inputVariables));
				starttime = System.currentTimeMillis();
				ns.iterateUntilMSVlessThan(resultPrecision, maximumNumberOfIterations);
				totaltime = System.currentTimeMillis()-starttime;
				ns.printCurrentAssign();
				System.out.println("Milliseconds: "+totaltime);
				System.out.println("Reliability: "+ns.getValue(rmc.getInitialModule().getName()+rmc.getInitialModule().getStart().getName()));
				System.out.println("MSV: "+ns.getMSV());
				System.out.println("Iterations: "+ns.getNumIterations());
		} catch (Exception e) {
			e.printStackTrace();
			throw new JobFailedException(e.getMessage());
		}
	}
	public ReliabilityAnalysisResult getResult() {
		return new ReliabilityAnalysisResult(rmc,  ns);
	}
	@Override
	public String getName() {
		return "Reliability analysis";
	}
	@Override
	public void rollback(IProgressMonitor monitor)
			throws RollbackFailedException {
	}
}