package eu.qimpress.ide.analysis.reliability.jobs;

import java.lang.reflect.InvocationTargetException;
import java.util.Date;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;

import de.uka.ipd.sdq.workflow.IBlackboardInteractingJob;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.MDSDBlackboard;
import eu.qimpress.ide.backbone.core.operations.SaveQModelUIOperation;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.Reliability;
import eu.qimpress.resultmodel.ReliabilityPredictionResult;
import eu.qimpress.resultmodel.ResultDistribution;
import eu.qimpress.resultmodel.ResultModelFactory;
import eu.qimpress.resultmodel.ResultModelPackage;
import eu.qimpress.samm.staticstructure.ServiceArchitectureModel;
import eu.qimpress.samm.usagemodel.UsageScenario;

/**
 * A job to insert reliability analysis results into the results metamodel.
 * @author Mauro Luigi Drago, Andrea Ciancone
 *
 */
public class FillResultsModelJob 
	implements IJob, IBlackboardInteractingJob<MDSDBlackboard> {
	
	/** The Class logger */
	private Logger logger = Logger.getLogger(FillResultsModelJob.class);
	
	/** The Job Name*/
	private static final String JOB_NAME = "Fill Results Model Job";
	
	/** The blackboard */
	private MDSDBlackboard blackboard;
	
	private SysCallAnalysisJob sysCallAnalysis;
	
	/** The configuration */
	private SysCallAnalysisJobConf conf;
	
	private ReliabilityPredictionResult reliabilityPredictionResult;
	
	public FillResultsModelJob(SysCallAnalysisJobConf conf, SysCallAnalysisJob sysCallAnalysis) {
		this.conf = conf;
		this.sysCallAnalysis = sysCallAnalysis;
	}
	
	/**
	 * Checks that the job can be executed. 
	 * @throws JobFailedException if the job could not be executed
	 */
	private void check() throws JobFailedException {
		if (!conf.isValid()) throw new JobFailedException(conf.getErrorMessage());
	}

	@Override
	public void execute(IProgressMonitor monitor) 
		throws JobFailedException, UserCanceledException {
		
		check();
		
		// get the model factory
		ResultModelFactory factory = ResultModelPackage.eINSTANCE.getResultModelFactory();
								
		AlternativeEvaluation ae = conf.getReliabilityConf().getAlternativeEvaluation();
		if(ae==null) {
			throw new JobFailedException("Failed to retrieve analysis result model.");
		}

		logger.debug("Adding prediction results...");
		reliabilityPredictionResult = factory.createReliabilityPredictionResult();
		ae.getAnalysisResults().add(reliabilityPredictionResult);
		reliabilityPredictionResult.getReliabilities();
					
		ResultDistribution distribution = factory.createResultDistribution();
		distribution.set_10PercentQuantile(sysCallAnalysis.getResult().getReliability());
		distribution.set_90PercentQuantile(sysCallAnalysis.getResult().getReliability());
		distribution.setArithmeticMean(sysCallAnalysis.getResult().getReliability());
		distribution.setMedian(sysCallAnalysis.getResult().getReliability());
			
		Reliability reliability = factory.createReliability();
		reliability.setResultDistribution(distribution);
		
		// TODO modify the usage scenario attribute in alternative result model
		for(UsageScenario us : conf.getReliabilityConf().getUsageModel().getUsageScenarios()) {
			if(us.getId().contentEquals(conf.getUsageScenarioId()))
				reliability.setSystemCall(us.getSystemCalls().get(0));
				//reliability.setUsageScenario(us);
		}
			
		logger.debug("Adding reliability result...");
		reliabilityPredictionResult.getReliabilities().add(reliability);
		
		// set the alternative
		logger.debug("Update alternative evaluation...");
		ae.setAlternativeId(conf.getReliabilityConf().getAlternativeId());
		ae.setRunDate(new Date());
		
		logger.debug("Loading samm...");
		ae.setServiceArchitectureModel(conf.getReliabilityConf().getServiceArchitecturalModel());

		logger.debug("Set usage model...");
		ae.setUsageModel(conf.getReliabilityConf().getUsageModel());
		
		try {
			new SaveQModelUIOperation(conf.getReliabilityConf().getResultsModel()).run(monitor);
		} catch (InvocationTargetException e) {
			throw new JobFailedException("Failed to save result model.");
		} catch (InterruptedException e) {
			throw new JobFailedException("Failed to save result model.");
		}
	}

	@Override
	public String getName() {
		return JOB_NAME;
	}

	@Override
	public void rollback(IProgressMonitor monitor) throws RollbackFailedException {
//		if (reliabilityPredictionResult != null) {
//			logger.debug("Revert reliability result writing...");
//			conf.getReliabilityConf().getAlternativeEvaluation().getAnalysisResults().remove(reliabilityPredictionResult);
//		}
	}

	@Override
	public void setBlackboard(MDSDBlackboard blackboard) {
		this.blackboard = blackboard;
	}
}
