package eu.qimpress.ide.analysis.reliability.jobs;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.Diagnostician;
import org.eclipse.emf.ecore.util.EcoreValidator;

/**
 * Validator Helper of EMF Models 
 * @author Andrea Ciancone
 *
 */
public class EmfValidatorHelper {
	/**
	 * Validates an input model w.r.t. its meta-model definition, included the meta-model constrains 
	 * @param eObject input model
	 * @throws ValidationException raised in case of model validation failure. It contains a list of the model errors
	 */
	public void validateEObject(EObject eObject) throws ValidationException {
		if (!validateEObjectSyntax(eObject)) {
			throw new ValidationException("Model is not comform to its meta-model");
		}
		List<String> errors = validateEObjectSemantic(eObject);
		if(errors.size() > 0)
			throw new ValidationException("Meta-model constraints check failed ("+errors.size()+")", errors);
	}
	/**
	 * Validates an input model w.r.t. its meta-model definition and constrains, assuming the model is syntactically valid.
	 * @param eObject input model
	 * @return a list of the model errors
	 */
	protected List<String> validateEObjectSemantic(EObject eObject) {
		Diagnostic diagnostic = Diagnostician.INSTANCE.validate(eObject);
		ArrayList<String> errs = new ArrayList<String>();
		if (diagnostic.getSeverity() != Diagnostic.ERROR && diagnostic.getSeverity() != Diagnostic.WARNING) {
			return errs;
		}
		errs.add(diagnostic.getMessage());
		for (Iterator<Diagnostic> i = diagnostic.getChildren().iterator(); i.hasNext();) {
			Diagnostic childDiagnostic = i.next();
			switch (childDiagnostic.getSeverity()) {
			case Diagnostic.ERROR:
				errs.add(childDiagnostic.getMessage());
			case Diagnostic.WARNING:
				errs.add(childDiagnostic.getMessage());
			}
		}
		return errs;
	}
	/**
	 * Validates an input model w.r.t. the syntax definition of EMF models
	 * @param eObject input model
	 * @return if the model is correct return true otherwise return false
	 */
	protected boolean validateEObjectSyntax(EObject eObject) {
		return EcoreValidator.INSTANCE.validate(eObject, null, null);
	}
}