package eu.qimpress.ide.tradeoff.ahp;

import java.util.ArrayList;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import Jama.Matrix;
import eu.qimpress.ide.tradeoff.ahp.AHPWizard.Qualities;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.CpuResourceUtilization;
import eu.qimpress.resultmodel.HddResourceUtilization;
import eu.qimpress.resultmodel.MaintainabilityPredictionResult;
import eu.qimpress.resultmodel.PerformancePredictionResult;
import eu.qimpress.resultmodel.Reliability;
import eu.qimpress.resultmodel.ReliabilityPredictionResult;
import eu.qimpress.resultmodel.ResponseTime;
import eu.qimpress.resultmodel.ResultDistribution;
import eu.qimpress.resultmodel.Throughput;
import eu.qimpress.samm.deployment.targetenvironment.ExecutionResource;
import eu.qimpress.samm.deployment.targetenvironment.StorageResource;
import eu.qimpress.samm.usagemodel.SystemCall;
import eu.qimpress.samm.usagemodel.UsageScenario;

public class ValueRatingPage extends WizardPage {
	public static final String PAGE_NAME = "Value Rating";

	private static double[] translate = { 1.0 / 9.0, 1.0 / 7.0, 1.0 / 5.0,
			1.0 / 3.0, 1.0, 3.0, 5.0, 7.0, 9.0 };
	private Matrix[] valueMatrix;
	private int[][] settings;
	private ArrayList<AlternativeEvaluation> evaluations;
	private int numEvals;
	private boolean[] QualityInUse;
	private ScrolledComposite sc;
	private Composite globalParent;

	public ValueRatingPage(Matrix[] m,
			ArrayList<AlternativeEvaluation> evaluations, int[][] is, boolean[] qualityInUse) {
		super(PAGE_NAME, "Assess alternatives", null);
		setDescription("Please rate actual values to be compared using AHP analysis");
		valueMatrix = m;
		settings = is;
		this.evaluations = evaluations;
		numEvals = evaluations.size();
		QualityInUse = qualityInUse;
	}

	@Override
	public void createControl(Composite parent) {
		globalParent = parent;
		drawControl(QualityInUse);
	}

	public void drawControl(boolean[] qualityInUse){
		QualityInUse = qualityInUse;
		Composite parent = globalParent;
		if (sc != null)
			sc.dispose();
		sc = new ScrolledComposite(parent, SWT.H_SCROLL
				| SWT.V_SCROLL);
		Composite topLevel = new Composite(sc, SWT.NONE);
		sc.setContent(topLevel);
		topLevel.setLayout(new GridLayout(3, false));

		GridData radioWidth = new GridData();
		radioWidth.widthHint = 30;
		radioWidth.horizontalAlignment = SWT.CENTER;
		radioWidth.verticalAlignment = SWT.CENTER;

		Label q1 = new Label(topLevel, SWT.NONE);
		q1.setText("Value");
		q1.setLayoutData(new GridData(SWT.END));

		Composite titlesComposite = new Composite(topLevel, SWT.NONE);
		titlesComposite.setLayout(new GridLayout(9, false));

		String[] titles = { "EX", "VS", "ST", "MO", "EQ", "MO", "ST", "VS",
				"EX" };

		for (int i = 0; i < 9; i++) {
			Label l = new Label(titlesComposite, SWT.CENTER);
			l.setText(titles[i]);
			l.setLayoutData(radioWidth);
		}
		titlesComposite.pack();
		Label q2 = new Label(topLevel, SWT.NONE);
		q2.setText("Value");
		q2.setLayoutData(new GridData(SWT.BEGINNING));

		// Utilization
		if (QualityInUse[Qualities.UTILIZATION.ordinal()])
			createComparisonBatch(topLevel, radioWidth, Qualities.UTILIZATION);

		// Cost
		if (QualityInUse[Qualities.COST.ordinal()])
			createComparisonBatch(topLevel, radioWidth, Qualities.COST);

		// Reliability
		if (QualityInUse[Qualities.RELIABILITY.ordinal()])
			createComparisonBatch(topLevel, radioWidth, Qualities.RELIABILITY);

		// Response time
		if (QualityInUse[Qualities.RESPONSETIME.ordinal()])
			createComparisonBatch(topLevel, radioWidth, Qualities.RESPONSETIME);

		// HDD Utilization
		if (QualityInUse[Qualities.HDDUTILIZATION.ordinal()])
			createComparisonBatch(topLevel, radioWidth,
					Qualities.HDDUTILIZATION);

		// Throughput
		if (QualityInUse[Qualities.THROUGHPUT.ordinal()])
			createComparisonBatch(topLevel, radioWidth, Qualities.THROUGHPUT);
		
		Label description = new Label(topLevel, SWT.NONE);
		description
				.setText("Legend of quality ratings:\nEX Extreme\nVS Very Strong\nST Strong\nMO Moderate\nEQ Equal");
		GridData descgrid = new GridData();
		descgrid.verticalSpan = 3;
		description.setLayoutData(descgrid);

		topLevel.pack(true);
		parent.layout(true, true);
		parent.pack();
		parent.redraw();
		setControl(sc);
		setPageComplete(true);

	}
	
	private void createComparisonBatch(Composite parent,
			GridData radioButtonWidth, Qualities quality) {
		for (int I = 0; I < numEvals; I++) {
			for (int J = I + 1; J < numEvals; J++) {
				Label a = new Label(parent, SWT.LEFT);
				a.setText(evaluations.get(I).getName()
						+ getValueForQuality(evaluations.get(I), quality));
				a.setToolTipText(getToolTipForQuality(evaluations.get(I),
						quality));
				GridData agd = new GridData();
				agd.horizontalAlignment = SWT.END;
				agd.verticalAlignment = SWT.CENTER;
				a.setLayoutData(agd);

				Composite radioGroup = new Composite(parent, SWT.NONE);
				radioGroup.setLayout(new GridLayout(9, false));
				for (int i = 0; i < 9; i++) {
					Button b = new Button(radioGroup, SWT.RADIO);
					b.setLayoutData(radioButtonWidth);
					b.addSelectionListener(new RadioListener(valueMatrix,
							quality.ordinal(), I, J, i));
					b.setSelection(i == settings[quality.ordinal()][I*numEvals+J]);
				}
				radioGroup.pack();

				Label b = new Label(parent, SWT.LEFT);
				b.setText(evaluations.get(J).getName()
						+ getValueForQuality(evaluations.get(J), quality));
				b.setToolTipText(getToolTipForQuality(evaluations.get(J),
						quality));
				GridData bgd = new GridData();
				bgd.horizontalAlignment = SWT.BEGINNING;
				bgd.verticalAlignment = SWT.CENTER;
				b.setLayoutData(bgd);
			}
		}
	}

	private static String newline = "\n";


	public static String getToolTipForQuality(AlternativeEvaluation e, Qualities q) {
		String retval = "Alternative evaluation: " + e.getName() + newline;

		if (q == Qualities.UTILIZATION) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					for (CpuResourceUtilization cru : p
							.getResourceUtilizations()) {
						retval += "Utilization: "
								+ myToString(cru.getResultDistribution())
								+ " on "
								+ myToString(cru.getExecutionResource())
								+ newline;
					}
				}
			}
		}

		if (q == Qualities.RESPONSETIME) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					for (ResponseTime r : p.getResponseTimes()) {
						retval += "Response time: "
								+ myToString(r.getResultDistribution())
								+ " on " + myToString(r.getUsageScenario())
								+ newline;
					}
				}
			}
		}

		if (q == Qualities.HDDUTILIZATION) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					for (HddResourceUtilization r : p.getHddResourceUtilizations()) {
						retval += "HDD Utilization: "
								+ myToString(r.getResultDistribution())
								+ " on " + myToString(r.getStorageResource())
								+ newline;
					}
				}
			}
		}

		if (q == Qualities.THROUGHPUT) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					for (Throughput r : p.getThroughputs()) {
						retval += "Throughput: "
								+ myToString(r.getResultDistribution())
								+ " on " + myToString(r.getUsageScenario())
								+ newline;
					}
				}
			}
		}

		if (q == Qualities.COST) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof MaintainabilityPredictionResult) {
					MaintainabilityPredictionResult m = (MaintainabilityPredictionResult) e
							.getAnalysisResults().get(i);
					retval += "Cost: " + m.getAggregatedCosts() + newline;
					retval += "Time effort: " + m.getAggregatedTimeEffort()
							+ newline;
				}
			}
		}

		if (q == Qualities.RELIABILITY) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof ReliabilityPredictionResult) {
					ReliabilityPredictionResult r = (ReliabilityPredictionResult) e
							.getAnalysisResults().get(i);
					for (Reliability rel : r.getReliabilities()) {
						retval += "Reliability: "
								+ myToString(rel.getResultDistribution())
								+ " on " + myToString(rel.getSystemCall())
								+ "\n";
					}
				}
			}
		}
		return retval;
	}

	private static String myToString(StorageResource storageResource) {
		return storageResource.getName();
	}

	public static String myToString(ExecutionResource executionResource) {
		String str = executionResource.toString();
		return str.substring(str.indexOf(' ') + 1);
	}

	public static String myToString(ResultDistribution rd) {
		String str = rd.toString();
		return str.substring(str.indexOf(' ') + 1);
	}

	public static String myToString(SystemCall systemCall) {
		return systemCall.getName();
	}

	public static String myToString(UsageScenario usageScenario) {
		return usageScenario.getName();
	}

	public static String getValueForQuality(AlternativeEvaluation e, Qualities q) {
		String retval = "";

		if (q == Qualities.UTILIZATION) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					double total = 0;
					int count = 0;
					for (CpuResourceUtilization cru : p
							.getResourceUtilizations()) {
						total += cru.getResultDistribution()
								.getArithmeticMean();
						count++;
					}
					retval += String.format("; Utilization: %.2f", total
							/ count);
				}
			}
		}

		if (q == Qualities.RESPONSETIME) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					double total = 0;
					int count = 0;
					for (ResponseTime r : p.getResponseTimes()) {
						total += r.getResultDistribution().getArithmeticMean();
						count++;
					}
					retval += String.format("; Response time: %.2f", total
							/ count);
				}
			}
		}

		if (q == Qualities.HDDUTILIZATION) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					double total = 0;
					int count = 0;
					for (HddResourceUtilization r : p.getHddResourceUtilizations()) {
						total += r.getResultDistribution().getArithmeticMean();
						count++;
					}
					retval += String.format("; HDD Utilization: %.2f", total
							/ count);
				}
			}
		}

		if (q == Qualities.THROUGHPUT) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof PerformancePredictionResult) {
					PerformancePredictionResult p = (PerformancePredictionResult) e
							.getAnalysisResults().get(i);
					double total = 0;
					int count = 0;
					for (Throughput r : p.getThroughputs()) {
						total += r.getResultDistribution().getArithmeticMean();
						count++;
					}
					retval += String.format("; Throughput: %.2f", total
							/ count);
				}
			}
		}

		if (q == Qualities.COST) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof MaintainabilityPredictionResult) {
					MaintainabilityPredictionResult m = (MaintainabilityPredictionResult) e
							.getAnalysisResults().get(i);
					retval += "; Cost: " + m.getAggregatedCosts()
							+ ";Time effort: " + m.getAggregatedTimeEffort();
				}
			}
		}

		if (q == Qualities.RELIABILITY) {
			for (int i = 0; i < e.getAnalysisResults().size(); i++) {
				if (e.getAnalysisResults().get(i) instanceof ReliabilityPredictionResult) {
					ReliabilityPredictionResult r = (ReliabilityPredictionResult) e
							.getAnalysisResults().get(i);
					double total = 0;
					int count = 0;
					for (Reliability rel : r.getReliabilities()) {
						total += rel.getResultDistribution()
								.getArithmeticMean();
						count++;
					}
					retval += String.format("; Reliability: %g", total
							/ count);
				}
			}
		}
		return retval;
	}

	class RadioListener implements SelectionListener {

		private int quality, value, alt1, alt2;
		private Matrix[] valueMatrix;

		public RadioListener(Matrix[] m, int quality, int alt1, int alt2,
				int value) {
			super();
			this.alt1 = alt1;
			this.alt2 = alt2;
			this.value = value;
			valueMatrix = m;
			this.quality = quality;
		}

		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);

		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			valueMatrix[quality].set(alt2, alt1, translate[value]);
			valueMatrix[quality].set(alt1, alt2, 1 / translate[value]);
			settings[quality][alt1*numEvals+alt2] = value;
		}

	}
}