package org.ow2.dsrg.fm.qabstractor.utils;

import java.util.Vector;

import de.fzi.gast.core.Package;
import de.fzi.gast.core.Root;
import de.fzi.gast.functions.Method;
import de.fzi.gast.types.GASTClass;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Logger;

/**
 * Utility class for finding classes, methods, ancestors or inner classes.
 * @author Josef Reidinger
 */
public class ClassFinder {

    /**
     * Find class in GAST tree by its fully qualified name.
     * @param root of gast tree
     * @param name of class
     * @return found class or null
     */
    public static GASTClass getClassByQualifiedName(Root root, String name) {
        boolean next = true;
        String pack = name;
        Vector<String> classHierarchy = new Vector<String>();
        while (next) {
            classHierarchy.add(0, pack.substring(pack.lastIndexOf(".") + 1));
            int lastindex = pack.lastIndexOf(".");
            if (lastindex == -1) {
                pack = "";
                next=false;
            } else {
                pack = pack.substring(0, lastindex);
            }
            Package p = root.getPackageByQualifiedName(pack);
            if (p != null) {
                for (GASTClass c : p.getClasses()) {
                    String topClass = classHierarchy.firstElement();
                    if (topClass.equals(c.getSimpleName())) {
                        classHierarchy.remove(0);
                        GASTClass ret = c;
                        while (!classHierarchy.isEmpty()) {
                            topClass = classHierarchy.firstElement();
                            boolean finded = false;
                            for (GASTClass inner : ret.getInnerClasses()) {
                                if (inner.getSimpleName().equals(topClass)) {
                                    classHierarchy.remove(0);
                                    ret = inner;
                                    finded = true;
                                    break;
                                }
                            }
                            if (!finded){
                                Logger.getLogger("ClassFinder").severe("Doesn't find class "+name);
                                return null;
                            }
                        }
                        if (classHierarchy.isEmpty()) {
                            return ret;
                        }
                    }
                }
            }
        }
        return null;
    }

    /**
     * Finds method in gast tree by its fully qualified name.
     * @param root of gast tree
     * @param name of method in format classFullyQualifiedName.methodName
     * @return
     */
    public static Method findMethod(Root root, String name){
        Method ret = null;
        String methodName = name.substring(name.lastIndexOf(".")+1);
        String className = name.substring(0,name.lastIndexOf("."));
        GASTClass cl = ClassFinder.getClassByQualifiedName(root, className);
        if (cl != null) {
            for (Method m : cl.getMethods()) {
                if (m.getSimpleName().equals(methodName)) {
                    ret = m;
                }
            }
        }
        return ret;
    }

    /**
     * Returns recursively all inner classes including given class.
     * @param clas to process
     * @return Set of inner classes including given class
     */
    public static Set<GASTClass> getAllInner(GASTClass clas){
        Set<GASTClass> ret = new TreeSet<GASTClass>(new ClassComparator());
        Vector<GASTClass> toProcess = new Vector<GASTClass>();
        toProcess.add(clas);
        do {
            GASTClass processed = toProcess.lastElement();
            ret.add(processed);
            toProcess.addAll(processed.getInnerClasses());
            toProcess.remove(processed);
        } while (!toProcess.isEmpty());
        return ret;
    }

    /**
     * Returns all ancestors of given class including itself.
     * @param clas to process
     * @return set of ancestors including given class
     */
    public static Set<GASTClass> getAllAncestors(GASTClass clas){
        Set<GASTClass> ret = new TreeSet<GASTClass>(new ClassComparator());
        Vector<GASTClass> toProcess = new Vector<GASTClass>();
        toProcess.add(clas);
        do {
            GASTClass processed = toProcess.lastElement();
            ret.add(processed);
            toProcess.addAll(processed.getSuperTypes());
            toProcess.remove(processed);
        } while (!toProcess.isEmpty());
        return ret;
    }

    private ClassFinder() {
    }
}
