package org.ow2.dsrg.fm.qabstractor.transformation;

import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.ow2.dsrg.fm.qabstractor.extract.MetadataExtractor;
import org.ow2.dsrg.fm.qabstractor.utils.GastCopier;

import de.fzi.gast.functions.Method;
import de.fzi.gast.statements.BlockStatement;
import de.fzi.gast.statements.Branch;
import de.fzi.gast.statements.BranchStatement;
import de.fzi.gast.statements.CatchBlock;
import de.fzi.gast.statements.ExceptionHandler;
import de.fzi.gast.statements.GASTExpression;
import de.fzi.gast.statements.JumpStatement;
import de.fzi.gast.statements.LoopStatement;
import de.fzi.gast.statements.SimpleStatement;
import de.fzi.gast.statements.Statement;
import de.fzi.gast.types.GASTClass;

/**
 * Basic class which provides traversion thrue gast tree. Also provides few
 * basic manipulation methods for gast.
 * @author Josef Reidinger
 */
public abstract class Transformation {

    protected Logger log;
    protected MetadataExtractor extractor;
    protected GastCopier copier = new GastCopier();

    public Transformation(Logger log, MetadataExtractor extractor) {
        this.log = log;
        this.extractor = extractor;
    }

    /**
     * Processes all classes in component.
     * @param componentName to process
     */
    public void processComponent(String componentName) {
        for (GASTClass clas : extractor.getImplClasses(componentName)) {
            processClass(clas);
        }
    }

    /**
     * Processes all methods.
     * Note: Only methods, no constructors.
     * @param clas to process
     */
    public void processClass(GASTClass clas) {
        for (Method m : clas.getMethods()) {
            processMethod(m);
        }
    }

    /**
     * Processes body of method.
     * @param method to process
     */
    protected void processMethod(Method method) {
        if (method.getBody() != null) {
            processBlock(method.getBody());
        }
    }

    /**
     * Processes all statements in block.
     * Note: is it safe to modify block.
     * @param block to process
     */
    protected void processBlock(BlockStatement block) {
        EList<Statement> list = block.getStatements();
        //copy to avoid concurent modification exception if block is modified
        Vector<Statement> proces = new Vector<Statement>(list);
        for (Statement s : proces) {
            processStatement(s);
        }
    }

    /**
     * Processes different kinds of statement.
     * @param statement to process
     */
    protected void processStatement(Statement statement) {
        if (statement instanceof LoopStatement) {
            processLoop((LoopStatement) statement);
        } else if (statement instanceof BranchStatement) {
            processBranch((BranchStatement) statement);
        } else if (statement instanceof BlockStatement) {
            processBlock((BlockStatement) statement);
        } else if (statement instanceof SimpleStatement) {
            processSimpleStatement((SimpleStatement) statement);
        } else if (statement instanceof JumpStatement) {
            processJumpStatement((JumpStatement) statement);
        } else if (statement instanceof ExceptionHandler) {
            processExceptionHandler((ExceptionHandler) statement);
        } else if (statement instanceof CatchBlock) {
            processCatchBlock((CatchBlock) statement);
        } else {
            throw new UnsupportedOperationException("unknown statement" + statement.getClass().toString());
        }
    }

    /**
     * Do nothing. Ready to overwrite.
     * @param jumpStatement to process
     */
    protected void processJumpStatement(JumpStatement jumpStatement) {
    }

    /**
     * Do nothing. Ready to overwrite.
     * @param simpleStatement to process
     */
    protected void processSimpleStatement(SimpleStatement simpleStatement) {
    }

    /**
     * Processes statements in all branches.
     * @param branchStatement to procees
     */
    protected void processBranch(BranchStatement branchStatement) {
        for (Branch branch : branchStatement.getBranches()) {
            processStatement(branch.getStatement());            
        }
    }

    /**
     * Processes body of loop.
     * @param loopStatement to process
     */
    protected void processLoop(LoopStatement loopStatement) {
        processStatement(loopStatement.getBody());
    }

    /**
     * Removes statement from gast tree.
     * @param toRemove statement to remove
     */
    protected void removeStatement(Statement toRemove) {
        replaceStatement(toRemove, null);
    }

    /**
     * Replace one statement with new one
     * @param old statement to replace
     * @param neww new statement, if statement is null then only remove old statement
     */
    protected void replaceStatement(Statement old, Statement neww) {
        if (neww != null) {
            log.debug("replace: " + old.getId() + " by " + neww.getId());
        } else {
            log.debug("remove: " + old.getId());
        }
        EObject outer = old.eContainer(); //EObject because BranchImpl is not statement
        if (outer == null) {
            //TODO access function and replace statement
        } else if (outer instanceof BlockStatement) {
            BlockStatement bl = (BlockStatement) outer;
            int pos = bl.getStatements().indexOf(old);
            bl.getStatements().remove(pos);
            if (neww != null) {
                bl.getStatements().add(pos, neww);
            }
        } else if (outer instanceof LoopStatement) {
            LoopStatement bl = (LoopStatement) outer;
            if (neww == null) {
                removeStatement(bl);
            } else {
                bl.setBody(neww);
            }
        } else if (outer instanceof Branch) {
            Branch bl = (Branch) outer;
            bl.setStatement(neww);            
        } else if (outer instanceof CatchBlock) {
            CatchBlock cb = (CatchBlock) outer;
            if (neww == null) {
                cb.getStatements().remove(old);
            } else {
                int pos = cb.getStatements().indexOf(old);
                cb.getStatements().add(pos, neww);
                cb.getStatements().remove(old);
            }
        }

    }

    /**
     * Creates deep copy of statement
     * @param orig statement to copy
     * @return newly created statement
     */
    protected Statement cloneStatement(Statement orig) {
        Statement ret = (Statement) copier.copy(orig);
        return ret;
    }

    /**
     * Creates deep copy of expression.
     * @param orig expression to copy
     * @return newly created expression
     */
    protected GASTExpression cloneExpression(GASTExpression orig) {
        GASTExpression ret = (GASTExpression) copier.copy(orig);
        return ret;
    }

    /**
     * Do nothing. Ready to overwrite.
     * @param catchBlock to process
     */
    protected void processCatchBlock(CatchBlock catchBlock) {
    }

    /**
     * Processes guarded block, all exception handlers and finally block.
     * @param exceptionHandler to process
     */
    protected void processExceptionHandler(ExceptionHandler exceptionHandler) {
        processBlock((exceptionHandler).getGuardedBlock());
        if (exceptionHandler.getCatchBlocks() != null) {
            for (CatchBlock cb : exceptionHandler.getCatchBlocks()) {
                Vector<Statement> proces = new Vector<Statement>(cb.getStatements());
                for (Statement s : proces) {
                    processStatement(s);
                }
            }
        }
        if (exceptionHandler.getFinallyBlock() != null && exceptionHandler.getFinallyBlock().getStatements() != null && !exceptionHandler.getFinallyBlock().getStatements().isEmpty()) {
            processBlock((exceptionHandler).getFinallyBlock());
        }
    }
}
