/**
 * 
 */
package org.ow2.dsrg.fm.qabstractor.transformation;

import java.util.LinkedList;
import java.util.List;
import java.util.TreeMap;

import org.eclipse.emf.common.util.EList;
import org.ow2.dsrg.fm.qabstractor.Transformer;
import org.ow2.dsrg.fm.qabstractor.extract.MetadataExtractor;
import org.ow2.dsrg.fm.tbplib.node.TBPAccept;
import org.ow2.dsrg.fm.tbplib.node.TBPAlternative;
import org.ow2.dsrg.fm.tbplib.node.TBPCondition;
import org.ow2.dsrg.fm.tbplib.node.TBPEmit;
import org.ow2.dsrg.fm.tbplib.node.TBPIf;
import org.ow2.dsrg.fm.tbplib.node.TBPImperativeNode;
import org.ow2.dsrg.fm.tbplib.node.TBPImperativeNull;
import org.ow2.dsrg.fm.tbplib.node.TBPImperativeSequence;
import org.ow2.dsrg.fm.tbplib.node.TBPMethodDeclaration;
import org.ow2.dsrg.fm.tbplib.node.TBPProvisionContainerNode;
import org.ow2.dsrg.fm.tbplib.node.TBPProvisionNode;
import org.ow2.dsrg.fm.tbplib.node.TBPRepetition;
import org.ow2.dsrg.fm.tbplib.node.TBPSpecification;
import org.ow2.dsrg.fm.tbplib.node.TBPSwitch;
import org.ow2.dsrg.fm.tbplib.node.TBPThreadContainerNode;
import org.ow2.dsrg.fm.tbplib.node.TBPVariableDefinition;
import org.ow2.dsrg.fm.tbplib.node.TBPWhile;
import org.ow2.dsrg.fm.tbplib.reference.MethodCall;
import org.ow2.dsrg.fm.tbplib.reference.MethodSignature;

import de.fzi.gast.accesses.BaseAccess;
import de.fzi.gast.accesses.FunctionAccess;
import de.fzi.gast.accesses.VariableAccess;
import de.fzi.gast.expressions.FunctionCall;
import de.fzi.gast.expressions.MemberAccessor;
import de.fzi.gast.functions.Method;
import de.fzi.gast.statements.BlockStatement;
import de.fzi.gast.statements.Branch;
import de.fzi.gast.statements.BranchStatement;
import de.fzi.gast.statements.GASTExpression;
import de.fzi.gast.statements.JumpStatement;
import de.fzi.gast.statements.LoopStatement;
import de.fzi.gast.statements.SimpleStatement;
import de.fzi.gast.statements.Statement;
import de.fzi.gast.types.GASTClass;
import eu.qimpress.sourcecodedecorator.InterfaceSourceCodeLink;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * Converts simplified GAST to TBP.
 * @author Josef Reidinger
 */
public class ToTBP {

    private MetadataExtractor extr;
    private String component;

    /**
     * Creates component specification for specified component.
     * @param extr extractor which contain information about component system
     *          and also connection to simplified gast
     * @param component name of compotent
     * @return specification of component
     */
    public TBPSpecification<String> createComponentSpecification(MetadataExtractor extr, String component, SourceCodeDecoratorRepository scdrep) {
    	String provIface;
    	String implProvIface;
    	
        this.extr = extr;
        this.component = component;
        GASTClass clas = extr.getExtractedClass(component);
        List<TBPMethodDeclaration<String>> reactions = new LinkedList<TBPMethodDeclaration<String>>();
        List<TBPProvisionContainerNode<String>> provisions = new LinkedList<TBPProvisionContainerNode<String>>();
        TBPProvisionNode<String> prov = null;
        List<TBPThreadContainerNode<String>> threads = new LinkedList<TBPThreadContainerNode<String>>();
        for (Method m : clas.getMethods()) {
            TBPImperativeNode<String> code = parseStatement(m.getBody());
        	if (code == null){
        		code = new TBPImperativeNull<String>();
        	}            	

            if (m.getSimpleName().matches("Thread_.*_run")) {
                threads.add(new TBPThreadContainerNode<String>(m.getSimpleName(), code,null));
            } else {            	
            	provIface = extr.getInterfaceForProvidedMethod(component, m);
            	implProvIface = null;
            	
            	for (InterfaceSourceCodeLink link : scdrep.getInterfaceSourceCodeLink()) {
            		if (link.getInterface().getName().equals(provIface)) {
            			implProvIface = link.getGastClass().getQualifiedName();
            			break;
            		}
            	}
            	
            	if (implProvIface == null) {
            		return null;
            	}
            	
                reactions.add(new TBPMethodDeclaration<String>(
                		implProvIface,
                        m.getSimpleName(),
                        new MethodSignature<String>(new TreeMap<String, String>(), null),
                        code,
                        null));

                if (prov == null) {
                	
                    prov = new TBPRepetition<String>(
                            new TBPAccept<String>(
                            new MethodCall<String>(
                            implProvIface,
                            //provIface,
                            m.getSimpleName(),
                            new LinkedList<String>()),
                            null));
                } else {
                    prov = new TBPAlternative<String>(
                            prov,
                            new TBPRepetition<String>(
                            new TBPAccept<String>(
                            new MethodCall<String>(
                            implProvIface,
                            //provIface,
                            m.getSimpleName(),
                            new LinkedList<String>()),
                            null)));
                }
            }
        }
        provisions.add(new TBPProvisionContainerNode<String>(
                prov, Transformer.fixName(clas.getSimpleName())));
        return new TBPSpecification<String>(Transformer.fixName(clas.getQualifiedName()),
                new LinkedList<TBPVariableDefinition<String>>(),
                provisions, reactions, threads);
    }

    /**
     * Transform gast statements to TBP nodes
     * @param statement to transform
     * @return created TBP node
     */
    private TBPImperativeNode<String> parseStatement(Statement statement) {
        if (statement instanceof LoopStatement) {
            return parseLoop((LoopStatement) statement);
        } else if (statement instanceof JumpStatement) {
            return parseJump((JumpStatement) statement);
        } else if (statement instanceof BlockStatement) {
            BlockStatement block = (BlockStatement) statement;
            TBPImperativeNode<String> prev = null;
            /*for (int i = 0; i < block.getStatements().size(); ++i) {
                if (i == 0) {
                    prev = parseStatement(block.getStatements().get(i));
                } else {
                    prev = new TBPImperativeSequence<String>(prev, parseStatement(block.getStatements().get(i)));
                }
            }*/
            for (Statement s: block.getStatements()){
            	if (prev==null){
            		prev = parseStatement(s);            		
            	}else {
            		TBPImperativeNode<String> next = parseStatement(s);
            		if (next!=null){
            			prev = new TBPImperativeSequence<String>(prev, next);            			
            		}
            	}
            }
            return prev;
        } else if (statement instanceof SimpleStatement) {
            if (((SimpleStatement) statement).getExpression() != null) {
                return parseExpression(((SimpleStatement) statement).getExpression());
            } else {
                return parseFunctionaccess(statement.getAccesses());
            }

        } else if (statement instanceof BranchStatement) {
            BranchStatement br = (BranchStatement) statement;
            TBPImperativeNode<String> ret = null;
            if (br.getBranches().size() == 1) { //only one branch


                Statement s = br.getBranches().iterator().next().getStatement();

                TBPImperativeNode<String> tmps = parseStatement(s);
                assert tmps != null : "Previous transformation failed: Problem during creating statement";
                ret = new TBPIf<String>(new TBPCondition<String>(), tmps, null);
            } else {
                List<TBPImperativeNode<String>> branches = new LinkedList<TBPImperativeNode<String>>();
                for (Branch branch : br.getBranches()) {
                    Statement s = branch.getStatement();
                    TBPImperativeNode<String> tmps = parseStatement(s);
                    assert tmps != null : "Previous transformation failed: Problem during creating statement";
                    branches.add(tmps);
                }
                ret = new TBPSwitch<String>(branches);
            }
            return ret;
        } else if (statement == null) {
        	return new TBPImperativeNull<String>();        
        } else {
            throw new UnsupportedOperationException("unknown statement");
        }
    }

    /**
     * Converts method call to TBP Emit
     * @param accesses to convert
     * @return TBP node
     */
    private TBPImperativeNode<String> parseFunctionaccess(EList<BaseAccess> accesses) {
        Method method = null;
        String requiredIface = null;
        for (BaseAccess a : accesses) {
            if (a instanceof FunctionAccess) {
                method = (Method) ((FunctionAccess) a).getTargetFunction();
            }
            else if (a instanceof VariableAccess) {
            	requiredIface = ((VariableAccess)a).getTargetVariable().getSimpleName();
            }
        }
        
        if (requiredIface == null) {
        	return new TBPImperativeNull<String>();
        }
        
        //MethodCall<String> call = new MethodCall<String>(extr.getInterfaceForRequiredMethod(component, method), method.getSimpleName(), new LinkedList<String>());
        MethodCall<String> call = new MethodCall<String>(requiredIface, method.getSimpleName(), new LinkedList<String>());
        return new TBPEmit<String>(call);
    }

    /**
     * transforms gast loop to TBP loop
     * @param loopStatement to convert
     * @return TBP node
     */
    private TBPImperativeNode<String> parseLoop(LoopStatement loopStatement) {
        
        TBPImperativeNode<String> body = parseStatement(loopStatement.getBody());
        return new TBPWhile<String>(new TBPCondition<String>(), body); 
    }

    /**
     * Convert expression to TBP node. In this phase of trasformation only
     * method call could be in expression. If another type of expression occur,
     * then UnsuporttedOperationalException is throwed.
     * @param expression to convert
     * @return tbp node
     */
    private TBPImperativeNode<String> parseExpression(GASTExpression expression) {
        if (expression instanceof MemberAccessor) {
            MemberAccessor ma = (MemberAccessor) expression;
            return parseFunctionCall((FunctionCall) ma.getRight());
        }
        throw new UnsupportedOperationException("unsuported simple statement");
    }

    /**
     * Converts method call to TBP Emit.
     * Note it is similar to parseFunctionAccess. This one is used when gast
     * contains expressions.
     * @param functionCall to convert
     * @return TBP node
     */
    private TBPEmit<String> parseFunctionCall(FunctionCall functionCall) {
        String iface = extr.getInterfaceForRequiredMethod(component, (Method) functionCall.getFunction().getTargetFunction());
        String method = functionCall.getFunction().getTargetFunction().getSimpleName();
        MethodCall<String> call = new MethodCall<String>(iface, method, new LinkedList<String>()); //TODO ideal is get from declaration
        return new TBPEmit<String>(call);
    }

    /**
     * checks if no jump is in gast and if a jump is found then throw
     * UnsupportedOperationException
     * Note: it is present here for future extension, because TBP with data can
     * have return statements.
     * @param jumpStatement  to check
     * @return never return anything
     */
    private TBPImperativeNode<String> parseJump(JumpStatement jumpStatement) {
        switch (jumpStatement.getKind()) {
            case RETURN:
            case JUMP:
            case THROW:
            default:
                throw new UnsupportedOperationException("unsuported jump");
        }
    }
}
