package org.ow2.dsrg.fm.qabstractor.transformation;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.apache.log4j.Logger;
import org.ow2.dsrg.fm.qabstractor.Transformer;
import org.ow2.dsrg.fm.qabstractor.exception.RecursiveException;
import org.ow2.dsrg.fm.qabstractor.extract.MetadataExtractor;
import org.ow2.dsrg.fm.qabstractor.pointsto.PointsTo;
import org.ow2.dsrg.fm.qabstractor.utils.ClassFinder;
import org.ow2.dsrg.fm.qabstractor.utils.MethodComparator;
import org.ow2.dsrg.fm.qabstractor.utils.MethodUtils;

import de.fzi.gast.accesses.BaseAccess;
import de.fzi.gast.accesses.FunctionAccess;
import de.fzi.gast.accesses.SelfAccess;
import de.fzi.gast.accesses.VariableAccess;
import de.fzi.gast.accesses.impl.accessesFactoryImpl;
import de.fzi.gast.core.Root;
import de.fzi.gast.expressions.FunctionCall;
import de.fzi.gast.expressions.MemberAccessor;
import de.fzi.gast.expressions.TypeReference;
import de.fzi.gast.expressions.Variable;
import de.fzi.gast.expressions.impl.expressionsFactoryImpl;
import de.fzi.gast.functions.Method;
import de.fzi.gast.statements.Branch;
import de.fzi.gast.statements.BranchStatement;
import de.fzi.gast.statements.SimpleStatement;
import de.fzi.gast.statements.Statement;
import de.fzi.gast.statements.statementsFactory;
import de.fzi.gast.statements.impl.statementsFactoryImpl;
import de.fzi.gast.types.GASTClass;
import de.fzi.gast.types.GASTType;
import de.fzi.gast.types.Visibilities;
import de.fzi.gast.types.typesFactory;
import de.fzi.gast.types.impl.typesFactoryImpl;

/**
 * Transformation creates new class where is all provided methods. In its body
 * is replaced all call by its body until there is only required calls. Calls
 * which is not required and goes outside of component is removed.
 * Note if performance occur then cache deep inlining of methods
 * @author Josef Reidinger
 */
public class Inline extends Transformation {

    protected GASTClass result;
    private statementsFactory statementFactory;
    private typesFactory typeFactory;
    private PointsTo pointsTo;
    private Root rootElement;
    private String componentName;
    private List<Collision> collisions = new LinkedList<Collision>();

    public Inline(MetadataExtractor extractor, PointsTo pointsTo, Root root) {
        super(Logger.getLogger(Transformer.class), extractor);
        statementFactory = new statementsFactoryImpl();
        typeFactory = new typesFactoryImpl();
        this.pointsTo = pointsTo;
        rootElement = root;
    }

    /**
     * It differs from standart parseComponent that it creates new class
     * and go only into provided methods instead off all classes.
     * @param componentName to parse
     */
    @Override
    public void processComponent(String componentName) {
        this.componentName = componentName;
        result = typeFactory.createGASTClass();
        result.setSimpleName(componentName);
        super.processComponent(componentName);
        for (Method m : extractor.getProvidedMethods(componentName)) {
            Method nm = (Method) copier.copy(m);
            result.getMethods().add(nm);
            processMethod(nm);
        }
        for (GASTClass cl : extractor.getImplClasses(componentName)) {
            if (isThread(cl)) {
                Method m = getRunMethod(cl);
                if (m == null) {
                    log.warn("Failed to find the run method of the Thread " + cl.getQualifiedName());
                    continue;
                }
                Method nm = (Method) copier.copy(m);
                nm.setSimpleName("Thread_" + cl.getSimpleName() + "_run");
                result.getMethods().add(nm);
                processMethod(nm);
            }
        }
        
        extractor.setExtractedClass(componentName,result);
    }

    /**
     * If simple statement doesn't contain method call, then it is removed.
     * If it contains method call then action depends on that method.
     * If method is required call, then let it be.
     * If method is outside of component, remove it.
     * In remaining case replace method call by its body using points-to
     * analysis. Points-to analysis is not used in three case. The first
     * case is if object is super, then it used first ancestor which contain
     * called method. The second case is private methods, where it is always
     * current class and third case is static methods.
     * @see PointsTo
     * @param simpleStatement to process
     */
    @Override
    protected void processSimpleStatement(SimpleStatement simpleStatement) {
        FunctionAccess fa = null;
        if (simpleStatement.getExpression() != null) {
            if (simpleStatement.getExpression() instanceof MemberAccessor && ((MemberAccessor) simpleStatement.getExpression()).getRight() instanceof FunctionCall) {
                MemberAccessor ma = (MemberAccessor) simpleStatement.getExpression();
                FunctionCall fc = (FunctionCall) ma.getRight();
                fa = fc.getFunction();
            } else if (simpleStatement.getExpression() instanceof FunctionCall){
                FunctionCall fc = (FunctionCall) simpleStatement.getExpression();
                fa = fc.getFunction();
            } else {
                removeStatement(simpleStatement);
                return;
            }
        } else {
            fa = getFunctionAccess(simpleStatement);
            if (fa == null) {
                removeStatement(simpleStatement);
                return;
            }
        }

        String callName = getFullyQualifiedCallName(simpleStatement, fa.getTargetFunction().getSimpleName());
        if (isRequired(callName)) {
            Statement n = cloneStatement(simpleStatement);
            replaceStatement(simpleStatement, n);
            //remember, that inlined function contain required call
            if (!collisions.isEmpty()){
                collisions.get(collisions.size()-1).setContainRequired(true);
            }
            return; // finded require call
        } else if (!isInCompomenent(fa)) {
            // remove
            removeStatement(simpleStatement);
            return;
        }
        Set<String> realFunctionsNames = getRuntimeclasses(simpleStatement);
        Set<Method> realFunctions = new TreeSet<Method>(new MethodComparator());

        if (((Method) fa.getTargetFunction()).getVisibility().equals(
                Visibilities.VISIBILITYPRIVAT)) {
            realFunctions.add((Method) fa.getTargetFunction());
        } else {
            for (String str : realFunctionsNames) {
                GASTClass clas = ClassFinder.getClassByQualifiedName(
                        rootElement, str);
                if (clas == null) {
                    continue;
                }
                for (Method m : clas.getMethods()) {
                    if (m.getSimpleName().equals(
                            fa.getTargetFunction().getSimpleName()) && m.getFormalParameters().equals(
                            fa.getTargetFunction().getFormalParameters())) {
                        realFunctions.add(m);
                    }
                }
            }
        }
        //specifies methods which is called recursive but doesn't contain 
        //required call, so it should not be inlined again
        List<Method> toRemove = new LinkedList<Method>();
        for (Method m : realFunctions) {
            for (Collision c : collisions){
                if (c.getMethodName().equals(MethodUtils.getFullyQualfiedName(m))){
                    for (Collision cc : collisions.subList(collisions.indexOf(c), collisions.size())){
                        if (cc.containRequired){
                            throw new RecursiveException(m.getPosition());
                        }
                    }
                    toRemove.add(m); //not contain required, so only remove from inlining
                }
            }
        }
        realFunctions.removeAll(toRemove);
        Statement n = null;
        assert !realFunctions.isEmpty() : "No function to inline after resolving virtual functions";
        if (realFunctions.size() == 1) {
            Method m = (Method) realFunctions.iterator().next();
            if (m.getBody() != null) { // unreachable code
                n = cloneStatement(m.getBody());
                collisions.add(new Collision(MethodUtils.getFullyQualfiedName(m)));
                processStatement(n);
                collisions.remove(collisions.size()-1);
            }
        } else {
            BranchStatement ret = statementFactory.createBranchStatement();
            for (Method m : realFunctions) {
                if (m.getBody() != null) {
                    collisions.add(new Collision(MethodUtils.getFullyQualfiedName(m)));
                    Branch br = statementFactory.createBranch();
                    br.setStatement(cloneStatement(m.getBody()));
                    ret.getBranches().add(br);
                    processStatement(br.getStatement());
                    collisions.remove(collisions.size()-1); //remove last
                }
            }
            n = ret;
        }
        replaceStatement(simpleStatement, n);
    }

    /**
     * helper that recognize if method is in component
     * @param callName fully qualilified method name in format
     *          classFullyqualifiedname.methodsimplename
     * @return true if method is requred
     */
    protected boolean isRequired(String callName) {
        for (Method m : extractor.getRequiredMethods(componentName)) {
        	String method = m.getSurroundingClass().getQualifiedName();
        	method += "." + m.getSimpleName();
            if (method.equals(callName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * helper that adds to call also fully qualified class name
     * @param s to find variable type
     * @param call to append after fully qualified type name
     * @return
     */
    private String getFullyQualifiedCallName(SimpleStatement s, String call) {
        GASTType gp = null;
        if (s.getExpression() != null) {
            if (s.getExpression() instanceof MemberAccessor) {
                MemberAccessor ma = (MemberAccessor) s.getExpression();
                // TODO check if left is variable
                if (ma.getLeft() instanceof Variable) {
                    gp = ((Variable) ma.getLeft()).getVariableAccess().getTargetVariable().getType();
                } else if (ma.getLeft() instanceof TypeReference) // call of
                // static
                // method
                {
                    gp = ((TypeReference) ma.getLeft()).getTypeaccess().getTargetType();
                } else {
                    throw new UnsupportedOperationException(
                            "unsupported type in Member Accessor");
                }
            }
        } else {
            // get last variable access, as accesses could be more

            for (BaseAccess a : s.getAccesses()) {
                if (a instanceof VariableAccess) {
                    gp = ((VariableAccess) a).getTargetVariable().getType();
                }
            }
        }
        if (gp != null) {
            String ret = gp.getQualifiedName();// .replace("$", "."); //FIXME
            // stupid gast qualified names
            return ret + "." + call;
        }
        assert false : "No variable access"; // TODO maybe for global methods???
        // not allowed in java
        return "call";
    }

    /**
     * Helper for points-to analysis interface
     * @param s to resolve
     * @return set of fully qualified names of possible runtime classes
     */
    private Set<String> getRuntimeclasses(SimpleStatement s) {
        if (s.getExpression() != null) { // gast with expressions
            if (!(s.getExpression() instanceof MemberAccessor || s.getExpression() instanceof FunctionCall)) {
                throw new UnsupportedOperationException(
                        "In this phase of transformation only MemAccessor should be presented");
            }
            //FIXME GAST should do this automatic
            if (s.getExpression() instanceof FunctionCall){
                SimpleStatement ns = statementFactory.createSimpleStatement();
                MemberAccessor ma = expressionsFactoryImpl.eINSTANCE.createMemberAccessor();
                ma.setRight((FunctionCall)s.getExpression());
                Variable va = expressionsFactoryImpl.eINSTANCE.createVariable();
                SelfAccess sa = accessesFactoryImpl.eINSTANCE.createSelfAccess();
                sa.setAccessedTarget(((FunctionCall)ma.getRight()).getFunction().getAccessedClass().getSelf());
                va.setVariableAccess(sa);
                ma.setLeft(va);
                ns.setExpression(ma);
                s = ns;
            }
            // if variable is super, then it is not recurse, and return direct
            // ancestor
            MemberAccessor ma = (MemberAccessor) s.getExpression();
            if (ma.getLeft() instanceof Variable) {
                Variable v = (Variable) ma.getLeft();
                if (v.getVariableAccess() instanceof SelfAccess) {
                    if (((SelfAccess) v.getVariableAccess()).isSuper()) {
                        Set<String> ret = new TreeSet<String>();
                        ret.add(v.getVariableAccess().getTargetVariable().getType().getQualifiedName().replace("$", "."));
                        return ret;
                    }
                }
                return pointsTo.getRuntimeClass(((MemberAccessor) s.getExpression()));
            } else if (ma.getLeft() instanceof TypeReference) {
                // static functions doesn't need points-to analysis
                Set<String> ret = new TreeSet<String>();
                ret.add(((TypeReference) (ma.getLeft())).getTypeaccess().getTargetType().getQualifiedName().replace("$", "."));
                return ret;
            } else {
                throw new UnsupportedOperationException(
                        "Member accesor should have on left side only type or variable");
            }

        } else {
            FunctionAccess fa = null;
            VariableAccess v = null;
            for (BaseAccess a : s.getAccesses()) {
                if (a instanceof VariableAccess) {
                    v = (VariableAccess) a;
                } else if (a instanceof FunctionAccess) {
                    fa = (FunctionAccess) a;
                }
            }
            if (v instanceof SelfAccess && ((SelfAccess) v).isSuper()) { // HACK
                // for
                // super
                // keyword
                Set<String> ret = new TreeSet<String>();
                ret.add(v.getTargetVariable().getType().getQualifiedName().replace("$", "."));
                return ret;
            }
            return pointsTo.getRuntimeClass(fa, v.getTargetVariable());
        }
    }

    /**
     * Tests if method is inside component
     * @param functionAccess to check
     * @return true if method is inside component
     */
    private boolean isInCompomenent(FunctionAccess functionAccess) {
        if (functionAccess.getTargetFunction() instanceof Method) { // process only method,
            // no pure functions or
            // constructors
            GASTClass containedClass = ((Method) functionAccess.getTargetFunction()).getSurroundingClass();
            if (extractor.getImplClasses(componentName).contains(containedClass)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Tests if class can act as thread
     * @param clas to test
     * @return true if class can act as thread
     */
    private boolean isThread(GASTClass clas) {
        for (GASTClass cl : ClassFinder.getAllAncestors(clas)) {
            if (cl.getSimpleName().equals("Thread") || cl.getSimpleName().equals("Runnable")) {
                return true;
            }
        }
        return false;
    }

    /**
     * gets run method of thread
     * @param clas
     * @return Method run or null if method not finded
     */
    private Method getRunMethod(GASTClass clas) {
        for (Method m : clas.getMethods()) {
            if (m.getSimpleName().equals("run")) {
                return m;
            }
        }
        return null; //TODO exception
    }

    /**
     * Gets function access from simple block. Useable only for gast without expressions
     * @param simpleStatement  to process
     * @return found Access or null
     */
    protected FunctionAccess getFunctionAccess(SimpleStatement simpleStatement) {
        for (BaseAccess a : simpleStatement.getAccesses()) {
            if (a instanceof FunctionAccess) {
                return (FunctionAccess) a;
            }
        }

        return null;
    }

    protected class Collision implements Comparable<Collision> {

        private boolean containRequired;
        private String methodName;


        public Collision(String name) {
            methodName = name;
            containRequired = false;
        }

        @Override
        public int compareTo(Collision o) {
            return methodName.compareTo(o.toString());
        }

        public boolean containRequired() {
            return containRequired;
        }

        public void setContainRequired(boolean containRequired) {
            this.containRequired = containRequired;
        }


        public String getMethodName() {
            return methodName;
        }

        
    }
}
