package org.ow2.dsrg.fm.qabstractor.transformation;

import java.util.Vector;

import org.apache.log4j.Logger;
import org.ow2.dsrg.fm.qabstractor.Transformer;
import org.ow2.dsrg.fm.qabstractor.exception.RequireInCatchException;
import org.ow2.dsrg.fm.qabstractor.extract.MetadataExtractor;

import de.fzi.gast.statements.BlockStatement;
import de.fzi.gast.statements.Branch;
import de.fzi.gast.statements.BranchStatement;
import de.fzi.gast.statements.CatchBlock;
import de.fzi.gast.statements.ExceptionHandler;
import de.fzi.gast.statements.JumpStatement;
import de.fzi.gast.statements.LoopStatement;
import de.fzi.gast.statements.Statement;
import de.fzi.gast.statements.statementsFactory;

/**
 * Transformation that cleans dead code, convert exception handler to one
 * block and removes unnecessary nondeterministic expressions.
 * 
 * @author Josef Reidinger
 * 
 */
public class Clean extends Transformation {

    public Clean(MetadataExtractor extractor) {
        super(Logger.getLogger(Transformer.class), extractor);
    }

    /**
     * Block processed in this transformation is removed if is empty or if
     * block is only statement in another block, then it is redured to one
     * block
     * @param block to process
     */
    @Override
    protected void processBlock(BlockStatement block) {
        super.processBlock(block);
        if (block.getStatements().isEmpty()) {
            // TODO handle of block is inside method
            removeStatement(block);
        }
        if (block.getStatements().size() == 1) {
            Statement s = block.getStatements().iterator().next();
            if (s instanceof BlockStatement) {
                replaceStatement(block, s);
            }
        }

    }

    /**
     * Loop processed in this transformation is removed if contain empty body.
     * Also if loop contain only another loop, then the loops is merged because
     * both iterate over nondeterministic expression.
     * If body contain only branch statement with one branch (conditional
     * execution) then this branch statement is replaced by body its only
     * branch.
     * @param loop to process
     */
    @Override
    protected void processLoop(LoopStatement loop) {
        super.processLoop(loop);
        if (isBodyEmpty(loop.getBody())) {
            log.info("remove: " + loop.getBody().getId());
            removeStatement(loop);
        }
        if (loop.getBody() instanceof LoopStatement) { //clean multiple non-deterministic loop
            replaceStatement(loop, loop.getBody());
        }
        if (loop.getBody() instanceof BlockStatement) {
            BlockStatement body = (BlockStatement) loop.getBody();
            if (body.getStatements().size() == 1 &&
                    body.getStatements().iterator().next() instanceof LoopStatement) {
                replaceStatement(loop, body.getStatements().iterator().next());
            }
            if (body.getStatements().size() == 1 &&
                    body.getStatements().iterator().next() instanceof BranchStatement) {
                BranchStatement br = (BranchStatement) body.getStatements().iterator().next();
                if (br.getBranches().size() == 1) { //nondeterministic branch with one statement, can be removed
                    loop.setBody(br.getBranches().iterator().next().getStatement());
                }
            }
        }
    }

    /**
     * Clean transformation removes all jump statement. Jumps which affects
     * workflow is transformed in previous transformation.
     * @param s
     */
    @Override
    protected void processJumpStatement(JumpStatement s) {
        removeStatement(s);
    }

    /**
     * Clean transformation removes all empty branches and if final branch
     * statement doesn't contain any branches, then removes it.
     * It also replaces branch statement which has only one branch which contains
     * only one branch statement or one loop statent by that inside branch
     * statement because all branch and loop statements decision is based on
     * nondeterministic expression.
     * @param branch to process
     */

    @Override
    protected void processBranch(BranchStatement branch) {
        super.processBranch(branch);
        Vector<Branch> toRemove = new Vector<Branch>();
        for (Branch br : branch.getBranches()) {
            if (br.getStatement() == null) {
                toRemove.add(br);
            }
        }

        for (Branch b : toRemove) {
            branch.getBranches().remove(b);
        }


        if (branch.getBranches().isEmpty()) {
            removeStatement(branch);
        }

        // remove multiple nondeterministic if or inside while
        if (branch.getBranches().size() == 1) {
            Branch bra = branch.getBranches().iterator().next();

            boolean cont = false;
            Statement s = bra.getStatement();
            do {
                cont = false;

                if (s instanceof BranchStatement || s instanceof LoopStatement) {
                    replaceStatement(branch, s);
                    break;
                }
                if (s instanceof BlockStatement && ((BlockStatement) s).getStatements().size() == 1) {
                    s = ((BlockStatement) s).getStatements().iterator().next();
                    cont = true;
                }
            } while (cont);

        }
    }

    /**
     * convert Exception handler to block which contains guarded block together
     * with finally block. It also checks if any of catch block doesn't contain
     * any required call.
     * @param exceptionHandler to process
     */
    @Override
    protected void processExceptionHandler(ExceptionHandler exceptionHandler) {
        super.processExceptionHandler(exceptionHandler);
        BlockStatement bl = statementsFactory.eINSTANCE.createBlockStatement();
        bl.getStatements().add(exceptionHandler.getGuardedBlock());
        if (exceptionHandler.getFinallyBlock() != null && exceptionHandler.getFinallyBlock().getStatements() != null && !exceptionHandler.getFinallyBlock().getStatements().isEmpty()) {
            bl.getStatements().add(exceptionHandler.getFinallyBlock());
        }
        if (exceptionHandler.getCatchBlocks() != null) {
            for (CatchBlock cb : exceptionHandler.getCatchBlocks()) {
                if (cb.getStatements().size() > 0) {
                    log.warn("The catch block contains a required call");
                    Statement s = cb.getStatements().get(0);
                    throw new RequireInCatchException(s.getPosition());
                }
            }

        }
        replaceStatement(exceptionHandler, bl);
    }

    /**
     * helper that decide for statement if its body is empty.
     * @param body to check
     * @return true if body is empty
     */
    protected boolean isBodyEmpty(Statement body) {
        if (body == null) {
            return true;
        }
        if (body instanceof BlockStatement) {
            BlockStatement block = (BlockStatement) body;
            return block.getStatements().isEmpty();
        } else if (body instanceof LoopStatement) {
            processLoop((LoopStatement) body);
            return isBodyEmpty(((LoopStatement) body).getBody());
        }
        return false;
    }
}
