/**
 * 
 */
package org.ow2.dsrg.fm.qabstractor.transformation;


import org.apache.log4j.Logger;
import org.ow2.dsrg.fm.qabstractor.Transformer;
import org.ow2.dsrg.fm.qabstractor.extract.MetadataExtractor;

import de.fzi.gast.expressions.AssignmentOperatorExpression;
import de.fzi.gast.expressions.Atom;
import de.fzi.gast.expressions.BooleanOperatorExpression;
import de.fzi.gast.expressions.CastExpression;
import de.fzi.gast.expressions.CompareExpression;
import de.fzi.gast.expressions.Conditional;
import de.fzi.gast.expressions.FunctionCall;
import de.fzi.gast.expressions.MemberAccessor;
import de.fzi.gast.expressions.NotExpression;
import de.fzi.gast.expressions.ProductExpression;
import de.fzi.gast.expressions.TermExpression;
import de.fzi.gast.expressions.TypeReference;
import de.fzi.gast.expressions.UnaryArithmeticExpression;
import de.fzi.gast.expressions.Variable;
import de.fzi.gast.statements.BlockStatement;
import de.fzi.gast.statements.Branch;
import de.fzi.gast.statements.BranchStatement;
import de.fzi.gast.statements.GASTExpression;
import de.fzi.gast.statements.LoopStatement;
import de.fzi.gast.statements.SimpleStatement;
import de.fzi.gast.statements.Statement;
import de.fzi.gast.statements.statementsFactory;
import de.fzi.gast.statements.impl.statementsFactoryImpl;

/**
 * Abstract transformation moves expression to separate simple statement.
 * Complex expression si separate to simple one.
 * e.g.
 * <code>
 *  if (a && b){
 *      ...
 *  }
 *
 * to:
 *  a;
 *  if (nondeterministic)
 *      b;
 *  if (nondeterministic){
 *      ...
 *  }
 * </code>
 *
 * @author Josef Reidinger
 */
public class Abstract extends Transformation {

    private statementsFactory sFactory = new statementsFactoryImpl();

    public Abstract(MetadataExtractor extractor) {
        super(Logger.getLogger(Transformer.class), extractor);
    }

    /**
     * Moves expression from loop to separate simpleStatement
     * @param loopStatement to process
     */
    @Override
    protected void processLoop(LoopStatement loopStatement) {
        super.processLoop(loopStatement);
        Statement st;
        if (loopStatement.getBreakConditionExpression() != null) {
            st = getStatement(loopStatement.getBreakConditionExpression());
        } else {
            st = sFactory.createSimpleStatement();
            st.getAccesses().addAll(loopStatement.getAccesses());
        }
        if (st != null) {
            BlockStatement outBlock = sFactory.createBlockStatement();
            BlockStatement inBlock = sFactory.createBlockStatement();
            inBlock.getStatements().add(loopStatement.getBody());
            inBlock.getStatements().add(cloneStatement(st));
            loopStatement.setBody(inBlock);
            outBlock.getStatements().add(st);
            replaceStatement(loopStatement, outBlock);
            outBlock.getStatements().add(loopStatement);
        }
    }

    /**
     * Moves expression from branchStatement to separate simpleStatements
     * @param branchStatement to process
     */
    @Override
    protected void processBranch(BranchStatement branchStatement) {
        super.processBranch(branchStatement);
        BlockStatement bl = sFactory.createBlockStatement();
        for (Branch br : branchStatement.getBranches()) {
            Statement st = getStatement(br.getConditionExpression()); // FIXME each except first should be in own branch
            if (st!=null){
                bl.getStatements().add(st);
            }
        }
        replaceStatement(branchStatement, bl);
        bl.getStatements().add(branchStatement);
    }

    /**
     * Transform complex expressions in simple statement to simple one
     * @param simpleStatement to proces
     */
    @Override
    protected void processSimpleStatement(SimpleStatement simpleStatement) {
        if (simpleStatement.getExpression() != null) {
            Statement n = getStatement(simpleStatement.getExpression());
            if (n != null) {
                replaceStatement(simpleStatement, n);
            }
        }
    }

    /**
     * helper that creates tree of statement from complex expression.
     * It recursive creates different statements from given expression.
     * @param expression to simplify
     * @return statement which contains simple statement with atomic expressions
     */
    protected Statement getStatement(GASTExpression expression) {
        if (expression == null) {
            return null;
        }

        if (expression instanceof MemberAccessor) {

            return ParseMemberAccesor((MemberAccessor) expression);

        } else if (expression instanceof AssignmentOperatorExpression) {

            //for assigment is important only expressions which is assigned
            return getStatement(((AssignmentOperatorExpression) expression).getRight());

        } else if (expression instanceof Conditional) {

            return ParseConditional((Conditional) expression);

        } else if (expression instanceof BooleanOperatorExpression) {

            return ParseBooleanExpression((BooleanOperatorExpression) expression);

        } else if (expression instanceof UnaryArithmeticExpression) {

            //unary operator is not important for result
            return getStatement(((UnaryArithmeticExpression) expression).getInner());

        } else if (expression instanceof NotExpression) {

            //not operator is not important
            //not and unaryArithmicOperator cannot be merged, as Unary (parent)
            // doesn't have getInner method
            return getStatement(((NotExpression) expression).getInner());

        } else if (expression instanceof CastExpression) {

            //cast is not important for result, so take only affected expression
            return getStatement(((CastExpression) expression).getExpression());

        } else if (expression instanceof TermExpression) {

            return ParseTermExpression((TermExpression) expression);

        } else if (expression instanceof ProductExpression) {

            return ParseProductExpression((ProductExpression) expression);


        } else if (expression instanceof CompareExpression) {

            return ParseCompareExpression((CompareExpression) expression);

        } else if (expression instanceof Atom) {

            // atomic expression is just returned
            SimpleStatement ret = sFactory.createSimpleStatement();
            ret.setExpression(expression);
            return ret;

        } else {

            throw new UnsupportedOperationException("unknown statement");

        }
    }

    /**
     * helper that create expression from MemberAccessor.
     * It creates method call and then recursive get Statement from method
     * arguments
     * @param memberAccessor to parse
     * @return Statement which contain all abstracted statements or null if no
     *          interesting expression finded
     */
    protected Statement ParseMemberAccesor(MemberAccessor memberAccessor) {
        if ((memberAccessor.getLeft() instanceof Variable || memberAccessor.getLeft() instanceof TypeReference) && memberAccessor.getRight() instanceof FunctionCall) {
            SimpleStatement s = sFactory.createSimpleStatement();
            s.setExpression(memberAccessor);
            log.info("create: " + s.getId());
            if (((FunctionCall) memberAccessor.getRight()).getParameterexpression() != null &&
                    ((FunctionCall) memberAccessor.getRight()).getParameterexpression().size() > 0) { //some arguments in function call
                BlockStatement bl = sFactory.createBlockStatement();
                for (GASTExpression ge : ((FunctionCall) memberAccessor.getRight()).getParameterexpression()) {
                    bl.getStatements().add(getStatement(ge));
                }
                bl.getStatements().add(s);
                return bl;
            } else {
                return s;
            }
        } else if (memberAccessor.getLeft() instanceof MemberAccessor) {
            //TODO exception
            return null;
        } else {
            return null;
        }
    }

    /**
     * Parses conditional expression ( ? : ). It creates branch Statement from
     * second and third operand and add after statement which contain first
     * operand.
     * @param conditional to parse
     * @return statement which contain all abstracted statements or null if no
     *          interesting expression finded
     */
    protected Statement ParseConditional(Conditional conditional) {
        Statement cond = getStatement(conditional.getCondition());
        Statement then = getStatement(conditional.getThen());
        Statement other = getStatement(conditional.getElse());

        BranchStatement branchStatement = sFactory.createBranchStatement();
        if (then != null) {
            Branch branchPositive = sFactory.createBranch();
            branchPositive.setStatement(then);
            branchStatement.getBranches().add(branchPositive);
        }

        if (other != null) {
            Branch branchNegative = sFactory.createBranch();
            branchNegative.setStatement(other);
            branchStatement.getBranches().add(branchNegative);
        }

        BlockStatement result = sFactory.createBlockStatement();

        if (cond != null) {
            result.getStatements().add(cond);
        }
        if (!branchStatement.getBranches().isEmpty()) {
            result.getStatements().add(branchStatement);
        }

        return result;
    }

    /**
     * Parse boolean expression. It is important then second argument due to
     * short-circuit evaluation needn't to be executed, so it must be added to
     * conditional execution.
     * @param booleanOperatorExpression to parse
     * @return statement which contain all abstracted statements or null if no
     *          interesting expression finded
     */
    protected Statement ParseBooleanExpression(BooleanOperatorExpression booleanOperatorExpression) {
        Statement result = getStatement(booleanOperatorExpression.getLeft());
        Statement right = getStatement(booleanOperatorExpression.getRight());
        if (right != null) {
            Branch branch = sFactory.createBranch();
            branch.setStatement(right);
            BranchStatement branchStatement = sFactory.createBranchStatement();
            branchStatement.getBranches().add(branch);
            if (result != null) {
                BlockStatement block = sFactory.createBlockStatement();
                block.getStatements().add(result);
                block.getStatements().add(branchStatement);
                result = block;
            } else {
                result = branchStatement;
            }
        }
        return result;
    }

    /**
     * Parses term expression. create left and right expression as separate
     * statements and enclose it to block statement
     * <i>Fixme</i>: Can be merged with parseProductExpression and
     * parseCompareExpression if there is interface which all uses and which
     * guarant getLeft and getRight methods.
     * @param termExpression to parse
     * @return statement which contain all abstracted statements or null if no
     *          interesting expression finded
     */
    protected Statement ParseTermExpression(TermExpression termExpression) {
        Statement left = getStatement(termExpression.getLeft());
        Statement right = getStatement(termExpression.getRight());
        return createFromTwoExpressions(left, right);
    }

    /**
     * Parses Product expression. create left and right expression as separate
     * statements and enclose it to block statement
     * <i>Fixme</i>: Can be merged with parseTermExpression and
     * parseCompareExpression if there is interface which all uses and which
     * guarant getLeft and getRight methods.
     * @param productExpression to parse
     * @return statement which contain all abstracted statements or null if no
     *          interesting expression finded
     */
    protected Statement ParseProductExpression(ProductExpression productExpression) {
        Statement left = getStatement(productExpression.getLeft());
        Statement right = getStatement(productExpression.getRight());
        return createFromTwoExpressions(left, right);
    }

    /**
     * Parses Compare expression. create left and right expression as separate
     * statements and enclose it to block statement
     * <i>Fixme</i>: Can be merged with parseTermExpression and
     * parseProductExpression if there is interface which all uses and which
     * guarant getLeft and getRight methods.
     * @param compareExpression to parse
     * @return statement which contain all abstracted statements or null if no
     *          interesting expression finded
     */
    protected Statement ParseCompareExpression(CompareExpression compareExpression) {
        Statement left = getStatement(compareExpression.getLeft());
        Statement right = getStatement(compareExpression.getRight());
        return createFromTwoExpressions(left, right);
    }

    /**
     * helper that avoid code duplication for expression which unconditionaly
     * execute two expression
     * @param first statement
     * @param second statement
     * @return statement if at least one argument is not null. If both arguments
     *          is null, then return null.
     */
    protected Statement createFromTwoExpressions(Statement first, Statement second) {
        if (first != null && second != null) {
            BlockStatement b = sFactory.createBlockStatement();
            b.getStatements().add(first);
            b.getStatements().add(second);
            log.info("create: " + b.getId());
            return b;
        }

        if (first != null) {
            return first;
        } else if (second != null) {
            return second;
        }
        return null;
    }
}
