package org.ow2.dsrg.fm.qabstractor.extract;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import org.ow2.dsrg.fm.qabstractor.utils.ClassFinder;

import de.fzi.gast.core.Root;
import de.fzi.gast.functions.Method;
import de.fzi.gast.types.GASTClass;

/**
 * Utility class which loads component description from file with predefined
 * format.
 * Invalid lines is skipped (and logged).
 * @author Josef Reidinger
 */
public class SpecFileLoader {

    static private MetadataExtractorImpl result;
    static private String currentComponent;

    private enum ActiveItem {

        COMPONENT,
        IMPLCLASS,
        REQUIRED,
        PROVIDED
    };
    static private ActiveItem activeItem;
    private final static Pattern component = Pattern.compile("^\\s*component\\s*:.*");
    private final static Pattern implclass = Pattern.compile("^\\s*classes\\s*:.*");
    private final static Pattern required = Pattern.compile("^\\s*required\\s*:.*");
    private final static Pattern provided = Pattern.compile("^\\s*provided\\s*:.*");
    /*private static Set<String> components = new TreeSet<String>();
    private static Map<String, Set<GASTClass>> implClasses = new TreeMap<String, Set<GASTClass>>();
    private static Map<String, Map<Method, String>> requiredMethods = new TreeMap<String, Map<Method, String>>();
    private static Map<String, Map<Method, String>> providedMethods = new TreeMap<String, Map<Method, String>>();*/
    private static Root root;

    /**
     * Loads metadataExtractor from file.
     * @param path to file
     * @param gastRoot loaded gastRoot to connect component specification
     *                  with gast tree
     * @return MetadataExtractor which contain information about component system
     * @throws IOException if problems with file occur like nonexist file or
     *                      insufficient permissions
     */
    public static MetadataExtractor loadSpecFile(String path, Root gastRoot) throws IOException {
        root = gastRoot;
        result = new MetadataExtractorImpl();
        FileInputStream fstream = new FileInputStream(path);
        DataInputStream in = new DataInputStream(fstream);
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        String line;
        try {
            for (line = br.readLine();
                    line != null; line = br.readLine()) {
                parseLine(line);
            }
        } finally {
            br.close();
        }
        /*result.setComponents(components);
        result.setImplClasses(implClasses);
        result.setProvidedMethods(providedMethods);
        result.setRequiredMethods(requiredMethods);*/
        return result;
    }

    /**
     * parse single line
     * @param line contain line
     */
    private static void parseLine(String line) {
        line = line.trim();
        if (line.contains(":")) { //metainformation
            parseMeta(line);
        } else {
            parseValue(line);
        }
    }

    /**
     * parse line which contains metadata information
     * @param line to parse
     */
    private static void parseMeta(String line) {
        if (component.matcher(line).matches()) {
            activeItem = ActiveItem.COMPONENT;
        } else if (implclass.matcher(line).matches()) {
            activeItem = ActiveItem.IMPLCLASS;
        } else if (required.matcher(line).matches()) {
            activeItem = ActiveItem.REQUIRED;
        } else if (provided.matcher(line).matches()) {
            activeItem = ActiveItem.PROVIDED;
        } else {
            throw new RuntimeException("unknown element " + line); //TODO exception
        }
        String value = line.substring(line.indexOf(":") + 1).trim();
        if (!value.isEmpty()) {
            parseValue(value);
        }
    }

    /**
     * parse line which contains definition
     * @param line to parse
     */
    private static void parseValue(String line) {
        if (line.isEmpty()) {
            return;
        }
        switch (activeItem) {
            case COMPONENT:
                currentComponent = line;
                /*components.add(line);
                if (!implClasses.containsKey(currentComponent)) {
                    implClasses.put(currentComponent, new TreeSet<GASTClass>(new ClassComparator()));
                }
                if (!requiredMethods.containsKey(currentComponent)) {
                    requiredMethods.put(currentComponent, new TreeMap<Method, String>(new MethodComparator()));
                }
                if (!providedMethods.containsKey(currentComponent)) {
                    providedMethods.put(currentComponent, new TreeMap<Method, String>(new MethodComparator()));
                }*/
                break;
            case IMPLCLASS:
                GASTClass cl = ClassFinder.getClassByQualifiedName(root, line);
                if (cl != null) {
                	result.addImplClass(currentComponent, cl);
                }
                break;
            case PROVIDED:
                if (!line.contains("->")) {
                    Logger.getAnonymousLogger().severe("line doesn't contain provide interface. Skipping");
                    return;
                }
                String[] tmp = line.split("->"); //regex so escape OR
                line = tmp[0].trim();
                Method m = ClassFinder.findMethod(root, line);
                if (m != null) {
                	result.addProvidedMethod(currentComponent, m, tmp[1].trim());                    
                }
                break;
            case REQUIRED:
                if (!line.contains("->")) {
                    Logger.getAnonymousLogger().severe("line doesn't contain require interface. Skipping");
                    return;
                }
                String[] tmp2 = line.split("->"); //regex so escape OR
                line = tmp2[0].trim();
                GASTClass clr = ClassFinder.getClassByQualifiedName(root, line);
                if (clr != null) {
                    for (Method mrc : clr.getMethods()) {
                    	result.addRequiredMethod(currentComponent, mrc, tmp2[1].trim());
                        //requiredMethods.get(currentComponent).put(mrc, tmp2[1].trim());
                    }
                } else {
                    Method mr = ClassFinder.findMethod(root, line);
                    if (mr != null) {
                    	result.addRequiredMethod(currentComponent, mr, tmp2[1].trim());
                        //requiredMethods.get(currentComponent).put(mr, tmp2[1].trim());
                    }
                }
                break;
        }
    }

    private SpecFileLoader() {
    }
}
