package org.ow2.dsrg.fm.qabstractor.extract;

import java.util.Collection;
import java.util.Set;

import de.fzi.gast.functions.Method;
import de.fzi.gast.types.GASTClass;

/**
 * Metadata extractor provides information about component system and its
 * relation to GAST structure. It is abstraction above different kind how to
 * define component.
 * @author Josef Reidinger
 */
public interface MetadataExtractor {

    /**
     * Gets all component in system
     * @return set of components name
     */
    public Set<String> getComponents();

    /**
     * Gets class which implement component with name
     * @param name of component
     * @return collection of classes which implement component
     */
    public Set<GASTClass> getImplClasses(String name);

    /**
     * Sets class which implement component with name
     * Use it when transformation make some change to component
     * @param name of component
     * @param classes which implement component
     */
    public void addImplClasses(String name, Collection<GASTClass> classes);

    /**
     * Gets required methods needed by interface
     * @param name of interface
     * @return required method
     */
    public Set<Method> getRequiredMethods(String name);

    /**
     * Gets provided methods for interface
     * @param name of interface
     * @return provided method
     */
    public Set<Method> getProvidedMethods(String name);

    /**
     * Gets component interface (could be different from GAST Interface)
     * which include given method.
     * If parameters is invalid, then return "interface" or AssertException if
     * assert is enabled.
     * @param component which contain interface
     * @param method of target interface
     * @return String identification of interface or 'thread' if method is part
     *          of thread
     */
    public String getInterfaceForProvidedMethod(String component, Method method);

    /**
     * Gets component interface (could be different from GAST Interface)
     * which include given method.
     * If parameters is invalid, then return "interface" or AssertException if
     * assert is enabled.
     * @param component which contain interface
     * @param method of target interface
     * @return String identification of interface
     */
    public String getInterfaceForRequiredMethod(String component, Method method);
    
    public void setExtractedClass(String componentName, GASTClass c);
    
    public GASTClass getExtractedClass(String componentName);
    
    public boolean isKnown(GASTClass c);
    
}
