package org.ow2.dsrg.fm.qabstractor.extract;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.EList;

import de.fzi.gast.core.File;
import de.fzi.gast.functions.Method;
import de.fzi.gast.types.GASTClass;
import de.fzi.gast.types.GASTType;
import de.fzi.gast.types.Member;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.Interface;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.sourcecodedecorator.FileLevelSourceCodeLink;
import eu.qimpress.sourcecodedecorator.MethodLevelSourceCodeLink;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * Loads component information from source code decorator repository which
 * contains connections between GAST and SOMOX. result is MetadataExtractor
 * @see MetadataExtractor
 * @author Josef Reidinger
 */
public class GastServiceExtractorLoader {

    private SourceCodeDecoratorRepository sourceCodeDecoratorRepository;
    private Set<String> components = new HashSet<String>();
    private Map<String, Set<GASTClass>> implClasses = new HashMap<String, Set<GASTClass>>();
    private Map<String, Map<Method, String>> requiredMethods = new HashMap<String, Map<Method, String>>();
    private Map<String, Map<Method, String>> providedMethods = new HashMap<String, Map<Method, String>>();

    /**
     * Creates new instance based on given argument
     * @param sourceCodeDecoratorRepository source of information about
     *                                  components and its relation to GAST tree
     */
    public GastServiceExtractorLoader(SourceCodeDecoratorRepository sourceCodeDecoratorRepository) {
        this.sourceCodeDecoratorRepository = sourceCodeDecoratorRepository;
    }

    /**
     * Gets metadata extractor
     * @return Metadata extractor which contains information about component
     *                  system
     */
    public MetadataExtractor load() {
        MetadataExtractorImpl result = new MetadataExtractorImpl();
        processComponents(result);
        return result;
    }

    /**
     * Sets to result all finded classes
     * 
     * Goes through the list of FileLevelSourceCode links which is a pair of the component name and Java filename.
     * All classes from the file implements the component. If there are more links for one component, the implementing
     * classes from both files are merged 
     * 
     * 
     * 
     * @param result
     */
    private void processComponents(MetadataExtractorImpl result) {
        components.clear();
        implClasses.clear();
        requiredMethods.clear();
        providedMethods.clear();
        for (FileLevelSourceCodeLink codeLink : sourceCodeDecoratorRepository.getFileLevelSourceCodeLink()) {
            if (codeLink.getFile() == null) { //no file associated => no class implement this component
                continue;
            }

            File f = codeLink.getFile();
            ComponentType ctype = codeLink.getComponentType();
            String componentName =ctype.getName(); 
            components.add(componentName);
            Set<GASTClass> implClasses = new HashSet<GASTClass>();
            for (GASTType clas : f.getTypes()) {
                if (clas instanceof GASTClass) {
                    implClasses.add((GASTClass) clas);
                }
            }
            
            result.addImplClasses(componentName,implClasses);
            processMethods(ctype, result);
        }
        
    }

    /**
     * Fills required and provided method information
     * @param ctype
     */
    private void processMethods(ComponentType ctype, MetadataExtractorImpl result) {
    	String componentName =ctype.getName();
        result.addProvidedMethods(componentName, processInterfacePort(ctype.getProvided(),false));
        result.addRequiredMethods(componentName, processInterfacePort(ctype.getRequired(),true));
        //result.addRequiredMethods(componentName, processRequiredInterfacePort(ctype,true));
    }

    /**
     * process interfaces of component
     * @param ports list of interface to process
     * @return Map of method and corresponding interface
     */
    private Map<Method, String> processInterfacePort(EList<InterfacePort> ports, boolean propagateToOverriden) {
        Map<Method, String> result = new HashMap<Method, String>();
        for (InterfacePort port : ports) {
            Interface i = port.getInterfaceType();
            String iname = i.getName();
            for (Operation o : i.getSignatures()) {
        		Method m = findMethod(o);
            	result.put(m, iname);            	
            	if (propagateToOverriden){
            		Member overriden = null;
            		while ((overriden = m.getOverriddenMember())!=null){
            			m = (Method)overriden;
            			result.put(m, iname);
            		}
            	}
            }
        }
        return result;
    }
    
    /*
    private Map<Method, String> processRequiredInterfacePort(ComponentType ctype, boolean propagateToOverriden) {
    	//here process required interfaces - the names of the variables should be used as "types", not the real types...
    	EList<InterfacePort> ports = ctype.getRequired();
        Map<Method, String> result = new HashMap<Method, String>();
        for (InterfacePort port : ports) {
            Interface i = port.getInterfaceType();
            String iname = i.getName();
            GASTClass implIface = null;
            
            for (InterfaceSourceCodeLink link : sourceCodeDecoratorRepository.getInterfaceSourceCodeLink()) {
            	if (link.getInterface().getName().equals(iname)) {
            		implIface = link.getGastClass();
            		break;
            	}
            }
            
            //here go through gast implementing classes and find an (!!!) appropriate field.
            //Set<GASTClass> classes = implClasses.get(ctype.getName());
            EList<ComponentImplementingClassesLink> classes = sourceCodeDecoratorRepository.getComponentImplementingClassesLink();
            
            OUT: for (ComponentImplementingClassesLink links : classes) {
            	if (links.getComponent().getName().equals(ctype.getName())) {
	            	for (GASTClass c : links.getImplementingClasses()) {
		            	for (Field f : c.getFields()) {
		            		if (f.getType().getQualifiedName().equals(implIface.getQualifiedName())) {
		            			iname = f.getSimpleName();
		            			break OUT;
		            		}
		            	}
	            	}
            	}
            
             }
            
            for (Operation o : i.getSignatures()) {
        		Method m = findMethod(o);
            	result.put(m, iname);            	
            	if (propagateToOverriden){
            		Member overriden = null;
            		while ((overriden = m.getOverriddenMember())!=null){
            			m = (Method)overriden;
            			result.put(m, iname);
            		}
            	}
            }
        }
        return result;

    }
    */

    /**
     * Gets corresponding GAST method to SOMOX operation
     * @param operation from SOMOX
     * @return GAST Method or null if not finded
     */
    private Method findMethod(Operation operation) {
    	    	
        for (MethodLevelSourceCodeLink link : sourceCodeDecoratorRepository.getMethodLevelSourceCodeLink()) {
            if (link.getOperation().equals(operation)) {
            	Method retItem = (Method) link.getFunction();
            	return retItem;//for now it provides only methods in java FIXME: some check for this case
            }
        }
        
        return null; //if not found FIXME: report that method is not finded
    }
}
