package org.ow2.dsrg.fm.qabstractor.extract;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.apache.log4j.Logger;
import org.ow2.dsrg.fm.qabstractor.Settings;
import org.ow2.dsrg.fm.qabstractor.transformation.Transformation;
import org.ow2.dsrg.fm.qabstractor.utils.ClassComparator;

import de.fzi.gast.accesses.Access;
import de.fzi.gast.accesses.BaseAccess;
import de.fzi.gast.accesses.CompositeAccess;
import de.fzi.gast.accesses.FunctionAccess;
import de.fzi.gast.functions.Constructor;
import de.fzi.gast.statements.SimpleStatement;
import de.fzi.gast.types.GASTClass;

/**
 * Extracts addional classes of component from already defined class.
 * Additional classes is identified as class which is allocated during runtime,
 * class which is inner class of already added class or interface of that class
 * if option includeInterface is enabled.
 * @author Josef Reidinger
 */
public class ClassExtractor {

    /**
     * Gets metadata extractor and add to it all class which is additionaly
     * found.
     * @param extr Metadata extractor which contain initial classes of
     *              components
     */
    public static void extractClasses(MetadataExtractor extr) {
        Extract extract = new Extract(extr);
        for (String component : extr.getComponents()) {
            extract.processComponent(component);
            extr.addImplClasses(component, extract.getResult());
        }
    }

    /** utility class*/
    private ClassExtractor() {
    }

    /**
     * Traverses know component class and add to that component all classes which
     * is dynamic allocated. For improved performance and to avoid endless loop
     * it uses stack instead of recursion.
     */
    private static class Extract extends Transformation {

        private Set<GASTClass> result;
        private List<GASTClass> stack;

        Extract(MetadataExtractor extr) {
            super(Logger.getLogger(ClassExtractor.class), extr);
        }

        /**
         * gets all classes in processed component.
         * @return Set of classes which belongs to processed component
         */
        public Set<GASTClass> getResult() {
            return result;
        }

        /**
         * process one component. Traverses all classes (even newly added) and
         * creating final set of class.
         * It doesn't return any result and to get result is used getResult.
         * Reason is that processComponent is used from Transformation to
         * benefit from automatic traversal.
         * @see getResult
         * @see Transformation
         * @param componentName name of component
         */
        @Override
        public void processComponent(String componentName) {
            result = new TreeSet<GASTClass>(new ClassComparator());
            stack = new LinkedList<GASTClass>(extractor.getImplClasses(componentName));
            while (!stack.isEmpty()) {
                GASTClass cl = stack.get(0);
                stack.remove(0);
                if (!result.contains(cl)) {
                    log.debug("add class " + cl.getQualifiedName() + " to component " + componentName);
                    result.add(cl);
                    if (!cl.getInnerClasses().isEmpty()) {
                        stack.addAll(cl.getInnerClasses());
                    }
                    if (Settings.instance().isIncludeInterface()) {
                        for (GASTClass gc : cl.getSuperTypes()) {
                            if (gc.isInterface()) {
                                stack.add(gc);
                            }
                        }
                    }
                    processClass(cl);
                }
            }
        }

        /**
         * Process each class. It adds processing of constructor, which is not
         * used during common transformation.
         * @param clas
         */
        @Override
        public void processClass(GASTClass clas) {
            super.processClass(clas);
            for (Constructor c : clas.getConstructors()) {
                if (c.getBody() != null) {
                    processBlock(c.getBody());
                }
            }
        }

        /**
         * Process simple statements which is the one which include dynamic
         * allocation.
         * @param simpleStatement to process
         */
        @Override
        protected void processSimpleStatement(SimpleStatement simpleStatement) {
            super.processSimpleStatement(simpleStatement);
            for (BaseAccess a : simpleStatement.getAccesses()) {
                findClass(a);
            }
        }

        /**
         * helper that indentifies calling of constructor and add it to stack
         * to process
         * @param access to search
         */
        private void findClass(BaseAccess access) {
            if (access instanceof CompositeAccess) {
                for (BaseAccess ac : ((CompositeAccess) access).getAccesses()) {
                    findClass(ac);
                }
            } else if (access instanceof FunctionAccess) {
                FunctionAccess fa = (FunctionAccess) access;
                if (fa.getTargetFunction() instanceof Constructor) {
                	GASTClass cls  =((Constructor) fa.getTargetFunction()).getSurroundingClass();
                	if (!extractor.isKnown(cls)){
                		stack.add(cls);
                	}
                }
            }
        }
    }
}
