package org.ow2.dsrg.fm.qabstractor;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLResourceFactoryImpl;
import org.ow2.dsrg.fm.qabstractor.exception.CodeException;
import org.ow2.dsrg.fm.qabstractor.extract.ClassExtractor;
import org.ow2.dsrg.fm.qabstractor.extract.MetadataExtractor;
import org.ow2.dsrg.fm.qabstractor.pointsto.DummyRuntimeClassResolver;
import org.ow2.dsrg.fm.qabstractor.transformation.Abstract;
import org.ow2.dsrg.fm.qabstractor.transformation.Clean;
import org.ow2.dsrg.fm.qabstractor.transformation.Inline;
import org.ow2.dsrg.fm.qabstractor.transformation.JumpReplacer;
import org.ow2.dsrg.fm.qabstractor.transformation.ToTBP;
import org.ow2.dsrg.fm.tbplib.architecture.Component;
import org.ow2.dsrg.fm.tbplib.node.TBPSpecification;
import org.ow2.dsrg.fm.tbplib.reference.EnumerationType;
import org.ow2.dsrg.fm.tbplib.util.CodeWriter;
import org.ow2.dsrg.fm.tbplib.util.TransformationChain;

import de.fzi.gast.core.Root;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * Entry point to aplication. Provides static method to start transformation.
 * 
 * @author JR
 */
public class Transformer {

	/**
	 * Store GAST tree to XML file. Could be used to see output of each
	 * transformation.
	 * 
	 * @param root
	 *            root of GAST tree to serialize
	 * @param outputFile
	 *            path to output file
	 * @throws IOException
	 *             if problem occurs during writing file
	 */
	public static void exportToXML(Root root, String outputFile)
			throws IOException {
		EObject eobj = root;

		ResourceSet rs = new ResourceSetImpl();
		rs.getResourceFactoryRegistry().getExtensionToFactoryMap().put("xml",
				new XMLResourceFactoryImpl());

		URI uri = URI.createFileURI(outputFile);

		Resource resource = rs.createResource(uri);

		resource.getContents().add(eobj);

		resource.save(null);
	}

	/**
	 * Main entry point to transformation. Can be used from command line
	 * interface or from eclipse plugin.
	 * 
	 * @param r
	 *            root of GAST tree
	 * @param extractor
	 *            instance of metadataExtractor with component description
	 * @return set of created TBP specifications
	 * @throws IOException
	 *             throwed if problem occur during writting TBP file
	 * @throws CodeException
	 *             if analysed code contains problematic patterns
	 */
	public static List<TBPSpecification<String>> transform(String filePrefix,
			Root r, MetadataExtractor extractor, SourceCodeDecoratorRepository scdrep) throws IOException,
			CodeException {
		ClassExtractor.extractClasses(extractor);
		Inline inline = new Inline(extractor, new DummyRuntimeClassResolver(r),
				r);
		Abstract abstrac = new Abstract(extractor);
		JumpReplacer jr = new JumpReplacer(extractor);
		List<TBPSpecification<String>> ret = new LinkedList<TBPSpecification<String>>();
		Logger logger = Logger.getLogger(Transformer.class);
		logger.info("==============================================================");
		logger.info("Q-Abstractor");
		logger.info("Copyright (C) 2009  DSRG, Charles University in Prague");
		logger.info("http://dsrg.mff.cuni.cz/");
		logger
				.info("==============================================================");
		for (String component : extractor.getComponents()) {
			try {
				logger.info("Process component " + component);
				logger.debug("Start abstracting ... ");
				abstrac.processComponent(component);
				if (Settings.instance().isReplacedReturns()) {
					logger.debug("Start replacing returns ...");
					jr.processComponent(component);
				}
				logger.debug("Start inlining ...");
				inline.processComponent(component);
				// exportToXML(r, "./deploy/exportafterinline" + component +
				// ".xml");
				logger.debug("Start cleaning ...");
				Clean cleaner = new Clean(extractor);
	
				cleaner.processClass(extractor.getExtractedClass(component));
				logger.debug("Start creating TBP ...");
				ToTBP gparser = new ToTBP();
				TBPSpecification<String> res = gparser
						.createComponentSpecification(extractor, component, scdrep);
				ret.add(res);
				// maybe move in future outside of transformation as writing is not
				// its goal
				logger.debug("Start writing TBP ...");
	
				String filename = createFilename(filePrefix, new String(component));
	
				CodeWriter wr = new CodeWriter(filename + ".tbp");
				TransformationChain.printTBP(wr, res, new Component(component,
						new LinkedList<EnumerationType>()));
				wr.close();
				logger.debug("Finish analyzing component " + component);
			}
			catch (CodeException e) {
				logger.warn("Warning - recursive method call, no protocol for '" + component + "' will be created: " + e.getMessage());
			}
			
		}
		return ret;
	}

	private Transformer() {
	}

	private static String createFilename(String filePrefix, String src) {
		
		src = fixName(src);
		return filePrefix + File.separator + src;
	}
	
	public static String fixName(String src) {
		src = src.replaceAll("<", "_");
		src = src.replaceAll(">", "_");
		src = src.replaceAll(" ", "_");
		src = src.replaceAll("\\.", "_");
		
		return src;
	}
}
