package org.ow2.dsrg.fm.qabstractor;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import org.ow2.dsrg.fm.qabstractor.transformation.JumpReplacer;

/**
 * Singleton class which stores configuration for this application. Can also
 * loads configuration file which stores options.
 * For adding new option provide getter and setter (possible to fill from
 * some ui dialog) and add loader from file (new enum in Options and in switch
 * in loadFromFile)
 *
 * @author Josef Reidinger
 */
public class Settings {

    private boolean replacedReturns;
    private String pointsTo;
    private boolean includeInterface;

    private enum Options {

        REPLACED_RETURNS,
        POINTS_TO,
        INCLUDE_INTERFACE
    }

    /* singleton */
    static private Settings instance;

    /**
     * Generic exception that inform about malformed configuration file.
     */
    public class ConfigurationError extends Exception {

        protected ConfigurationError(String message) {
            super(message);
        }
    }

    /**
     * Exception that reports invalid value for option such as not integer or
     * value which is not in allowed range.
     */
    public class InvalidValue extends ConfigurationError {

        public InvalidValue(String variable, String value) {
            super("Variable " + variable + " from configuration file has invalid value " + value);
        }
    }

    /**
     * Exception that reports unknown option.
     */
    public class UnknownOption extends ConfigurationError {

        public UnknownOption(String variable) {
            super("Variable " + variable + " from configuration file is unknown");
        }
    }

    /**
     * Exception that reports malformed line.
     */
    public class InvalidLine extends ConfigurationError {

        public InvalidLine(String line) {
            super("Line " + line + " from configuration file is not recognized as configuration line");
        }
    }

    /**
     * Loads settings to singleton from file. Options which is not specified in
     * configuration file is preserved.
     * @param file to load
     * @throws IOException thrown in case of input error such as file not exist
     *                      or not enought permissions
     * @throws ConfigurationError thrown in case of malformed configuration file
     */
    public void loadFromFile(String file) throws IOException, ConfigurationError {
        FileInputStream fstream = new FileInputStream(file);
        DataInputStream in = new DataInputStream(fstream);
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        String line;
        try {
            for (line = br.readLine(); line != null; line = br.readLine()) {
                if (line.matches("^\\s*#.*") || line.trim().isEmpty()) //comment or empty line
                {
                    continue;
                }
                String[] elem = line.split("=", 2);
                if (elem.length != 2) //there is no '=' or too much '='
                {
                    throw new InvalidLine(line);
                }
                try {
                    switch (Options.valueOf(elem[0])) {
                        case INCLUDE_INTERFACE:
                            includeInterface = Boolean.valueOf(elem[1].trim());
                            break;
                        case POINTS_TO:
                            pointsTo = elem[1].trim();
                            break;
                        case REPLACED_RETURNS:
                            replacedReturns = Boolean.valueOf(elem[1].trim());
                            break;
                        default:
                            assert false : "Unknown option from enum";
                    }
                } catch (IllegalArgumentException ia) {
                    throw new UnknownOption(elem[0]);
                } catch (Exception e) {
                    throw new InvalidValue(elem[0], elem[1]);
                }
            }
        } finally {
            br.close();
        }
    }

    /**
     * get instance of class, if doesn't exist create new one
     * @return instance of class
     */
    static public Settings instance() {
        if (instance == null) {
            instance = new Settings();
        }
        return instance;
    }

    private Settings() {
        replacedReturns = true;
        pointsTo = "dummy";
        includeInterface = true;
    }

    /**
     * Gets choosen points-to analysis implementation
     * @return String which contains indentifier of implementation
     */
    public String getPointsTo() {
        return pointsTo;
    }

    /**
     * Sets choosen points-to analysis implementation
     * @param pointsTo String which contains indentifier of implementation
     */
    public void setPointsTo(String pointsTo) {
        this.pointsTo = pointsTo;
    }

    /**
     * Specifies if transformation should safety replace 'return'.
     * @see JumpReplacer
     * @return true if 'return' must be safety replace
     */
    public boolean isReplacedReturns() {
        return replacedReturns;
    }

    /**
     * Sets if transformation should safety replace 'return'.
     * @param replacedReturns true if 'return' must be safety replace
     * @see JumpReplacer
     */
    public void setReplacedReturns(boolean replacedReturns) {
        this.replacedReturns = replacedReturns;
    }

    /**
     * Gets true if add implicitely add interface of object of component to that
     * component.
     * @return true if implicitely add
     */
    public boolean isIncludeInterface() {
        return includeInterface;
    }

    /**
     * Gets true if add implicitely add interface of object of component to that
     * component.
     * @param includeInterface true if implicitely add
     */
    public void setIncludeInterface(boolean includeInterface) {
        this.includeInterface = includeInterface;
    }
}
