package eu.qimpress.reverseengineering.gast2seff.visitors;

import java.util.BitSet;
import java.util.Map;

import org.apache.log4j.Logger;

import de.fzi.gast.accesses.Access;
import de.fzi.gast.accesses.BaseAccess;
import de.fzi.gast.accesses.FunctionAccess;
import de.fzi.gast.core.Position;
import de.fzi.gast.statements.BlockStatement;
import de.fzi.gast.statements.Branch;
import de.fzi.gast.statements.BranchStatement;
import de.fzi.gast.statements.LoopStatement;
import de.fzi.gast.statements.SimpleStatement;
import de.fzi.gast.statements.Statement;
import de.fzi.gast.statements.util.statementsSwitch;
import eu.qimpress.reverseengineering.gast2seff.jobs.GAST2SEFFJob;
import eu.qimpress.reverseengineering.gast2seff.visitors.FunctionCallClassificationVisitor.FunctionCallType;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.seff.AbstractBranchTransition;
import eu.qimpress.seff.BranchAction;
import eu.qimpress.seff.ExternalCallAction;
import eu.qimpress.seff.InternalAction;
import eu.qimpress.seff.LoopAction;
import eu.qimpress.seff.ResourceDemandingBehaviour;
import eu.qimpress.seff.seffFactory;
import eu.qimpress.sourcecodedecorator.InterfaceSourceCodeLink;
import eu.qimpress.sourcecodedecorator.MethodLevelSourceCodeLink;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * A visitor which traverses a GAST behaviour and creates a SEFF matching the traversed behaviour. The 
 * generated SEFF is abstracted based on a classification of the GAST statements into external and
 * internal service calls.
 * <br>
 * <code>functionClassificationAnnotation</code> classifies which elements to hold when traversing the
 * GAST behaviour.
 * 
 * @author Steffen Becker, Klaus Krogmann
 */
public class GastStatementVisitor 
extends statementsSwitch<Object> {

	private static final Logger logger = Logger.getLogger(GastStatementVisitor.class);
	
	/**
	 * The RD-Behaviour to generate
	 */
	private ResourceDemandingBehaviour seff;
	
	/**
	 * Mapping to SAMM repository (for external call lookup)
	 */
	private SourceCodeDecoratorRepository sourceCodeDecoratorRepository;
	
	/**
	 * The component the created SEFF belongs to.
	 */
	private PrimitiveComponent primitiveComponent;
	
	/**
	 * Map which contains for each statement in the GAST model the type of the statement 
	 * classified according to {@link FunctionCallType}. Nodes of control flow constructs
	 * like loops and branches carry the union of the annotations of their child statements
	 */
	private Map<Statement, BitSet> functionClassificationAnnotation;

	/**
	 * Classification annotation of the last visited statement. Used to skip generating
	 * SEFF actions if they should be omitted because of the SEFFs abstraction rule 
	 */
	private BitSet lastType = null;
	
	/**
	 * Constructor
	 * @param functionClassificationAnnotations A map containing the type annotations for the
	 * nodes of the GAST model. Generated by a {@link FunctionCallClassificationVisitor}.
	 * @param resourceDemandingBehaviour The RD-behaviour to generate
	 * @param gastBehaviourRepository The gast behaviour which maps gast statements and SAMM repository.
	 * @param primitiveComponent 
	 */
	public GastStatementVisitor(Map<Statement,BitSet> functionClassificationAnnotations,
			ResourceDemandingBehaviour resourceDemandingBehaviour, 
			SourceCodeDecoratorRepository gastBehaviourRepository, 
			PrimitiveComponent primitiveComponent) {
		super();
		
		this.seff = resourceDemandingBehaviour;
		this.functionClassificationAnnotation = functionClassificationAnnotations;
		this.sourceCodeDecoratorRepository = gastBehaviourRepository;
		this.primitiveComponent = primitiveComponent;
	}

	@Override
	public Object caseBlockStatement(BlockStatement object) {
		for (Statement s : object.getStatements()) {
			BitSet thisType = this.functionClassificationAnnotation.get(s);
			if (!shouldSkip(lastType,thisType)) { // Only generate elements for statements which should not be abstracted away
				// avoid infinite recursion
				if(!isVisitedStatement(thisType)) {
					setVisited(thisType);
					doSwitch(s);
				}
			}
			lastType = thisType;
		}		
		return null;
	}

	@Override
	public Object caseBranchStatement(BranchStatement object) {
		if (containsExternalCall(object)) {
			BranchAction branch = seffFactory.eINSTANCE.createBranchAction();
			seff.getSteps().add(branch);
			branch.setName(positionToString(object.getPosition()));
			branch.setDocumentation(blockToString(object.getBlockstatement()));			
			
			for (Branch b : object.getBranches()) {
				AbstractBranchTransition bt = seffFactory.eINSTANCE.createProbabilisticBranchTransition();
				bt.setResourceDemandingBehaviour(seffFactory.eINSTANCE.createResourceDemandingBehaviour());
				bt.getResourceDemandingBehaviour().getSteps().add(seffFactory.eINSTANCE.createStartAction());				
				bt.setName("parent " + positionToString(object.getPosition()) + "/" + positionToString(b.getPosition())); //use parent position since branch position is empty
				branch.getAbstractBranchTransition().add(bt);
				GastStatementVisitor visitor = new GastStatementVisitor(this.functionClassificationAnnotation,
						bt.getResourceDemandingBehaviour(), this.sourceCodeDecoratorRepository, this.primitiveComponent);
				Statement s = b.getStatement();
				visitor.doSwitch(s);				
				bt.getResourceDemandingBehaviour().getSteps().add(seffFactory.eINSTANCE.createStopAction());
				GAST2SEFFJob.connectActions(bt.getResourceDemandingBehaviour());
			}
		} else {
			createInternalAction(object);
		}
		return null;
	}

	@Override
	public Object caseLoopStatement(LoopStatement object) {
		if (containsExternalCall(object)) {
			LoopAction loop = seffFactory.eINSTANCE.createLoopAction();
			loop.setBodyBehaviour(seffFactory.eINSTANCE.createResourceDemandingBehaviour());
			seff.getSteps().add(loop);
			loop.getBodyBehaviour().getSteps().add(seffFactory.eINSTANCE.createStartAction());
			loop.setName(positionToString(object.getPosition()));
			loop.setDocumentation(blockToString(object.getBlockstatement()));

			new GastStatementVisitor(this.functionClassificationAnnotation,
					loop.getBodyBehaviour(), this.sourceCodeDecoratorRepository,
					this.primitiveComponent).doSwitch(object.getBody());

			loop.getBodyBehaviour().getSteps().add(seffFactory.eINSTANCE.createStopAction());
			GAST2SEFFJob.connectActions(loop.getBodyBehaviour());
		} else {
			createInternalAction(object);
		}
		return null;
	}
	
	@Override
	public Object caseExceptionHandler(de.fzi.gast.statements.ExceptionHandler object) {
		if (containsExternalCall(object)) {

			// visit guarded block
			new GastStatementVisitor(this.functionClassificationAnnotation,
					seff, this.sourceCodeDecoratorRepository,
					this.primitiveComponent).doSwitch(object.getGuardedBlock());

			// visit finally block if exists 
			if(object.getFinallyBlock() != null) {
				new GastStatementVisitor(this.functionClassificationAnnotation,
					seff, this.sourceCodeDecoratorRepository,
					this.primitiveComponent).doSwitch(object.getFinallyBlock());
			}
			
		} else {
			createInternalAction(object);
		}
		return null;
	}	

	@Override
	public Object caseSimpleStatement(SimpleStatement object) {
		BitSet statementAnnotation = this.functionClassificationAnnotation.get(object);
		if (isExternalCall(statementAnnotation)) {
			createExternalCallAction(object);
		} else if (isInternalCall(statementAnnotation)) {
			FunctionAccess functionAccess = getFunctionAccess(object);
			BlockStatement body = functionAccess.getTargetFunction().getBody();
			if (body != null) {

				// avoid infinite recursion
				BitSet thisType = this.functionClassificationAnnotation.get(object);
				if(!isVisitedStatement(thisType)) {
					setVisited(thisType);
					doSwitch(body);
				}
			} else {
				String msg = "Behaviour not set in GAST for "+functionAccess.getTargetFunction().getSimpleName();
				if(object.getPosition() != null && object.getPosition().getSourceFile() != null && object.getPosition().getSourceFile().getPathName() != null) {
					msg += ". Tried to call from " + object.getPosition().getSourceFile().getPathName() + ".";
				} else {
					msg += ". (caller position unknown)";
				}
				logger.warn(msg);
			}
		} else {
			createInternalAction(object);
		}
		return null;
	}

	/** Returns true if the statement with thisType should not generate an action in the newly generated SEFF
	 * @param lastType The type of the preceeding statement
	 * @param thisType The type of the statement to test
	 * @return true if the current statement should not generate an element in the SEFF, i.e., it should be abstracted and thrown away
	 */
	private boolean shouldSkip(BitSet lastType, BitSet thisType) {
		if (lastType == null)
			return false;
		
		if (isExternalCall(thisType))
			return false;
		
		// Here I know that thisType is internal or library
		// Hence, I can skip this if the last type was not an external call
		return !isExternalCall(lastType);
	}

	private void createExternalCallAction(SimpleStatement object) {
		ExternalCallAction call = seffFactory.eINSTANCE.createExternalCallAction();
		FunctionAccess access = getFunctionAccess(object);
		call.setName(access.getTargetFunction().getSimpleName());
		InterfacePortOperationTuple ifOperationTuple = getCalledInterfacePort(access);	
		call.setCalledInterfacePort(ifOperationTuple.interfacePort);
		call.setCalledService(ifOperationTuple.operation);
		call.setDocumentation(positionToString(object.getPosition()));
		seff.getSteps().add(call);
	}

	/**
	 * Query the interface port for the function access using the source code decorator.
	 * @param access The access to find in the SAMM
	 * @return interface port and operation for corresponding to the access.
	 */
	private InterfacePortOperationTuple getCalledInterfacePort(FunctionAccess access) {
		InterfacePortOperationTuple interfacePortOperationTuple = new InterfacePortOperationTuple();
		
		for(InterfacePort ifPort : this.primitiveComponent.getRequired()) {			
			for(InterfaceSourceCodeLink ifLink : this.sourceCodeDecoratorRepository.getInterfaceSourceCodeLink()) {				
				if(ifPort.getInterfaceType().equals(ifLink.getInterface())) {
					if(ifLink.getGastClass().equals(access.getAccessedClass())) {
						
						logger.trace("accessed interface port " + ifPort.getName());
						interfacePortOperationTuple.interfacePort = ifPort;		
						//query operation:
						interfacePortOperationTuple.operation = queryInterfaceOperation(access);
						
						return interfacePortOperationTuple;
					}
				}
			}
		}
		
		logger.warn("found no if port for " + access.getAccessedClass().getSimpleName());
		
		return interfacePortOperationTuple;
	}

	/**
	 * Interface operation query
	 * @param access The access to find in the SAMM
	 * @return Operation corresponding to function access
	 */
	private Operation queryInterfaceOperation(FunctionAccess access) {					
		for(MethodLevelSourceCodeLink methodLink : this.sourceCodeDecoratorRepository.getMethodLevelSourceCodeLink()) {
			
			if(methodLink.getFunction().equals(access.getTargetFunction())) {
														
				logger.trace("accessed operation " + methodLink.getOperation().getName());				
				return methodLink.getOperation();  									
			}
		}
		
		logger.warn("no accessed operation found for " + access.getTargetFunction().getSimpleName());			
		return null;
	}

	private boolean isExternalCall(BitSet statementAnnotation) {
		return statementAnnotation.get(FunctionCallClassificationVisitor.getIndex(FunctionCallType.EXTERNAL));
	}

	private boolean isInternalCall(BitSet statementAnnotation) {
		return statementAnnotation.get(FunctionCallClassificationVisitor.getIndex(FunctionCallType.INTERNAL));
	}
	
	private boolean isVisitedStatement(BitSet statementAnnotation) {
		return statementAnnotation.get(FunctionCallClassificationVisitor.getIndex(FunctionCallType.VISITED));
	}	
	
	private void setVisited(BitSet thisType) {
		thisType.set(FunctionCallClassificationVisitor.getIndex(
				FunctionCallType.VISITED), true);

	}

	private FunctionAccess getFunctionAccess(SimpleStatement object) {
		for (BaseAccess a : object.getAccesses()) {
			if (a instanceof FunctionAccess) {
				return (FunctionAccess) a;
			}
		}
		return null;
	}

	private void createInternalAction(Statement statement) {
		InternalAction ia = seffFactory.eINSTANCE.createInternalAction();
		
		ia.setName("IA " + positionToString(statement.getPosition()));
		ia.setDocumentation(blockToString(statement.getBlockstatement()) + "; Statement SISSyID: " + statement.getSissyId());
		seff.getSteps().add(ia);
	}
	
	private String blockToString(BlockStatement blockstatement) {
		if(blockstatement != null) {
			StringBuilder blockString = new StringBuilder("Block: ");
			blockString.append(blockstatement.toString());
			if( blockstatement.getAccesses() != null &&  
					blockstatement.getAccesses().size() >= 1
					) {
				     BaseAccess firstAccess = blockstatement.getAccesses().get(0);
				     if (firstAccess instanceof Access) {
				    	 Access access = (Access)firstAccess;
				    	 
				    	 if (access.getAccessedClass() != null) {
				    		 blockString.append(" " + access.getAccessedClass().getSimpleName() + "...");				    		 
				    	 }
				     }
			}
			return blockString.toString();
		} else {
			return "No blockstatement";
		}
	}
	
	private String positionToString(Position position) {
		StringBuilder positionString = new StringBuilder("position: ");
		if(position != null) {
			if(position.getSourceFile() != null && position.getSourceFile().getClass() != null) {
				positionString.append(position.getSourceFile().getClass());
			}			
			positionString.append(" from " + position.getStartLine());
			positionString.append(" to " + position.getEndLine());
		} else {			
			positionString.append("no position information available");
		}
		return positionString.toString();
	}

	/**
	 * Returns true if the statement or one of its child statements (e.g., for loops or branches) is an external service call
	 * @param object The statement to check
	 * @return true if the statement or one of its child statements is an external service call
	 */
	private boolean containsExternalCall(Statement object) {
		return this.functionClassificationAnnotation.get(object).get(FunctionCallClassificationVisitor.getIndex(FunctionCallType.EXTERNAL));
	}
	
	private class InterfacePortOperationTuple {
		public InterfacePort interfacePort;
		public Operation operation;
	}

}


