package org.somox.analyzerrules;

import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.somox.configuration.SoMoXConfiguration;
import org.somox.metrics.Coupling;
import org.somox.metrics.DMS;
import org.somox.metrics.DirectoryMapping;
import org.somox.metrics.InterfaceViolation;
import org.somox.metrics.NameResemblance;
import org.somox.metrics.PackageMapping;
import org.somox.metrics.SliceLayerArchitectureQuality;
import org.somox.metrics.SubsystemComponent;

/**
 * Requires the following metrics: Abstractness, Coupling, and NameResemblance
 * @author kelsaka
 */
public class WeightedMetricsAnalyzerRule implements IAnalyzerRule {
	
	private SoMoXConfiguration somoxConfig;
	//private Properties properties;
	private Logger logger = Logger.getLogger(WeightedMetricsAnalyzerRule.class);
	private double packageMappingWeight, directoryMappingWeight, DMSWeight;

	public WeightedMetricsAnalyzerRule(SoMoXConfiguration somoxConfiguration) throws AnalyzerRuleException {
		// TODO: replace properties by somoxConfiguration
		//this.properties = ModelAnalyzerWeights.getProperties();
		somoxConfig=somoxConfiguration;
		
		getWeightsFromConfiguration();
	}

	/**
	 * Computes the overall metric score based on Landrys weighting functions
	 * 
	 * @param argAB The ID for every metric and its calculated score for the components A B
	 * @param argBA The ID for every metric and its calculated score for the components B A
	 * @return the overall score of the metric
	 * @throws AnalyzerRuleException 
	 */
	public double computeOverallMetricValue(Map<Integer, Double> argAB,
			Map<Integer, Double> argBA) throws AnalyzerRuleException {
		
		if (argAB.size() == 0 && argBA.size() == 0)
			throw new AnalyzerRuleException("Metric not set", new IllegalArgumentException());

		double nameResemblance = argAB.get(NameResemblance.getSpecificMID());
		double subsystemComponent = argAB.get(SubsystemComponent.getSpecificMID());
		double packageMapping = argAB.get(PackageMapping.getSpecificMID());
		double directoryMapping = argAB.get(DirectoryMapping.getSpecificMID());
		double dms = argAB.get(DMS.getSpecificMID());
		double slaq = argAB.get(SliceLayerArchitectureQuality.getSpecificMID());

		double couplingAB = argAB.get(Coupling.getSpecificMID());
		double couplingBA = argBA.get(Coupling.getSpecificMID());
		double coupling = Math.max(couplingAB, couplingBA);

		double interfaceViolationAB = argAB.get(InterfaceViolation.getSpecificMID());
		double interfaceViolationBA = argBA.get(InterfaceViolation.getSpecificMID());
		double interfaceViolation = Math.max(interfaceViolationAB, interfaceViolationBA);
		
		double nameResemblanceWeight = getNameResemblanceWeight(nameResemblance, coupling);
		double interfaceViolationWeight = getInterfaceViolationWeight(coupling, interfaceViolation);
		double subsystemComponentWeight = getSubsystemComponentWeight(slaq);
		
		//compute overall metric score		
		double sum = nameResemblanceWeight+interfaceViolationWeight+subsystemComponentWeight+packageMappingWeight+DMSWeight+directoryMappingWeight; //TODO: check use of constants (packageMappingWeight, ...)
		double score = nameResemblance*(double)nameResemblanceWeight + interfaceViolation*(double)interfaceViolationWeight 
		               + subsystemComponent*(double)subsystemComponentWeight + packageMapping*(double)packageMappingWeight - dms*(double)DMSWeight + directoryMapping*(double)directoryMappingWeight;
		
		score = score/sum;
		
		return score;
	}

	private double getNameResemblanceWeight(double nameResemblance,
			double coupling) {
		double w1;
		//determine weight of nameResemblance
		if (coupling < 0.2) {
			//w1 = Double.parseDouble(properties.getProperty("org.somox.nameResemblance.weightLowCoupling"));
			w1 = somoxConfig.getWeightLowCoupling();
		} else if (coupling < 0.7) {
			//w1 = Double.parseDouble(properties.getProperty("org.somox.nameResemblance.weightHighCoupling"));
			w1 = somoxConfig.getWeightHighCoupling();
		} else {
			if (nameResemblance < 0.5) {
				//w1 = Double.parseDouble(properties.getProperty("org.somox.nameResemblance.weightLowNameResemblance"));
				w1 = somoxConfig.getWeightLowNameResemblance();
			} else if (nameResemblance < 0.7) {
				//w1  = Double.parseDouble(properties.getProperty("org.somox.nameResemblance.weightMidNameResemblance"));
				w1  = somoxConfig.getWeightMidNameResemblance();
			} else if (nameResemblance < 0.9) {
				//w1  = Double.parseDouble(properties.getProperty("org.somox.nameResemblance.weightHighNameResemblance"));
				w1  = somoxConfig.getWeightHighNameResemblance();
			} else {
				//w1  = Double.parseDouble(properties.getProperty("org.somox.nameResemblance.weightHighestNameResemblance"));
				w1  = somoxConfig.getWeightHighestNameResemblance();
			}
		}
		return w1;
	}

	private double getInterfaceViolationWeight(double coupling,
			double interfaceViolation) {
		double w2;
		//determine weight of interfaceViolation
		if (coupling >=0.5 && interfaceViolation > 0.0) {
			//w2  = Double.parseDouble(properties.getProperty("org.somox.interfaceViolation.weightInterfaceViolationRelevant"));
			w2  = somoxConfig.getWeightInterfaceViolationRelevant();
		} else {
			//w2  = Double.parseDouble(properties.getProperty("org.somox.interfaceViolation.weightInterfaceViolationIrrelevant"));
			w2  = somoxConfig.getWeightInterfaceViolationIrrelevant();
		}
		return w2;
	}

	private double getSubsystemComponentWeight(double slaq) {
		double w3;
		//determine weight of subsystemComponent
		if (slaq >= 0.5) {
			//w3 = Double.parseDouble(properties.getProperty("org.somox.subsystemComponent.weightHighSLAQ"));
			w3 = somoxConfig.getWeightHighSLAQ();
		} else {
			//w3 = Double.parseDouble(properties.getProperty("org.somox.subsystemComponent.weightLowSLAQ"));
			w3 = somoxConfig.getWeightLowSLAQ();
		}
		return w3;
	}

	private void getWeightsFromConfiguration() throws AnalyzerRuleException {
		try {
//			packageMappingWeight = Double.parseDouble(properties.getProperty("org.somox.packageMapping.weightPackageMapping"));
//			directoryMappingWeight = Double.parseDouble(properties.getProperty("org.somox.directoryMapping.weightDirectoryMapping"));
//			DMSWeight = Double.parseDouble(properties.getProperty("org.somox.dms.weightDMS"));
			packageMappingWeight = somoxConfig.getWeightPackageMapping();
			directoryMappingWeight = somoxConfig.getWeightDirectoryMapping();
			DMSWeight = somoxConfig.getWeightDMS();

		} catch (Exception ex) {
			logger.error("Failed to retrieve metric configuration: " + ex);
			throw new AnalyzerRuleException("Failed to retrieve metric configuration",ex);
		}
	}	
}
