package org.somox.analyzer.simplemodelanalyzer.detection;

import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.somox.analyzer.ModelAnalyzerException;
import org.somox.analyzer.metriccomputation.ClusteringRelation;
import org.somox.analyzer.metriccomputation.TupleIterator;
import org.somox.analyzer.simplemodelanalyzer.SimpleAnalysisResult;
import org.somox.analyzerrules.AnalyzerRuleException;
import org.somox.analyzerrules.ModelAnalyzerWeights;
import org.somox.configuration.SoMoXConfiguration;

import de.fzi.gast.core.Root;
import de.fzi.gast.types.GASTClass;

public class ComponentDetectionByClustering implements IDetectionStrategy {

	/**
	 * The logger of this analyser 
	 */
	private Logger logger = Logger.getLogger(ComponentDetectionByClustering.class);
	
	public List<List<GASTClass>> startDetection(
			Root root,
			SimpleAnalysisResult analysisResult,
			SoMoXConfiguration somoxConfiguration,
			IProgressMonitor progressMonitor,
			List<List<GASTClass>> filteredComponents)
			throws ModelAnalyzerException {
		
		TupleIterator tI = null;
		try {
			tI = new TupleIterator(somoxConfiguration.getBlacklist(),filteredComponents,somoxConfiguration);
			tI.initialize(root, somoxConfiguration);
		} catch (AnalyzerRuleException e) {
			throw new ModelAnalyzerException("Metric computation failed", e);
		}
		
		List<List<GASTClass>> componentsFound = startClustering(analysisResult, filteredComponents, root, tI, progressMonitor);
		return componentsFound;
	}
	
	/**
	 * TODO: Document me
	 * @param result The analysis result used to store the result of the clustering algorithm
	 * @param filteredComponents A list of classes where the classes from the blacklist have been removed already
	 * @param root The root element of the GAST model
	 * @param tI TODO
	 * @param progressMonitor 
	 * @return TODO
	 * @throws ModelAnalyzerException
	 */
	private List<List<GASTClass>> startClustering(SimpleAnalysisResult result, List<List<GASTClass>> filteredComponents, Root root, TupleIterator tI, IProgressMonitor progressMonitor) throws ModelAnalyzerException {

		double threshold = initializeThreshold();
		int componentCount = filteredComponents.size();
		boolean newComponentsFound = true;
		int iteration = 0;

		while (newComponentsFound && filteredComponents.size()>1) {

			iteration++;

			logger.info("Clustering iteration nr.: " + iteration);
			logger.info("NR Components: " + filteredComponents.size());

			List<ClusteringRelation> results = computeAllMetrics(
					filteredComponents, root, tI, progressMonitor);

			//clustering
			filteredComponents = doClustering(results, threshold);

			if (filteredComponents.size() == componentCount) {
				newComponentsFound = false;
			} else {
				componentCount = filteredComponents.size();
			}
		}

		printComponentsFound(filteredComponents);
		
		return filteredComponents;
	}
	
	/**
	 * For the given list of potential components, i.e., classes, compute a triangular matrix of metrics indicating the 
	 * relationship of the two classes.
	 * @param filteredComponents The list of potential components
	 * @param root The root element of the GAST model used to analyse the code
	 * @param tI TODO document me
	 * @param progressMonitor The progress monitor used to indicate clustering progress
	 * @return The elements of the triangular matrix showing the relationship of all classes pairwise
	 * @throws ModelAnalyzerException Thrown if the metric computation fails unexpectedly
	 */
	private List<ClusteringRelation> computeAllMetrics(
			List<List<GASTClass>> filteredComponents, Root root,
			TupleIterator tI, IProgressMonitor progressMonitor) throws ModelAnalyzerException {
		int totalCount = filteredComponents.size() * (filteredComponents.size() - 1) / 2;

		IProgressMonitor clusteringProgressMonitor = new SubProgressMonitor(progressMonitor,totalCount);
		long startTimeClustering = System.nanoTime();

		//compute all metrics
		List<ClusteringRelation> results = new LinkedList<ClusteringRelation>();
		ListIterator<List<GASTClass>> firstIterator = filteredComponents.listIterator();
		
		int stepNo = 0;
		while (firstIterator.hasNext()) {
			ListIterator<List<GASTClass>> secondIterator = filteredComponents.listIterator(firstIterator.nextIndex()+1);
			List<GASTClass> currentFirst = firstIterator.next();
			while (secondIterator.hasNext()) {
				List<GASTClass> currentSecond = secondIterator.next();
				stepNo++;
				clusteringProgressMonitor.worked(1);

				results.add(computeClusteringRelation(
						filteredComponents, root, tI, currentFirst,
						currentSecond));
			}
			logger.debug(Math.round(stepNo * 100.0 / totalCount) + "% of clustering completed");
		}

		long clusteringTime = System.nanoTime() - startTimeClustering;
		logger.debug("TIME for Compute All Metrics: " + clusteringTime / Math.pow(10,9) + " s");			

		clusteringProgressMonitor.done();
		
		return results;
	}
	
	/**
	 * Perform the actual clustering of classes into composite components
	 * @param results The triangular matrix containing the metrics for the relationship of pairwise classes
	 * @param threshold The threshold used to filter clustering results
	 * @return TODO
	 */
	private List<List<GASTClass>> doClustering(
			List<ClusteringRelation> results, double threshold) {

		List<List<GASTClass>> newComponents = new LinkedList<List<GASTClass>>();

		List<Integer> alreadyUsed = new LinkedList<Integer>();

		findMaximumCusteringRelation(
				results, threshold, newComponents, alreadyUsed);

		clusterComponents(results, newComponents, alreadyUsed);
		
		return newComponents;
	}
	
	/**
	 * Combine components into composite components based on their relationship expressed by their pairwise metric value
	 * @param metrics The triangluar matrix containing the metrics of the pairwise component comparisions
	 * @param newComponents The result list. Should finally contain the new set of components including the clustered ones
	 * @param alreadyUsed TODO: ????
	 */
	private void clusterComponents(
			List<ClusteringRelation> metrics,
			List<List<GASTClass>> newComponents, 
			List<Integer> alreadyUsed) {
		// Copy the non-clustered components into the list of new component candidates
		// The metrics list only contains those metrics which have not been used to 
		// form a new composite component. All others have been removed by the previous 
		// method
		for(ClusteringRelation currentRelation : metrics) {
			if (!alreadyUsed.contains(currentRelation.getComponentA().hashCode())) {
				List<GASTClass> foundComponent = new LinkedList<GASTClass>();
				foundComponent.addAll(currentRelation.getComponentA());
				newComponents.add(foundComponent);					
				alreadyUsed.add(currentRelation.getComponentA().hashCode());
			} 
			if (!alreadyUsed.contains(currentRelation.getComponentB().hashCode())) {
				List<GASTClass> foundComponent = new LinkedList<GASTClass>();
				foundComponent.addAll(currentRelation.getComponentB());
				newComponents.add(foundComponent);
				alreadyUsed.add(currentRelation.getComponentB().hashCode());
			}
		}
	}
	
	/**
	 * Create a new {@link ClusteringRelation} based on the metrics computed for two given classes
	 * @param filteredComponents
	 * @param root
	 * @param tI
	 * @param currentFirst
	 * @param currentSecond
	 * @return
	 * @throws ModelAnalyzerException
	 */
	private ClusteringRelation computeClusteringRelation(
			List<List<GASTClass>> filteredComponents, Root root,
			TupleIterator tI, List<GASTClass> currentFirst,
			List<GASTClass> currentSecond) throws ModelAnalyzerException {
		ClusteringRelation currentRelation = new ClusteringRelation(currentFirst, currentSecond);
		try {
			currentRelation.setResult(tI.compute(root, currentFirst, currentSecond, filteredComponents));
		} catch (AnalyzerRuleException e) {
			logger.error("Model analyzer failed",e);
			throw new ModelAnalyzerException("Metric computation failed",e);
		}
		return currentRelation;
	}

	/**
	 * @return The threshold for the clustering algorithm taken from the configuration
	 */
	private double initializeThreshold() {
		double threshold = 0.5; //Default
		try {
			threshold = Double.parseDouble(ModelAnalyzerWeights.getProperties().getProperty("org.somox.clusteringThreshold"));
		} catch (NumberFormatException e1) {
			logger.error("config clustering threshold no number", e1);
		} catch (AnalyzerRuleException e1) {
			logger.error("loading config error");
		}
		return threshold;
	}

	private void printComponentsFound(List<List<GASTClass>> components) {
		int i;
		i=0;
		for (List<GASTClass> currentList : components) {
			i++;
			logger.debug("Comp Nr." + i + ":");
			for (GASTClass currentClass : currentList) {
				logger.debug(currentClass.getQualifiedName());
			}
		}
	}
	
	/** TODO: This method is unclear. What is the data computed and returned in which parameter?????
	 * @param results The triangular matrix containing the metrics for the relationship of pairwise classes
	 * @param threshold
	 * @param newComponents
	 * @param alreadyUsed
	 */
	private void findMaximumCusteringRelation(
			List<ClusteringRelation> results, 
			double threshold,
			List<List<GASTClass>> newComponents, 
			List<Integer> alreadyUsed) {
		ClusteringRelation maxRelation = null;
		double maxResult = 0.0;
		boolean found = true;
		ListIterator<ClusteringRelation> resultIterator = results.listIterator();

		while (found) {
			resultIterator = results.listIterator();
			// look at each element in the matrix of relationships
			while (resultIterator.hasNext()) {
				ClusteringRelation currentRelation = resultIterator.next();
				// Check whether either componentA or componentB has been used in a merger into
				// a new composite component before
				if (alreadyUsed.contains(currentRelation.getComponentA().hashCode()) ||
						alreadyUsed.contains(currentRelation.getComponentB().hashCode())) {
					resultIterator.remove();
				} else {
					if (currentRelation.getResult() >= maxResult) {
						maxResult = currentRelation.getResult();
						maxRelation = currentRelation;					
					}
				}
			}

			// among the not yet used components in a pairwise clustering, there are still components
			// above the threshold, thus, merge them into a new component
			if (maxResult > threshold) {
				// For the found pair of component candidates: merge them into a new component candidate
				List<GASTClass> foundComponent = new LinkedList<GASTClass>();
				foundComponent.addAll(maxRelation.getComponentA());
				foundComponent.addAll(maxRelation.getComponentB());
				newComponents.add(foundComponent);
				// mark componentA and componentB as used in a newly defined composite component 
				// Therefore both componentA and componentB cannot participate in a merger
				// in further iterations to form a new composite component
				alreadyUsed.add(maxRelation.getComponentA().hashCode());
				alreadyUsed.add(maxRelation.getComponentB().hashCode());
				maxResult = 0.0;
			} else {
				found = false;
			}
		}
	}

}
