package org.somox.analyzer.simplemodelanalyzer;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.somox.analyzer.AnalysisResult;
import org.somox.analyzer.ModelAnalyzer;
import org.somox.analyzer.ModelAnalyzerException;
import org.somox.analyzer.metriccomputation.ClusteringRelation;
import org.somox.analyzer.metriccomputation.TupleIterator;
import org.somox.analyzer.simplemodelanalyzer.detection.ComponentDetectionByClustering;
import org.somox.analyzer.simplemodelanalyzer.detection.ComponentDetectionByDirectoryStructure;
import org.somox.analyzer.simplemodelanalyzer.detection.ComponentDetectionStrategyFactory;
import org.somox.analyzer.simplemodelanalyzer.detection.IDetectionStrategy;
import org.somox.analyzerrules.AnalyzerRuleException;
import org.somox.analyzerrules.ModelAnalyzerWeights;
import org.somox.common.SoMoXProjectPreferences;
import org.somox.configuration.SoMoXConfiguration;
import org.somox.configuration.SoMoXConfiguration.ComponentDetectionStrategy;
import org.somox.extractor.ExtractionResult;
import org.somox.metrics.BlacklistFilter;
import org.somox.metrics.ClassAndCountStruct;
import org.somox.metrics.init.Initialization;

import de.fzi.gast.core.Root;
import de.fzi.gast.types.GASTClass;
import de.uka.ipd.sdq.workflow.ExecutionTimeLoggingProgressMonitor;
import eu.qimpress.qimpressgast.GASTBehaviourRepository;
import eu.qimpress.qimpressgast.qimpressgastFactory;
import eu.qimpress.samm.staticstructure.Repository;
import eu.qimpress.samm.staticstructure.StaticstructureFactory;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorFactory;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * This class runs a component detection based on a GAST input model and returns the results
 * to the SoMoX core for storing or further processing
 * 
 * @author Michael Hauck, Klaus Krogmann, Steffen Becker
 */
public class SimpleModelAnalyzer implements ModelAnalyzer {

	/**
	 * The current status of the analyzer
	 * @uml.property  name="status"
	 * @uml.associationEnd  
	 */
	private ModelAnalyzer.Status status = ModelAnalyzer.Status.READY;
	
	/**
	 * The logger of this analyser 
	 */
	private Logger logger = Logger.getLogger(SimpleModelAnalyzer.class);
	
	// ---------------------------------
	// Business Methods
	// ---------------------------------

	public void init() {}

	/* (non-Javadoc)
	 * @see org.somox.analyzer.ModelAnalyzer#analyze(java.util.HashMap, eu.qimpress.samm.staticstructure.Repository, java.util.HashMap, org.eclipse.core.runtime.IProgressMonitor)
	 */
	 public AnalysisResult analyze(SoMoXConfiguration somoxConfiguration,
			Repository internalArchitectureModel,
			HashMap<String, ExtractionResult> extractionResultMap,
			IProgressMonitor progressMonitor) throws ModelAnalyzerException {

		this.status = ModelAnalyzer.Status.RUNNING;
		logger.info("SISSy Analyzer started with"
				+"\n SOMOX Configuration: "+somoxConfiguration
				+"\n internalArchitectureModel "+internalArchitectureModel
				+"\n extractionResultMap "+extractionResultMap);


		SimpleAnalysisResult analysisResult = initializeAnalysisResult();
		analysisResult.setResultStatus(AnalysisResult.ResultStatus.FAILED);
		
		String platformPath = somoxConfiguration.getAnalyserInputFile(); // preferences.get(SoMoXProjectPreferences.SOMOX_ANALYZER_INPUT_FILE);
		if (platformPath != null) {
			URI fileURI = URI.createPlatformResourceURI(platformPath, true);
			if (fileURI.fileExtension().toLowerCase().equals("gast")) {
				Initialization init = new Initialization();
				List<List<GASTClass>> originalComponents = init.extractLists(fileURI);
				Root root = init.getRoot();
	
				analyzeGASTModel(root, analysisResult, originalComponents,
						somoxConfiguration, progressMonitor);
			}
		}
		return analysisResult;
	}

	/**
	 * Analyze the given GAST model to find components
	 * @param root The root of the GAST model to analyze
	 * @param analysisResult The result to produce
	 * @param originalComponents The unmodified list of GAST classes contained in the original GAST model 
	 * @param preferences Preferences containing the configuration of the analysis
	 * @param progressMonitor Progress monitor used to indicate detection progress
	 * @throws ModelAnalyzerException
	 */
	private void analyzeGASTModel(Root root,
			SimpleAnalysisResult analysisResult,
			List<List<GASTClass>> originalComponents,
			SoMoXConfiguration somoxConfiguration,
			IProgressMonitor progressMonitor) throws ModelAnalyzerException {

		List<List<GASTClass>> filteredComponents = 
			BlacklistFilter.filterComponents(somoxConfiguration.getBlacklist(),originalComponents);

		// Save the original model to allow finally building full SAMM instance models including references to
		// non-clustered and reverse engineered types:
		Root originalModelRoot = (Root) EcoreUtil.copy(root);

		IProgressMonitor subProgressMonitor = new ExecutionTimeLoggingProgressMonitor(progressMonitor, 0);
		subProgressMonitor.beginTask("Cluster components", IProgressMonitor.UNKNOWN);
		
		ComponentDetectionStrategy usedStrategy = somoxConfiguration.getUsedComponentDetectionStrategy();
		IDetectionStrategy detectionStrategy = ComponentDetectionStrategyFactory.createDetectionStrategy(usedStrategy);
		List<List<GASTClass>> componentsFound = detectionStrategy.startDetection(root, analysisResult, somoxConfiguration, progressMonitor,
				filteredComponents);

		subProgressMonitor.done();

		if (componentsFound != null) {
			subProgressMonitor = new SubProgressMonitor(progressMonitor, 1);
			subProgressMonitor.beginTask("Write model", 1);
			// Create NEW SammModelGenerator. Pass the original GAST classes, not the filtered ones. 			
			SammModelGenerator sammModelGenerator = new SammModelGenerator(originalModelRoot);	
			sammModelGenerator.addToRepository(componentsFound, analysisResult,subProgressMonitor);	
			subProgressMonitor.worked(1);
			subProgressMonitor.done();
		}
		
		this.status = ModelAnalyzer.Status.FINISHED;
		analysisResult.setResultStatus(AnalysisResult.ResultStatus.SUCCESS);
	}

	/**
	 * Create an analysis result with newly initialised root models
	 * @return A new analysis result
	 */
	private SimpleAnalysisResult initializeAnalysisResult() {
		SimpleAnalysisResult analysisResult = new SimpleAnalysisResult(this);
		Repository resultRepository = StaticstructureFactory.eINSTANCE.createRepository();
		SourceCodeDecoratorRepository sourceCodeDecoratorRepository = SourceCodeDecoratorFactory.eINSTANCE.createSourceCodeDecoratorRepository();
		GASTBehaviourRepository gastBehaviourRepository = qimpressgastFactory.eINSTANCE.createGASTBehaviourRepository(); 
		analysisResult.setInternalArchitectureModel(resultRepository);
		analysisResult.setGastBehaviourRepository(gastBehaviourRepository);
		analysisResult.setSourceCodeDecoratorRepository(sourceCodeDecoratorRepository);
		
		return analysisResult;
	}

	/**
	 * @return
	 * @uml.property  name="status"
	 */
	public ModelAnalyzer.Status getStatus() {
		return status;
	}
}
