package org.somox.analyzer.simplemodelanalyzer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.ecore.util.EcoreUtil;

import de.fzi.gast.accesses.InheritanceTypeAccess;
import de.fzi.gast.core.Root;
import de.fzi.gast.functions.Method;
import de.fzi.gast.types.GASTClass;
import de.fzi.gast.types.Visibilities;
import de.fzi.gast.variables.FormalParameter;
import de.uka.ipd.sdq.workflow.ExecutionTimeLoggingProgressMonitor;
import eu.qimpress.qimpressgast.GASTBehaviour;
import eu.qimpress.qimpressgast.GASTBehaviourRepository;
import eu.qimpress.qimpressgast.qimpressgastFactory;
import eu.qimpress.samm.behaviour.BehaviourFactory;
import eu.qimpress.samm.behaviour.GastBehaviourStub;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.CompositeComponent;
import eu.qimpress.samm.staticstructure.Interface;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.MessageType;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.samm.staticstructure.Parameter;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.samm.staticstructure.Repository;
import eu.qimpress.samm.staticstructure.StaticstructureFactory;
import eu.qimpress.samm.staticstructure.SubcomponentInstance;
import eu.qimpress.sourcecodedecorator.FileLevelSourceCodeLink;
import eu.qimpress.sourcecodedecorator.MethodLevelSourceCodeLink;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorFactory;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * Creates SAMM model instances for lists of GAST classes.  
 * @author kelsaka, Michael Hauck
 *
 */
public class SammModelGenerator {

	/**
	 * Map of all <b>required</b> interfaces. Maps a GAST interface
	 * class ID (NOT SISSy ID!) to a SAMM Interface.
	 */
	private HashMap<String, Interface> requiredInterfaces;
	/**
	 * Map of <b> all</b> (provided + required) interfaces. 
	 * Maps a GAST interface class ID (NOT SISSy ID!) to a SAMM Interface.
	 */
	private HashMap<String, Interface> allInterfaces;
	/**
	 * Hashmap of all GAST components; key: SAMM component ID;
	 * value: List of GASTClass representing a single component
	 * TODO: check: probably duplicate of successfullyHandledGastClassToComponentType
	 */
	private HashMap<String, List<GASTClass>> myComponentsGast;
	/**
	 * Hashmap of all SAMM components; key: SAMM component ID; value: SAMM component
	 */	
	private HashMap<String, ComponentType> myComponents;
	/**
	 * Map of created component types; maps GAST classes to a resulting component type
	 * (opposite of successfullyHandledComponentTypeToGASTClass)
	 */
	private HashMap<GASTClass, ComponentType> successfullyHandledGastClassToComponentType;
	/**
	 * Map of created component types; maps the SAMM component type to GAST class (opposite 
	 * of successfullyHandledGastClassToComponentType)
	 */
	private HashMap<ComponentType, GASTClass> successfullyHandledComponentTypeToGASTClass;
	/**
	 * Map of created GAST Methods; mapped to SAMM operations
	 */
	private HashMap<Method, Operation> successfullyHandledMethodToOperation;
	/**
	 * Set of all created composite components; each list represents a component;
	 * the inner list holds the inner components
	 */
	private HashSet<List<ComponentType>> existingCompositeComponents;
	/**
	 * General logger
	 */
	private Logger logger = Logger.getLogger(SammModelGenerator.class);
	/**
	 * Naming facility
	 */
	private ComponentAndTypeNaming componentAndTypeNaming;	
	/**
	 * A copy of the original GAST models root; non-blacklisted
	 */
	private Root originalGastModelRoot;

	/**
	 * 
	 * @param originalComponents A list of <b>all</a> components including 
	 * blacklisted ones.
	 * @param originalGastModelRoot Non-blacklisted (copy of the) GAST model including
	 * all possibly references but not reverse engineered types
	 */
	public SammModelGenerator(Root originalGastModelRoot) {
		initialize(originalGastModelRoot);
	}

	/**
	 * Call before re-running another reverse engineering run
	 * @param originalComponents non-blacklisted component list
	 */
	private void initialize(Root originalGastModelRoot) {
		this.originalGastModelRoot = originalGastModelRoot;	
		this.requiredInterfaces = new HashMap<String, Interface>();
		this.allInterfaces = new HashMap<String, Interface>();
		this.myComponentsGast = new HashMap<String, List<GASTClass>>();
		this.myComponents = new HashMap<String, ComponentType>();
		this.successfullyHandledGastClassToComponentType = new HashMap<GASTClass, ComponentType>();
		this.successfullyHandledComponentTypeToGASTClass = new HashMap<ComponentType, GASTClass>();
		this.successfullyHandledMethodToOperation = new HashMap<Method, Operation>(); 
		this.existingCompositeComponents = new HashSet<List<ComponentType>>();
		this.componentAndTypeNaming = new ComponentAndTypeNaming();
	}

	/**
	 * Main method. Creates components, interfaces, ports, and types
	 * for the given list of lists of components and adds them to the 
	 * repositories passed via results. 
	 * @param components SAMM components are created for theses components
	 * @param result target repositories
	 */
	public void addToRepository(List<List<GASTClass>> components,
			SimpleAnalysisResult result,
			IProgressMonitor monitor) {
		logger.info("Writing intermediate results to SAMM");
		IProgressMonitor subProgressMonitor = new ExecutionTimeLoggingProgressMonitor(monitor, 1);
		subProgressMonitor.beginTask("Writing SAM", 3);
		
		createComponents(components, result, subProgressMonitor);

		addProvidedInterfaceForRequiredInterfaces(
				this.requiredInterfaces,
				this.myComponentsGast,
				this.myComponents,
				result.getGastBehaviourRepository(),
				result.getSourceCodeDecoratorRepository(),
				subProgressMonitor);

		assignPublicMethodsAsInterfaceForComponentsWithoutInterface(
				this.myComponentsGast,
				result,
				subProgressMonitor);
		
		subProgressMonitor.done();
		logger.info("Writing of SAMM finished");
	}

	/**
	 * 
	 * check arguments: use for classes that have been "loose classes" before?
	 * @param innerComponents inner components of the composite strucure
	 */
	private CompositeComponent createCompositeComponent(List<ComponentType> innerComponents) {
		
		// avoid duplicate creation
		if(!existingCompositeComponents.contains(innerComponents)) {
			CompositeComponent newComponent = StaticstructureFactory.eINSTANCE.createCompositeComponent();
			newComponent.setName(
					componentAndTypeNaming.createCompositeComponentName(
							innerComponents, successfullyHandledComponentTypeToGASTClass));
			
			for(ComponentType ct : innerComponents) {
				SubcomponentInstance subcomponentInstance = StaticstructureFactory.eINSTANCE.createSubcomponentInstance();
				subcomponentInstance.setRealizedBy(ct);
				subcomponentInstance.setName(ct.getName() + "-instance");
				newComponent.getSubcomponents().add(subcomponentInstance);
			}
			
			existingCompositeComponents.add(innerComponents);
			return newComponent;
		} else {
			return null;
		}
	}

	private void createComponents(List<List<GASTClass>> components,
			SimpleAnalysisResult result, IProgressMonitor monitor) {
		
		IProgressMonitor subProgressMonitor = new ExecutionTimeLoggingProgressMonitor(monitor, 1);
		subProgressMonitor.beginTask("Exporting components", components.size());
		int i = 0;
		for (List<GASTClass> currentList : components) {
			i++;
			
			List<ComponentType> currentNewCompositeComponentInnersList = new ArrayList<ComponentType>();

			for (GASTClass gastClass : currentList) {				
								
				// check whether GAST class has been handled before to avoid duplicates
				if( !successfullyHandledGastClassToComponentType.containsKey(gastClass)) {
					PrimitiveComponent newComponent = StaticstructureFactory.eINSTANCE.createPrimitiveComponent();					
					newComponent.setName(componentAndTypeNaming.createSimpleComponentName(i, gastClass));
					updateSourceCodeDecorator(result, gastClass, newComponent);

					//TODO: start todo clean up and remove list argument; replace by single GAST class
					//this.myComponentsGast.put(newComponent.getId(), originalList); // original
					List<GASTClass> singleElementList = new ArrayList<GASTClass>(); //fake
					singleElementList.add(gastClass);
					this.myComponentsGast.put(newComponent.getId(), singleElementList);		
					//TODO: end todo
					
					this.myComponents.put(newComponent.getId(), newComponent);
					this.successfullyHandledGastClassToComponentType.put(gastClass, newComponent);
					this.successfullyHandledComponentTypeToGASTClass.put(newComponent, gastClass);
					currentNewCompositeComponentInnersList.add(newComponent);
					//addGastBehaviourToPrimitiveComponent(newComponent, gastClass);
					result.getInternalArchitectureModel().getComponenttype().add(newComponent);					
					
 					// primitive components encapsulate only one class
					findAndAddRequiredInterfaces(result.getInternalArchitectureModel(), gastClass, newComponent, this.requiredInterfaces);
 									
				} else {
					// find existing component and add to composite
					currentNewCompositeComponentInnersList.add(
							this.successfullyHandledGastClassToComponentType.get(gastClass));
					Assert.isNotNull(this.successfullyHandledGastClassToComponentType.get(gastClass));					
				}
			}
			
			//Create composite components for > 1 inner components:
			if(currentNewCompositeComponentInnersList.size() > 1) {
				CompositeComponent newComposite = createCompositeComponent(currentNewCompositeComponentInnersList);
				if(newComposite != null) {
					result.getInternalArchitectureModel().getComponenttype().add(newComposite);
				}
			}
			subProgressMonitor.worked(1);
		}
		subProgressMonitor.done();
	}

	/**
	 * Stores class names here in source code decorator.
	 * @param result
	 * @param gastClass
	 * @param newComponent
	 */
	private void updateSourceCodeDecorator(SimpleAnalysisResult result,
			GASTClass gastClass, PrimitiveComponent newComponent) {
		//TODO inner classes?
		FileLevelSourceCodeLink link = SourceCodeDecoratorFactory.eINSTANCE.createFileLevelSourceCodeLink();
		link.setComponentType(newComponent);
		if(gastClass.getPosition() != null && gastClass.getPosition().getSourceFile() != null) { // can be null for C code
			link.setFile(gastClass.getPosition().getSourceFile());
		}
		result.getSourceCodeDecoratorRepository().getFileLevelSourceCodeLink().add(link);
	}

	/**
	 * Finally, check if components exist that do not provide any interfaces. This components should
	 * then provide an interface with all public methods
	 * @param myComponentsGast Map of existing components 
	 * @param result Target repository for the new interfaces 
	 * @param monitor ProgressMonitor
	 */
	private void assignPublicMethodsAsInterfaceForComponentsWithoutInterface(
			HashMap<String, List<GASTClass>> myComponentsGast, SimpleAnalysisResult result, IProgressMonitor monitor) {
		logger.debug("Assigning public methods as interfaces");
		IProgressMonitor subProgressMonitor = new ExecutionTimeLoggingProgressMonitor(monitor, 1);
		subProgressMonitor.beginTask("Assigning public methods as interfaces", result.getInternalArchitectureModel().getComponenttype().size());
		
		for (ComponentType component : result.getInternalArchitectureModel().getComponenttype()) {
			if (component.getProvided().isEmpty()) {
				
				//GASTClass gastClass = successfullyHandledComponentTypeToGASTClass.get(component.getId());
				List<GASTClass> gastClasses = myComponentsGast.get(component.getId());
				if(gastClasses != null) {
					for(GASTClass gastClass : gastClasses) {
						
						List<MethodOperationTuple> publicMethods = new ArrayList<MethodOperationTuple>();
						
						Interface compInterface = createInterfaceBasedOnPublicMethods(
								result.getInternalArchitectureModel(),	gastClass, publicMethods);
						
						if (compInterface != null) {
							result.getInternalArchitectureModel().getInterface().add(compInterface);
							createProvidedPort(compInterface, component);
							
							//add behaviour for primitive components
							if (component instanceof PrimitiveComponent) {								
								addGASTBehaviourToPrimitiveComponent((PrimitiveComponent)component, publicMethods, result.getGastBehaviourRepository(), result.getSourceCodeDecoratorRepository());								
							}
						}
					}
				} else {
					logger.warn("No gast classes found for component: " + component.getName() + " id: " + component.getId());
				}
			}
			subProgressMonitor.worked(1);
		}
		subProgressMonitor.done();
	}

	/**
	 * Creates a new interface for the given gastClass if it was
	 * not present in allInterfaces before. 
	 * @param repository SAMM repository where which to add the methods and types from the
	 * interface. The interface itself is <b>not</b> added to the repository.
	 * @param gastClass interface is created for the public methods of this class
	 * @param publicMethods this list of filled with all 
	 * @return A new interface if a interface for gastClass did not exist;
	 * otherwise the existing interface is returned
	 */
	private Interface createInterfaceBasedOnPublicMethods(Repository repository,
			GASTClass gastClass,
			List<MethodOperationTuple> publicMethods) {		
		Interface compInterface = null;
				
		if(this.allInterfaces.containsKey(gastClass.getId())) {
			compInterface = this.allInterfaces.get(gastClass.getId());
		} else { 
			//create new interface
			compInterface = StaticstructureFactory.eINSTANCE.createInterface();			
			compInterface.setName("I" + gastClass.getSimpleName());
			
			//add methods
			for (Method method : gastClass.getMethods()) {
				if (method.getVisibility().equals(Visibilities.VISIBILITYPUBLIC)) {
					Operation tmpOperation = createOperationParametersAndMessageType(method, repository);
					compInterface.getSignatures().add(tmpOperation);
					publicMethods.add(new MethodOperationTuple(method, tmpOperation));
				}
			}			
			
			this.allInterfaces.put(gastClass.getId(), compInterface);
		}			
		return compInterface;
	}
	
	private void addGASTBehaviourToPrimitiveComponent(PrimitiveComponent component, List<MethodOperationTuple> publicMethods,
			GASTBehaviourRepository gastBehaviourRepository, SourceCodeDecoratorRepository sourceCodeDecoratorRepository) {		
		for (MethodOperationTuple methodOperationTuple : publicMethods) {
			addGASTBehaviourToPrimitiveComponent(component, methodOperationTuple, gastBehaviourRepository, sourceCodeDecoratorRepository);
		}
	}
	
	private void addGASTBehaviourToPrimitiveComponent(PrimitiveComponent component, MethodOperationTuple methodOperationTuple,
			GASTBehaviourRepository gastBehaviourRepository, SourceCodeDecoratorRepository sourceCodeDecoratorRepository) {		

		MethodLevelSourceCodeLink link = SourceCodeDecoratorFactory.eINSTANCE.createMethodLevelSourceCodeLink();
		link.setComponentType(component);
		if(methodOperationTuple.method.getPosition() != null && 
				methodOperationTuple.method.getPosition().getSourceFile() != null) {
			link.setFile(methodOperationTuple.method.getPosition().getSourceFile());
		}
		link.setFunction(methodOperationTuple.method);
		link.setOperation(methodOperationTuple.operation);
		sourceCodeDecoratorRepository.getMethodLevelSourceCodeLink().add(link);
	
		GastBehaviourStub gastBehaviourStub = BehaviourFactory.eINSTANCE.createGastBehaviourStub();
		gastBehaviourStub.setOperation(methodOperationTuple.operation);
		GASTBehaviour gastBehaviour = qimpressgastFactory.eINSTANCE.createGASTBehaviour();
		component.getOperationBehaviour().add(gastBehaviourStub);
		gastBehaviour.setBlockstatement(methodOperationTuple.method.getBody());
		gastBehaviour.setGastbehaviourstub(gastBehaviourStub);
		gastBehaviourRepository.getGastbehaviour().add(gastBehaviour);
	}


	/**
	 * Checks which components provide the required interfaces and adds a
	 * corresponding provided interface 
	 * <b>Precondition:</b> All components and all required interfaces exist. 
	 * @param requiredInterfaces
	 * @param myComponentsGast
	 * @param myComponents
	 * @param gastBehaviourRepository
	 * @param sourceCodeDecoratorRepository
	 */
	private void addProvidedInterfaceForRequiredInterfaces(HashMap<String, Interface> requiredInterfaces,
			HashMap<String, List<GASTClass>> myComponentsGast, HashMap<String, ComponentType> myComponents,
			GASTBehaviourRepository gastBehaviourRepository, SourceCodeDecoratorRepository sourceCodeDecoratorRepository,
			IProgressMonitor monitor) {
		IProgressMonitor subProgressMonitor = new ExecutionTimeLoggingProgressMonitor(monitor, 1);
		subProgressMonitor.beginTask("Export provided interfaces",requiredInterfaces.size());
		// Now, all components exist and all required interfaces.
		// Now look, which components provide the required interfaces
		for (String requiredInterfaceId : requiredInterfaces.keySet()) {			
			Interface requiredInterface = requiredInterfaces.get(requiredInterfaceId);

			// Look in all components if the required interface is provided
			for (String componentId : myComponentsGast.keySet()) {
				
				List<GASTClass> gastClasses = myComponentsGast.get(componentId);
				for (GASTClass gastClass : gastClasses) {
					//TODO get _ALL_ superclasses (on upper levels)
					for (GASTClass superType : gastClass.getSuperTypes()) {
						if (superType.isInterface()) {							
							if (requiredInterfaceId.equals(superType.getId())) {								
								// We found a component that provides a required interface
								ComponentType component = successfullyHandledGastClassToComponentType.get(gastClass);
								logger.debug("provided port match for component: " + component.getName() + " and interface: "+ requiredInterface.getName());
																
								// If the interface has already been added to component, do not add it again
								if (!componentProvidesInterface(requiredInterface, component)) {
									createProvidedPort(requiredInterface, component);
								
									updateSourceCodeDecorator(
											sourceCodeDecoratorRepository, gastClass);
									
									//add behaviour
									if (component instanceof PrimitiveComponent) {
										addGastBehavioursForRequiredInterfaceOperations(gastClass, superType, (PrimitiveComponent)component, gastBehaviourRepository, sourceCodeDecoratorRepository);
									}
								}
							}
						}
					}
				}
			}
			subProgressMonitor.worked(1);
		}
		subProgressMonitor.done();
	}

	/**
	 * save source information to sourceCodeDecorator
	 * @param sourceCodeDecoratorRepository write target
	 * @param gastClass an GAST <b>interface</b>
	 */
	private void updateSourceCodeDecorator(
			SourceCodeDecoratorRepository sourceCodeDecoratorRepository,
			GASTClass gastClass) {
		//TODO: add interface link
		FileLevelSourceCodeLink link = SourceCodeDecoratorFactory.eINSTANCE.createFileLevelSourceCodeLink();	
		link.setFile(gastClass.getPosition().getSourceFile());
		sourceCodeDecoratorRepository.getFileLevelSourceCodeLink().add(link);
	}
	
	private void addGastBehavioursForRequiredInterfaceOperations(GASTClass theClass, GASTClass implementedInterface, PrimitiveComponent component, GASTBehaviourRepository gastBehaviourRepository, SourceCodeDecoratorRepository sourceCodeDecoratorRepository) {
		for (Method method : theClass.getMethods()) {
			Method interfaceMethod = findCorrespondingInterfaceMethod(implementedInterface, method);
			if (interfaceMethod != null) { //some methods are not present in the interface but in a supertype
				Operation operation = successfullyHandledMethodToOperation.get(method);				
				addGASTBehaviourToPrimitiveComponent(component, new MethodOperationTuple(method, operation), gastBehaviourRepository, sourceCodeDecoratorRepository);
			}
		}
	}
	
	/**
	 * find mapping theClass method and implementedInterface method
	 * @param theInterface
	 * @param method
	 * @return
	 */
	private Method findCorrespondingInterfaceMethod(GASTClass theInterface, Method method) {
		int myHashCode = getSignatureId(method);
		for (Method interfaceMethod : theInterface.getMethods()) {
			if (myHashCode == getSignatureId(interfaceMethod)) {
				return interfaceMethod;
			}			
		}
		return null; 
	}

	/**
	 * Creates a signature ID based on parameter names + method name 
	 * @param interfaceMethod
	 * @return 
	 */
	private int getSignatureId(Method interfaceMethod) {
		//FIXME: introduce Sissy ID-based identification
		String interfaceMethodString = interfaceMethod.getSimpleName();
				
		for (FormalParameter param : interfaceMethod.getFormalParameters()) {
			param = lookupOriginalGastFormalParameter(param); //avoid null below: look up in the original model
			
			if (param.getType() == null) {
				logger.warn("ParameterType was null: "+ param.getSimpleName() + " method: "+ interfaceMethod.getSimpleName());
			} else {
				interfaceMethodString += "," + param.getType().getQualifiedName();
			}
		}
		return interfaceMethodString.hashCode();
	}		

	private InterfacePort createProvidedPort(Interface theInterface,
			ComponentType component) {
		InterfacePort provInterfacePort = StaticstructureFactory.eINSTANCE.createInterfacePort();
		provInterfacePort.setName("Provided " + component.getName() + " : " + theInterface.getName());
		provInterfacePort.setInterfaceType(theInterface);
		component.getProvided().add(provInterfacePort);
		return provInterfacePort;
	}

	private void findAndAddRequiredInterfaces(Repository resultRepository,
			GASTClass gastClass, PrimitiveComponent component, HashMap<String,Interface> requiredInterfaces) {

		List<Class<?>> blackListedAccessTypes = new ArrayList<Class<?>>();
		blackListedAccessTypes.add(InheritanceTypeAccess.class);
		
		List<GASTClass> filteredAccessedClasses = AccessFilter.filterAccessList(gastClass.getAllAccesses(), blackListedAccessTypes);
		for (GASTClass accessedClass : filteredAccessedClasses) {
			if (accessedClass.isInterface()) {					
				Interface reqInterface = createInterface(accessedClass, requiredInterfaces, resultRepository);
				// If the interface has already been added to component, do not add it again
				if (!doesComponentAlreadyRequireInterface(reqInterface, component)) {
					createRequiredPort(component, reqInterface);
				}
			}
		}
	}

	private void createRequiredPort(PrimitiveComponent component,
			Interface reqInterface) {
		InterfacePort reqInterfacePort = StaticstructureFactory.eINSTANCE.createInterfacePort();
		reqInterfacePort.setName("Required " + component.getName() + " : " + reqInterface.getName());
		reqInterfacePort.setInterfaceType(reqInterface);
		component.getRequired().add(reqInterfacePort);
	}

	/**
	 * 
	 * @param interfaceClass
	 * 			  the interface to add for gastClass
	 * @param interfaces
	 *            a list of interfaces in which the interface should be put. If
	 *            the interface is already contained in the list, this interface
	 *            is returned 
	 * @param repository
	 *            the SAMM repository in which the interface should be contained
	 * @return the interface
	 */
	public Interface createInterface(GASTClass interfaceClass,
			HashMap<String, Interface> interfaces, Repository repository) {
		
		// check for existing interface:
		Interface returnInterface = getExistingInterface(interfaceClass, interfaces); 

		// new interface
		if(returnInterface == null) {				
			returnInterface = StaticstructureFactory.eINSTANCE.createInterface();
			for (InheritanceTypeAccess inheritanceTypeAccess : interfaceClass.getInheritanceTypeAccesses()) {
				GASTClass superType = (GASTClass) inheritanceTypeAccess.getTargetType();
				if (superType.isInterface()) {
					Interface parentInterface = createInterface((GASTClass) superType, interfaces, repository);  
					returnInterface.getInheritance().add(parentInterface);
				}
			}
			returnInterface.setName(interfaceClass.getQualifiedName());
					
			for (Method method : interfaceClass.getMethods()) {
				if ((method.getVisibility().equals(Visibilities.VISIBILITYPACKAGE))
						|| (method.getVisibility()
								.equals(Visibilities.VISIBILITYPUBLIC))) {		
					returnInterface.getSignatures().add(
							createOperationParametersAndMessageType(method, repository));
		
				}
			}
			interfaces.put(interfaceClass.getId(), returnInterface);
			this.allInterfaces.put(interfaceClass.getId(), returnInterface);
			repository.getInterface().add(returnInterface);
		}
		return returnInterface;
	}

	/**
	 * Checks whether the interface is already present (created before)
	 * @param gastClass
	 * @param interfaces
	 * @return null if no interface could not be found
	 */
	private Interface getExistingInterface(GASTClass gastClass,
			HashMap<String, Interface> interfaces) {
		Interface returnInterface = null;
		
		if (interfaces.containsKey(gastClass.getId())) {
			returnInterface = interfaces.get(gastClass.getId());
		}
		
		if (allInterfaces.containsKey(gastClass.getId())) {
			returnInterface = allInterfaces.get(gastClass.getId());
			interfaces.put(gastClass.getId(), returnInterface);
		}
		return returnInterface;
	}

	/**
	 * Adds MessageTypes to the resultRepository, set parameter names and types.
	 * First looks if a MessageType already exists and creates one only if it does not exist in the repository.
	 * @param method GAST method to add 
	 * @param resultRepository repository to write to
	 * @return a new operation for which parameter names and types already exist in the resultRepository  
	 */
	private Operation createOperationParametersAndMessageType(Method method, Repository resultRepository) {
		
		Operation operation = StaticstructureFactory.eINSTANCE.createOperation();
		operation.setName(method.getSimpleName());
		
		ArrayList<String> paramNames = new ArrayList<String>();
		ArrayList<String> paramTypes = new ArrayList<String>();
		for (FormalParameter inputParameter : method.getFormalParameters()) {
			
			// use original (non-blacklisted) model
			// to avoid null below: look up in the original model:
			inputParameter = lookupOriginalGastFormalParameter(inputParameter);
						
			paramNames.add(inputParameter.getSimpleName());
			if(inputParameter.getType() != null) {
				paramTypes.add(inputParameter.getType().getSimpleName());
			} else {
				logger.error("Input parameter type was null. Could not set the parameter type \"" +
						inputParameter.getSimpleName() + "\" of method \"" + method.getSimpleName() + "\"");
			}
		}
		if (paramNames.size() > 0) {
			MessageType messageType = findMessageTypeInRepository(resultRepository, paramNames, paramTypes);
			if (messageType == null) {
				messageType = createMessageType(paramNames, paramTypes, resultRepository);
			}
			operation.setInput(messageType);
		}
		
		successfullyHandledMethodToOperation.put(method, operation);
		
		return operation;
	}

	/**
	 * Original GAST model lookup. Fetches model elements from the original GAST model
	 * which might have been eliminated during blacklisting.
	 * @param formalParameter parameter to find original instance for
	 * @return corresponding instance from the non-blacklisted model; null if not found in
	 * the original GAST model
	 */
	private FormalParameter lookupOriginalGastFormalParameter(FormalParameter formalParameter) {
		Collection<FormalParameter> allInputParameters =
			EcoreUtil.getObjectsByType(originalGastModelRoot.getAllModelElements(), formalParameter.eClass());
		
		for(FormalParameter currentFP : allInputParameters) {
			if(currentFP.getId() == formalParameter.getId()) {
				return currentFP;
			}
		}
		
		return null;
	}	

	/**
	 * Checks whether the given component requires the given interface. 
	 * @param theInterface
	 * @param component
	 * @return true if already required; false else
	 */
	private boolean doesComponentAlreadyRequireInterface(Interface theInterface,
			ComponentType component) {
		for (InterfacePort port : component.getRequired()) {
			if ((port.getInterfaceType() != null) && (port.getInterfaceType().getId().equals(theInterface.getId()))) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Create a message type
	 * 
	 * @param parameterNames
	 *            the names of the parameters
	 * @param parameterTypes
	 *            the type names of the parameter. SAMM types are created (if
	 *            they do not already exist) for these types
	 * @param repository
	 *            the Repository in which the MessageType has to be stored
	 * @return the created message type. Returns null if the size of
	 *         parameterNames does not equal the size of parameterTypes
	 */
	private MessageType createMessageType(List<String> parameterNames,
			List<String> parameterTypes, Repository repository) {
		if (parameterNames == null) {
			parameterNames = new ArrayList<String>();
		}
		if (parameterTypes == null) {
			parameterTypes = new ArrayList<String>();
		}
		if (parameterNames.size() != parameterTypes.size()) {
			return null;
		}
		MessageType messageType = StaticstructureFactory.eINSTANCE.createMessageType();
		String messageTypeName = "";
		if (parameterTypes.size() > 0) {
			for (int i = 0; i < parameterTypes.size(); i++) {
				if (messageTypeName.length() > 0) {
					messageTypeName += "_";
				}
				messageTypeName += parameterTypes.get(i);
				Parameter param = StaticstructureFactory.eINSTANCE.createParameter();
				param.setName(parameterNames.get(i));
				param.setType(componentAndTypeNaming.getType(parameterTypes.get(i), repository));
				messageType.getParameters().add(param);
			}
		}
		messageType.setName(messageTypeName);
		repository.getMessagetype().add(messageType);
		return messageType;

	}

	private boolean componentProvidesInterface(Interface theInterface,
			ComponentType component) {
		for (InterfacePort port : component.getProvided()) {
			if ((port.getInterfaceType() != null) && (port.getInterfaceType().getId().equals(theInterface.getId()))) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Look if a message type that contains the parameters specified by name and
	 * type already exists in the repository
	 * 
	 * @return the MessageType. Returns null, if no message type is found, or if
	 *         the size of parameterNames does not equal the size of
	 *         parameterTypes.
	 */
	private MessageType findMessageTypeInRepository(Repository theRepository,
			List<String> parameterNames, List<String> parameterTypes) {
		if (parameterNames == null) {
			parameterNames = new ArrayList<String>();
		}
		if (parameterTypes == null) {
			parameterTypes = new ArrayList<String>();
		}
		if (parameterNames.size() != parameterTypes.size()) {
			return null;
		}
		for (MessageType messageType : theRepository.getMessagetype()) {
			if (messageType.getParameters().size() != parameterNames.size()) {
				continue;
			}
			boolean parametersMatch = true;
			for (int i = 0; i < messageType.getParameters().size(); i++) {
				Parameter param = messageType.getParameters().get(i);
				if (!param.getName().equals(parameterNames.get(i))) {
					parametersMatch = false;
					break;
				}
				if (!param.getType().getName().toLowerCase().equals(parameterTypes.get(i).toLowerCase())) {
					parametersMatch = false;
					break;
				}
			}
			if (parametersMatch == true) {
				return messageType;
			}
		}
		return null;
	}
	
	/**
	 * @author  Snowball
	 */
	class MethodOperationTuple {
		public MethodOperationTuple(Method method, Operation operation) {
			this.method = method;
			this.operation = operation;
		}
		/**
		 * @uml.property  name="method"
		 * @uml.associationEnd  
		 */
		public Method method;
		/**
		 * @uml.property  name="operation"
		 * @uml.associationEnd  
		 */
		public Operation operation;
	}
	

}
