package org.somox.analyzer.simplemodelanalyzer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.apache.log4j.Logger;

import de.fzi.gast.types.GASTClass;
import eu.qimpress.samm.datatypes.CollectionDataType;
import eu.qimpress.samm.datatypes.DatatypesFactory;
import eu.qimpress.samm.datatypes.PrimitiveDataType;
import eu.qimpress.samm.datatypes.Type;
import eu.qimpress.samm.datatypes.XSDPrimitiveDataTypes;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.Repository;

/**
 * Component and package naming
 * @author kelsaka
 *
 */
public class ComponentAndTypeNaming {
	
	/**
	 * General logger
	 */
	private Logger logger = Logger.getLogger(ComponentAndTypeNaming.class);
	/**
	 * Counter for composite component names
	 */
	private int compositeComponentNumber;
	
	public ComponentAndTypeNaming() {
		this.compositeComponentNumber = 0;
	}

	public String createSimpleComponentName(int i, GASTClass gastClass) {
		return "Comp. " + i + ": " + gastClass.getSimpleName();
	}
	
	public String createSimpleComponentName(int i, List<GASTClass> currentList) {
		StringBuilder sb = new StringBuilder();
		for (GASTClass currentClass : currentList) {
			sb.append(currentClass.getSimpleName() + " ");
		}
		return "Comp No. " + i + " " + sb.toString();
	}
	
	public String createCompositeComponentName(List<ComponentType> composedSubComponents,
			HashMap<ComponentType, GASTClass> successfullyHandledComponentTypeToGASTClass) {
		compositeComponentNumber++;
		List<GASTClass> gastClasses = new ArrayList<GASTClass>();
		for(ComponentType ct : composedSubComponents) {
			gastClasses.add(successfullyHandledComponentTypeToGASTClass.get(ct));
		}
		
		return createComponentNameBasedOnPackageName(compositeComponentNumber, gastClasses);
	}

	/**
	 * Search for the package name that occurs most often
	 * @param i running number
	 * @param currentList List of classes belonging to the component
	 * @return Component name
	 */
	public String createComponentNameBasedOnPackageName(int i,
			List<GASTClass> currentList) {
		String returnComponentName = "";
		HashMap<String, Integer> numberOfPackageNames = new HashMap<String, Integer>();
		HashMap<String, String> packageNames = new HashMap<String, String>();
		String maxNumberPackageId = null;
		String directoryName = "";
		int maxNumber = 0;
		for (GASTClass currentClass : currentList) {
			if (currentClass.getSurroundingPackage() != null) {
				Integer tmpNumber = numberOfPackageNames.get(currentClass.getSurroundingPackage().getId());
				if (tmpNumber != null) {
					tmpNumber++;
					numberOfPackageNames.put(currentClass.getSurroundingPackage().getId(), tmpNumber);
					if (tmpNumber > maxNumber) {
						maxNumber = tmpNumber;
						maxNumberPackageId = currentClass.getSurroundingPackage().getId();
					}
				} else {
					numberOfPackageNames.put(currentClass.getSurroundingPackage().getId(), 1);
					packageNames.put(currentClass.getSurroundingPackage().getId(), currentClass.getSurroundingPackage().getQualifiedName());
					if (1 > maxNumber) {
						maxNumber = 1;
						maxNumberPackageId = currentClass.getSurroundingPackage().getId();
					}
				}
			} else if (currentClass.getPosition() != null && currentClass.getPosition().getSourceFile() != null){							
				directoryName = currentClass.getPosition().getSourceFile().getPathName();				
			} else {
				logger.debug("found neither packages nor directories for GAST class " + currentClass.getQualifiedName());
			}
		}

		returnComponentName = "";
		if(maxNumber > 0) {
			String compName = packageNames.get(maxNumberPackageId);
			if (compName != null) {
				returnComponentName += compName;
			}
		}
		
		returnComponentName += " " + createSimpleComponentName(i, currentList);
		if(!directoryName.equals("")) {
			returnComponentName += "(dir: " + directoryName + ")";
		}
		
		return returnComponentName;
	}
	
	public Type getType(String typeName, Repository repository) {
		if (typeName.toLowerCase().equals("int")) {
			// Do not create 2 datatypes for int and integer
			typeName = "integer";
		} else if (typeName.toLowerCase().equals("bool")) {
			// Do not create 2 datatypes for bool and boolean
			typeName = "boolean";
		}
		for (Type type : repository.getType()) {
			if (type.getName().toLowerCase().equals(typeName.toLowerCase())) {
				return type;
			}
		}
		Type type = null;
		if (typeName.toLowerCase().equals("integer")) {
			type = DatatypesFactory.eINSTANCE.createPrimitiveDataType();
			type.setName("INTEGER");
			((PrimitiveDataType) type).setType(XSDPrimitiveDataTypes.INT);
		} else if (typeName.toLowerCase().equals("float")) {
			type = DatatypesFactory.eINSTANCE.createPrimitiveDataType();
			type.setName("FLOAT");
			((PrimitiveDataType) type).setType(XSDPrimitiveDataTypes.FLOAT);
		} else if (typeName.toLowerCase().equals("string")) {
			type = DatatypesFactory.eINSTANCE.createPrimitiveDataType();
			type.setName("STRING");
			((PrimitiveDataType) type).setType(XSDPrimitiveDataTypes.STRING);
		} else if (typeName.toLowerCase().equals("boolean")) {
			type = DatatypesFactory.eINSTANCE.createPrimitiveDataType();
			type.setName("BOOLEAN");
			((PrimitiveDataType) type).setType(XSDPrimitiveDataTypes.BOOLEAN);
		} else if (typeName.endsWith("[]")) {
			// Create a collection data type
			String tmpTypeName = typeName.substring(0, typeName.length() - 2);
			type = DatatypesFactory.eINSTANCE.createCollectionDataType();
			type.setName(typeName);
			((CollectionDataType) type).setInnertype(getType(tmpTypeName, repository));
		} else {
			type = DatatypesFactory.eINSTANCE.createComplexDataType();
			type.setName(typeName);
		}
		repository.getType().add(type);
		return type;
	}
}
