package org.somox.analyzer.metriccomputation;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.somox.analyzerrules.AnalyzerRuleException;
import org.somox.analyzerrules.IAnalyzerRule;
import org.somox.analyzerrules.WeightedMetricsAnalyzerRule;
import org.somox.metrics.BlacklistFilter;
import org.somox.metrics.ClassAndCountStruct;
import org.somox.metrics.Coupling;
import org.somox.metrics.DMS;
import org.somox.metrics.DirectoryMapping;
import org.somox.metrics.Instability;
import org.somox.metrics.InterfaceViolation;
import org.somox.metrics.NameResemblance;
import org.somox.metrics.PackageMapping;
import org.somox.metrics.SliceLayerArchitectureQuality;
import org.somox.metrics.SubsystemComponent;
import org.somox.configuration.SoMoXConfiguration;

import de.fzi.gast.core.Root;
import de.fzi.gast.types.GASTClass;

/**
 * TODO: Document me!!!
 * @author Klaus Krogmann, Grischa Liebel
 */
public class TupleIterator {
	/**
	 * Strategy pattern. Contains the strategy how to compose the overall metric based on the single metrics available
	 * @uml.property  name="rule"
	 * @uml.associationEnd  
	 */
	private IAnalyzerRule rule;
	
	public TupleIterator(Set<String> blacklist, List<List<GASTClass>> filteredComponents, SoMoXConfiguration somoxConfiguration) throws AnalyzerRuleException {
		super();
		
		this.rule = new WeightedMetricsAnalyzerRule(somoxConfiguration);
		setAccessMap(BlacklistFilter.computeFilteredClass2ClassAccessMap(blacklist, filteredComponents));
	}
	
	/** 
	 * Basically this sets a graph containing GAST classes as nodes. The keys in the map are nodes representing GAST classes by their
	 * FQN. The set used as value models all outgoing accesses going from the class with the given FQN to the class contained in the 
	 * {@link ClassAndCountStruct}. The counter models the number of (different) accesses going from the class with the FQN to the class 
	 * in the struct
	 * @param accessMap The graph as described in this method's documentation
	 */
	private void setAccessMap(Map<String,Set<ClassAndCountStruct>> accessMap) {
		Coupling.setClass2ClassAccessMap(accessMap);
		InterfaceViolation.setClass2ClassAccessMap(accessMap);
		Instability.setClass2ClassAccessMap(accessMap);
	}
	
	/**
	 * This method is used to initalize the used metrics
	 * @param root
	 * @param somoxConfiguration 
	 * @throws AnalyzerRuleException
	 */
	 public void initialize(Root root, SoMoXConfiguration somoxConfiguration) throws AnalyzerRuleException {
		Coupling.initializeSpecific(root);
		DMS.initializeSpecific(root);
		InterfaceViolation.initializeSpecific(root);
		NameResemblance.initializeSpecific(root, somoxConfiguration);
		PackageMapping.initializeSpecific(root);
		DirectoryMapping.initializeSpecific(root);
		SliceLayerArchitectureQuality.initializeSpecific(root);
		SubsystemComponent.initializeSpecific(root);
	}
	
	/**
	 * Computes the overall metric of the relationship between element1 and element2
	 * TODO: Document me!!!!
	 * @param root The root node of the GAST model containing the software metadata to analyze
	 * @param elements1
	 * @param elements2
	 * @param components
	 * @return
	 * @throws AnalyzerRuleException
	 */
	public double compute (Root root, List<GASTClass> elements1, List<GASTClass> elements2, List<List<GASTClass>> components) throws AnalyzerRuleException {
		Map<Integer, Double> argAB = new HashMap<Integer, Double>();
		Map<Integer, Double> argBA = new HashMap<Integer, Double>();
		
		double couplingAB = Coupling.computeSpecific(root, elements1, elements2, components);
		
		double couplingBA = Coupling.computeSpecific(root, elements2, elements1, components);
		
		double coupling = Math.max(couplingAB, couplingBA);

		int elementsSum = elements1.size()+elements2.size();
		
		double value;
		//TODO: Document these constants (5 and 0.2)
		if (elementsSum < 5 || coupling > 0.2) {
			argAB.put(Coupling.getSpecificMID(), couplingAB);
			argBA.put(Coupling.getSpecificMID(), couplingBA);
			
			argAB.put(InterfaceViolation.getSpecificMID(), InterfaceViolation.computeSpecific(root, elements1, elements2, components));
			argBA.put(InterfaceViolation.getSpecificMID(), InterfaceViolation.computeSpecific(root, elements2, elements1, components));
			
			argAB.put(NameResemblance.getSpecificMID(), NameResemblance.computeSpecific(root, elements1, elements2, components));
			argAB.put(PackageMapping.getSpecificMID(), PackageMapping.computeSpecific(root, elements1, elements2, components));
			argAB.put(DirectoryMapping.getSpecificMID(), DirectoryMapping.computeSpecific(root, elements1, elements2, components));
			argAB.put(SubsystemComponent.getSpecificMID(), SubsystemComponent.computeSpecific(root, elements1, elements2, components));
			argAB.put(DMS.getSpecificMID(), DMS.computeSpecific(root, elements1, elements2, components));
			argAB.put(SliceLayerArchitectureQuality.getSpecificMID(), SliceLayerArchitectureQuality.computeSpecific(root, elements1, elements2, components));
			value = rule.computeOverallMetricValue(argAB, argBA);
		} else {
			value = -1.0; //default: bad value
		}
		return value;
	}
}

